package org.mkui.colormap.editor

import com.macrofocus.common.collection.Iterables
import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.format.CPFormat
import com.macrofocus.common.format.FormatFactory
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.common.selection.SimpleSingleSelection
import com.macrofocus.common.selection.SingleSelectionEvent
import com.macrofocus.common.selection.SingleSelectionListener
import org.mkui.colormap.MutableColorMap
import org.mkui.colormap.editor.ColorMapPanel
import org.mkui.component.CPComponent
import org.mkui.component.CPComponentProvider
import org.mkui.component.CPFactory
import org.mkui.component.button.CPRadioButtons
import org.mkui.component.panel.CPVerticalFlowPanel
import kotlin.reflect.KClass

class CPColorMapPanel<C>(
    factory: CPFactory,
    categoricalColorMapEditor: CategoricalColorMapEditor,
    predefinedColorMapEditor: PredefinedColorMapEditor,
    customColorMapEditor: CustomColorMapEditor
) : AbstractColorMapPanel<C>(), CPComponentProvider {
    private val mainPanel: CPVerticalFlowPanel
    private val editorPanel: CPVerticalFlowPanel

    private val type: MutableSingleSelection<ColorMapPanel.Type?> = SimpleSingleSelection(null)

    private val radioButtons: CPRadioButtons<ColorMapPanel.Type?>

    private val categoricalColorMapEditor: CategoricalColorMapEditor = categoricalColorMapEditor
    private val predefinedColorMapEditor: PredefinedColorMapEditor = predefinedColorMapEditor
    private val customColorMapEditor: CustomColorMapEditor = customColorMapEditor

    init {
        editorPanel = factory.createVerticalFlowPanel()

        type.addSingleSelectionListener(object : SingleSelectionListener<ColorMapPanel.Type?> {
            override fun selectionChanged(event: SingleSelectionEvent<ColorMapPanel.Type?>) {
                when (event.currentSelection) {
                    ColorMapPanel.Type.Categorical -> editorPanel.set(categoricalColorMapEditor)
                    ColorMapPanel.Type.Predefined -> editorPanel.set(predefinedColorMapEditor)
                    ColorMapPanel.Type.Custom -> editorPanel.set(customColorMapEditor)
                    null -> TODO()
                }
                updateColorMap()
            }
        })

        radioButtons = factory.createRadioButtons(type, ColorMapPanel.Type.entries.toTypedArray())

        val components = factory.createFillMiddlePanel().setCenter(radioButtons)
        mainPanel = factory.createVerticalFlowPanel(components, editorPanel)
    }

//    fun setStyleClass(vararg styleClasses: String?) {
//        mainPanel.setStyleClass(styleClasses)
//    }

    override val component: CPComponent
        get() = mainPanel

    override fun updateColorMap() {
        if (model != null) {
            val field: C? = model!!.colorSelection.selected
            if (field != null && model!!.getType(field) != null) {
                val type: KClass<*> = model!!.getType(field)
                categoricalColorMapEditor.setColorMap(model!!.getCategoricalColorMap(field), type)
                var format: CPFormat<Any?>? = model!!.getFormat(field)
                if (format == null) {
                    if (CPHelper.instance.isNumericType(type)) {
                        format = null
                    } else if (CPHelper.instance.isTemporalType(type)) {
                        format = FormatFactory.instance.createDateFormat()
                    }
                }
                customColorMapEditor.setColorMap(
                    model!!.getCustomColorMap(field),
                    type,
                    model!!.getNumericMin(field)!!,
                    model!!.getNumericMax(field)!!,
                    format
                )
                predefinedColorMapEditor.setColorMap(
                    model!!.getPredefinedColorMap(field),
                    model!!.getNumericMin(field),
                    model!!.getNumericMax(field),
                    format
                )

                when (this.type.selected) {
                    ColorMapPanel.Type.Categorical -> {
                        val colorMap: MutableColorMap = model!!.getCategoricalColorMap(field)
                        this.colorMap = colorMap
                        model!!.setCurrentColorMap(field, colorMap)
                    }

                    ColorMapPanel.Type.Predefined -> {
                        val colorMap: MutableColorMap = model!!.getPredefinedColorMap(field)
                        this.colorMap = colorMap
                        model!!.setCurrentColorMap(field, colorMap)
                    }

                    ColorMapPanel.Type.Custom -> {
                        val colorMap: MutableColorMap = model!!.getCustomColorMap(field)
                        this.colorMap = colorMap
                        model!!.setCurrentColorMap(field, colorMap)
                    }

                    null -> this.colorMap = null
                }
            } else {
                this.colorMap = null
            }
        }
    }

    override fun updateSelectedVariable() {
        val field: C? = model!!.colorSelection.selected

        val numericType =
            field != null && (CPHelper.instance.isNumericType(model!!.getType(field)) || CPHelper.instance.isTemporalType(model!!.getType(field)))

        if (model!!.getCurrentColorMap(field!!) != null && model!!.getCategoricalColorMap(field) === model!!.getCurrentColorMap(
                field!!
            )
        ) {
            type.selected = ColorMapPanel.Type.Categorical
        } else if (model!!.getCurrentColorMap(field) != null && model!!.getCustomColorMap(field) === model!!.getCurrentColorMap(
                field!!
            )
        ) {
            type.selected = ColorMapPanel.Type.Custom
        } else if (model!!.getCurrentColorMap(field) != null && model!!.getPredefinedColorMap(field) === model!!.getCurrentColorMap(
                field!!
            )
        ) {
            type.selected = ColorMapPanel.Type.Predefined
        } else {
            if (numericType) {
                type.selected = ColorMapPanel.Type.Predefined
            } else {
                type.selected = ColorMapPanel.Type.Categorical
            }
        }

        radioButtons.setEnabled(
            ColorMapPanel.Type.Categorical,
            !numericType || (field != null && model!!.getType(field) === Int::class)
        )
        radioButtons.setEnabled(ColorMapPanel.Type.Predefined, numericType)
        radioButtons.setEnabled(ColorMapPanel.Type.Custom, numericType)

        mainPanel.setVisible(field != null)
    }

    override fun setType(type: ColorMapPanel.Type?) {
        this.type.selected = type
    }
}
