package org.mkui.colormap.editor
/*
 * Copyright (c) 2017 Macrofocus GmbH. All Rights Reserved.
 */

import com.macrofocus.common.command.AbstractUICommand
import com.macrofocus.common.format.CPFormat
import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.common.properties.adapter.IntervalProperty
import com.macrofocus.common.selection.PropertySingleSelection
import org.mkui.canvas.CPCanvas
import org.mkui.color.MkColor
import org.mkui.colormap.ColorMapFactory
import org.mkui.colormap.MutableColorMap
import org.mkui.component.CPComponent
import org.mkui.component.CPComponentProvider
import org.mkui.component.CPFactory
import org.mkui.component.button.CPCheckBox
import org.mkui.component.button.CPComboBox
import org.mkui.component.button.CPSplitMenuButton
import org.mkui.component.input.CPColorPicker
import org.mkui.component.input.CPSlider
import org.mkui.component.input.CPSpinner
import org.mkui.component.panel.CPFormPanel
import org.mkui.component.panel.CPOverlayPanel
import org.mkui.component.panel.CPScrollPane
import org.mkui.geom.Point2D
import org.mkui.geom.Rectangle
import org.mkui.geom.Rectangle2D
import org.mkui.graphics.IDrawing
import org.mkui.graphics.IDrawingListener
import org.mkui.graphics.IGraphics
import org.mkui.palette.CustomPalette
import org.mkui.palette.MutablePalette
import org.mkui.palette.PaletteType
import kotlin.math.*

class CPPredefinedColorMapEditor(
    private val factory: CPFactory,
    private val colorMapFactory: ColorMapFactory
) : CPComponentProvider, AbstractPredefinedColorMapEditor() {

    private val mainPanel: CPFormPanel = factory.createFormPanel()
    private val setToFullRangeButton: CPSplitMenuButton = factory.createSplitMenuButton()
    private val maximumSpinner: CPSpinner<Double>
    private val minimumSpinner: CPSpinner<Double>
    private val underflowColor: CPColorPicker
    private val missingValueColor: CPColorPicker
    private val overflowColor: CPColorPicker
    private val paletteComboBox: CPComboBox<MutablePalette?>
    private val colorMapPreviewContainer: CPOverlayPanel
    private val numberOfStepsCheckBox: CPCheckBox
    private val invertedCheckBox: CPCheckBox
    private val numberOfStepsSpinner: CPSpinner<Int?>
    private val underColorCheckBox: CPCheckBox
    private val overColorCheckBox: CPCheckBox
    private val brightnessSlider: CPSlider
    private val saturationSlider: CPSlider
    private val minMaxPanel: CPFormPanel
    private val optionsPanel: CPFormPanel? = null
    private val optionsScrollPane: CPScrollPane? = null

    init {
        maximumSpinner = factory.createDoubleSpinner()
        maximumSpinner.setStyleClass("spinner", "maximumSpinner")
        minimumSpinner = factory.createDoubleSpinner()
        setToFullRangeButton.setStyleClass("setToFullRangeButton")
        setToFullRangeButton.setDefaultCommand(object : AbstractUICommand("Set to Data Range") {
            override fun execute(source: Any?) {
                minimumSpinner.setValue(if (min != null) min!!.toDouble() else null)
                maximumSpinner.setValue(if (max != null) max!!.toDouble() else null)
            }
        })
        setToFullRangeButton.addCommand(object : AbstractUICommand("Set to Data Range") {
            override fun execute(source: Any?) {
                minimumSpinner.setValue(if (min != null) min!!.toDouble() else null)
                maximumSpinner.setValue(if (max != null) max!!.toDouble() else null)
            }
        })
        setToFullRangeButton.addCommand(object : AbstractUICommand("Set to Rounded Range") {
            override fun execute(source: Any?) {
                val s: Double = sign(min!!.toDouble()) * 10.0.pow(floor(log10(abs(min!!.toDouble()))))
                minimumSpinner.setValue(s)
                val e: Double = sign(max!!.toDouble()) * 10.0.pow(ceil(log10(abs(max!!.toDouble()))))
                maximumSpinner.setValue(e)
            }
        })
        setToFullRangeButton.addCommand(object : AbstractUICommand("Set to Symmetrical Range around 0") {
            override fun execute(source: Any?) {
                if (min!!.toDouble() < 0 && min!!.toDouble() > 0) {
                    val diffMin: Double = 0 - min!!.toDouble()
                    val diffMax: Double = max!!.toDouble()
                    val maxDiff = max(diffMin, diffMax)
                    minimumSpinner.setValue(0 - maxDiff)
                    maximumSpinner.setValue(maxDiff)
                } else {
                    minimumSpinner.setValue(-max!!.toDouble())
                    maximumSpinner.setValue(if (max != null) max!!.toDouble() else null)
                }
            }
        })
        underflowColor = factory.createColorPicker()
        missingValueColor = factory.createColorPicker()
        overflowColor = factory.createColorPicker()
        paletteComboBox = factory.createComboBox<MutablePalette?>()
        paletteComboBox.setRenderer(object : CPComboBox.ItemRenderer<MutablePalette?> {
            override fun render(item: MutablePalette?): CPComponent {
                if (item != null) {
                    val canvas: CPCanvas = factory.createCanvas()
                    canvas.addLayer(object : IDrawing {
                        private val mode: CustomPalette.Mode = CustomPalette.Mode.Ramps

                        override val isActive: Boolean
                            get() = true

                        override fun draw(g: IGraphics, point: Point2D?, width: Double, height: Double, clipBounds: Rectangle) {
                            val insetTop = 0
                            val insetLeft = 0
                            val insetBottom = 0
                            val insetRight = 0

                            when (mode) {
                                CustomPalette.Mode.Bands -> {
                                    var i = 0
                                    while (i < item.colorCount) {
                                        val color: MkColor = item.getColorAt(i)

                                        val x1 =
                                            insetLeft + (width * i.toDouble() / item.colorCount as Double).toInt()
                                        val x2 =
                                            insetLeft + (width * (i + 1).toDouble() / item.colorCount as Double).toInt()
                                        g.setColor(color)
                                        g.fillRectangle2D(
                                            Rectangle2D.Double(
                                                x1.toDouble(),
                                                insetTop.toDouble(),
                                                x2 - x1 + 1.0,
                                                height - insetTop - insetBottom
                                            )
                                        )
                                        i++
                                    }
                                }

                                CustomPalette.Mode.Ramps -> {
                                    val colorCount = 256
                                    var i = 0
                                    while (i < colorCount) {
                                        val color: MkColor? = item.getColor((i / (colorCount - 1.0)))
                                        val x1 = insetLeft + (width * i.toDouble() / colorCount.toDouble()).toInt()
                                        val x2 =
                                            insetLeft + (width * (i + 1).toDouble() / colorCount.toDouble()).toInt()
                                        g.setColor(color!!)
                                        g.fillRectangle2D(
                                            Rectangle2D.Double(
                                                x1.toDouble(),
                                                insetTop.toDouble(),
                                                x2 - x1 + 1.0,
                                                height - insetTop - insetBottom
                                            )
                                        )
                                        i++
                                    }
                                }
                            }
                        }

                        override fun addIDrawingListener(listener: IDrawingListener) {
                        }

                        override fun removeIDrawingListener(listener: IDrawingListener) {
                        }
                    })
                    return canvas
                } else {
                    return factory.createLabel("")
                }
            }
        })
        colorMapPreviewContainer = factory.createOverlayPanel()
        numberOfStepsCheckBox = factory.createCheckBox("Number of Steps:", null)
        invertedCheckBox = factory.createCheckBox("Inverted", null)
        numberOfStepsSpinner = factory.createIntegerSpinner()
        underColorCheckBox = factory.createCheckBox("Underflow Color:", null)
        overColorCheckBox = factory.createCheckBox("Overflow Color:", null)
        brightnessSlider = factory.createSlider()
        brightnessSlider.setMinimum(-150.0)
        brightnessSlider.setMaximum(150.0)
        saturationSlider = factory.createSlider()
        saturationSlider.setMinimum(-150.0)
        saturationSlider.setMaximum(150.0)
        minMaxPanel = factory.createFormPanel()

        //        optionsPanel = factory.createScrollPane();
        mainPanel.add(factory.createLabel("Palette:"), paletteComboBox)
        mainPanel.add(factory.createLabel("Maximum:"), maximumSpinner)
        mainPanel.add(factory.createLabel("Minimum:"), minimumSpinner)
        mainPanel.add(null, setToFullRangeButton)
        mainPanel.add(numberOfStepsCheckBox, numberOfStepsSpinner)
        mainPanel.add(invertedCheckBox, null)
        mainPanel.add(factory.createLabel("Brightness:"), brightnessSlider)
        mainPanel.add(factory.createLabel("Saturation:"), saturationSlider)
        mainPanel.add(overColorCheckBox, overflowColor)
        mainPanel.add(underColorCheckBox, underflowColor)
        mainPanel.add(factory.createLabel("Missing Values Color:"), missingValueColor)
    }

    override fun setColorMap(colorMap: MutableColorMap, min: Number?, max: Number?, format: CPFormat<Any?>?) {
        this.colorMap = colorMap
        this.min = min
        this.max = max
        this.format = format

        maximumSpinner.setProperty(IntervalProperty(colorMap.interval!!, IntervalProperty.Value.End) as MutableProperty<Double?>)
        maximumSpinner.setFormat(format)
        minimumSpinner.setProperty(IntervalProperty(colorMap.interval!!, IntervalProperty.Value.Start) as MutableProperty<Double?>)
        minimumSpinner.setFormat(format)
        underflowColor.setProperty(colorMap.underColorProperty)
        missingValueColor.setProperty(colorMap.nullColorProperty)
        overflowColor.setProperty(colorMap.overColorProperty)
        val palettes: MutableList<MutablePalette> = ArrayList<MutablePalette>()
        for (entry in colorMapFactory.getPaletteFactory().getEntries()) {
            if (entry.type !== PaletteType.QUALITATIVE) {
                palettes.add(entry.getPalette())
            }
        }
        paletteComboBox.setModel(PropertySingleSelection<MutablePalette?>(colorMap.paletteProperty), palettes)

        numberOfStepsCheckBox.setProperty(colorMap.colorCountSetProperty)
        numberOfStepsSpinner.setProperty(colorMap.colorCountProperty as MutableProperty<Int?>)
        invertedCheckBox.setProperty(colorMap.isInvertedProperty)
        underColorCheckBox.setProperty(colorMap.underColorSetProperty)
        overColorCheckBox.setProperty(colorMap.overColorSetProperty)
        brightnessSlider.setProperty(colorMap.brightnessProperty)
        saturationSlider.setProperty(colorMap.saturationProperty)
    }

    fun setStyleClass(vararg styleClasses: String?) {
//        mainPanel.setStyleClass(styleClasses)
    }

    override val component: CPComponent
        get() = mainPanel
}
