package com.macrofocus.visual

import com.macrofocus.common.filter.Filter
import com.macrofocus.common.selection.Selection
import com.macrofocus.common.selection.SelectionEvent
import com.macrofocus.common.selection.SelectionListener
import com.macrofocus.common.selection.SingleSelection
import org.mkui.visual.VisualObjects

open class SelectedVisualLayer<O>(visualObjects: VisualObjects<O>?, selection: Selection<O>, probing: SingleSelection<O>, filter: Filter<O>?) :
    AbstractVisualLayer<O>() {
    private val visualObjects: VisualObjects<O>?
    private val selection: Selection<O>
    private val probing: SingleSelection<O>
    private val filter: Filter<O>?
    private val selectionListener: SelectionListener<O> = object : SelectionListener<O> {
        override fun selectionChanged(event: SelectionEvent<O>) {
            notifyVisualChanged()
        }
    }
    override val isActive: Boolean
        get() = selection.isActive

    override fun getObject(i: Int): O {
        return visualObjects!!.getObject(i)
    }

    override val objectCount: Int
        get() = if (visualObjects != null && selection.isActive) visualObjects.objectCount else 0

    override operator fun iterator(): Iterator<O> {
        return if (objectCount > 0) {
            if (probing.isActive || filter != null && filter.isActive) object : FilteredIterable<O>(selection.iterator()) {
                override fun condition(`object`: O): Boolean {
                    return !probing.isSelected(`object`) && (filter == null || !filter.isFiltered(`object`))
                }
            }.iterator() else selection.iterator()
        } else {
            emptyList<O>().iterator()
        }
    }

    override fun iterable(from: Int, to: Int): Iterable<O> {
        return object : ObjectsIterable<O>(this, from, to) {
            override fun condition(`object`: O): Boolean {
                return selection.isSelected(`object`) && !probing.isSelected(`object`) && (filter == null || !filter.isFiltered(`object`))
            }
        }
    }

    protected abstract inner class FilteredIterable<O>(private val iterator: Iterator<O>) : Iterable<O> {
        override fun iterator(): Iterator<O> {
            return object : MutableIterator<O> {
                var nextEntry: O? = null
                var queryNext = true
                override fun hasNext(): Boolean {
                    if (queryNext) {
                        nextEntry = findNext()
                        queryNext = false
                    }
                    return nextEntry != null
                }

                override fun next(): O {
                    if (queryNext) {
                        nextEntry = findNext()
                    }
                    queryNext = true
                    return nextEntry!!
                }

                private fun findNext(): O? {
                    while (iterator.hasNext()) {
                        val `object` = iterator.next()
                        if (condition(`object`)) {
                            return `object`
                        }
                    }
                    return null
                }

                override fun remove() {
                    throw UnsupportedOperationException()
                }
            }
        }

        abstract fun condition(`object`: O): Boolean
    }

    init {
        this.visualObjects = visualObjects
        this.selection = selection
        this.probing = probing
        this.filter = filter
        selection.addWeakSelectionListener(selectionListener)
    }
}