package org.mkui.colormap.editor
/*
 * Copyright (c) 2017 Macrofocus GmbH. All Rights Reserved.
 */

import com.macrofocus.colormapping.implementation.SimpleColorMapping
import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.delegate.mutableLazy
import com.macrofocus.common.format.CPFormat
import com.macrofocus.common.format.FormatFactory
import com.macrofocus.common.math.isNaN
import com.macrofocus.common.selection.MutableSingleSelection
import com.macrofocus.common.selection.PropertySingleSelection
import com.macrofocus.common.selection.SimpleSingleSelection
import com.macrofocus.visual.SimpleVisual
import com.macrofocus.visual.SimpleVisualObjects
import org.mkui.color.MkColor
import org.mkui.color.colorOf
import org.mkui.colormap.ColorMapFactory
import org.mkui.colormap.MutableColorMap
import org.mkui.component.CPComponent
import org.mkui.component.CPComponentProvider
import org.mkui.component.CPFactory
import org.mkui.component.button.CPCheckBox
import org.mkui.component.button.CPRadioButtons
import org.mkui.component.input.CPColorPicker
import org.mkui.component.input.CPSlider
import org.mkui.component.panel.CPFormPanel
import org.mkui.component.panel.CPOverlayPanel
import org.mkui.component.panel.CPVerticalFlowPanel
import org.mkui.palette.CustomPalette
import org.mkui.palette.InterpolatedPalette
import org.mkui.palette.PaletteEvent
import org.mkui.palette.PaletteListener
import org.mkui.table.Table
import org.mkui.table.TableFactory
import org.molap.dataframe.AbstractMutableDataFrame
import org.molap.dataframe.DataFrameEvent
import org.molap.dataframe.MutableDataFrame
import org.molap.index.DefaultUniqueIndex
import org.molap.index.IntegerRangeUniqueIndex
import org.molap.series.Series
import kotlin.reflect.KClass

class CPCustomColorMapEditor(
    private val factory: CPFactory,
    private val colorMapFactory: ColorMapFactory,
    tableFactory: TableFactory
) : AbstractCustomColorMapEditor(), CPComponentProvider {

    private val mode: MutableSingleSelection<CustomPalette.Mode?> = SimpleSingleSelection<CustomPalette.Mode?>(null)

    private val mainPanel: CPVerticalFlowPanel = factory.createVerticalFlowPanel()

    //    private CPButton generateColorsButton;
    private val bandsRampsRadioButtons: CPRadioButtons<CustomPalette.Mode>
    private val underflowColor: CPColorPicker
    private val overflowColor: CPColorPicker
    private val missingValueColor: CPColorPicker
    private val table: Table<Int, String, Any?>
    private val colorMapPreviewContainer: CPOverlayPanel
    private val brightnessSlider: CPSlider
    private val saturationSlider: CPSlider
    private val overColorCheckBox: CPCheckBox
    private val underColorCheckBox: CPCheckBox

    init {
        val formPanel: CPFormPanel = factory.createFormPanel()
        bandsRampsRadioButtons = factory.createRadioButtons(CustomPalette.Mode.entries.toTypedArray())
        underflowColor = factory.createColorPicker()
        overflowColor = factory.createColorPicker()
        missingValueColor = factory.createColorPicker()
        table = tableFactory.createTable()
        table.setColumnResizePolicy(Table.ColumnResizePolicy.Constrained)
        table.setPrefHeight(200.0)
        colorMapPreviewContainer = factory.createOverlayPanel()
        brightnessSlider = factory.createSlider()
        brightnessSlider.setMinimum(-150.0)
        brightnessSlider.setMaximum(150.0)
        saturationSlider = factory.createSlider()
        saturationSlider.setMinimum(-150.0)
        saturationSlider.setMaximum(150.0)
        overColorCheckBox = factory.createCheckBox("Overflow Color:", null)
        underColorCheckBox = factory.createCheckBox("Underflow Color:", null)

        mainPanel.add(table)
        formPanel.add(null, bandsRampsRadioButtons)
        formPanel.add(factory.createLabel("Brightness:"), brightnessSlider)
        formPanel.add(factory.createLabel("Saturation:"), saturationSlider)
        formPanel.add(overColorCheckBox, overflowColor)
        formPanel.add(underColorCheckBox, underflowColor)
        formPanel.add(factory.createLabel("Missing Values Color:"), missingValueColor)
        mainPanel.add(formPanel)
    }

    override fun setColorMap(
        colorMap: MutableColorMap,
        type: KClass<*>,
        min: Number,
        max: Number,
        format: CPFormat<Any?>?
    ) {
        this.colorMap = colorMap
        this.min = min
        this.max = max

        val dataFrame: MutableDataFrame<Int, String, Any?> = ColorsDataFrame(colorMap.palette as CustomPalette, type, min, max)
        val visualObjects: SimpleVisualObjects<Int> = SimpleVisualObjects(dataFrame)
        val visual: SimpleVisual<Int, String> =
            SimpleVisual(visualObjects, SimpleColorMapping<Int, String>(colorMapFactory, dataFrame))
        table.setModel(dataFrame, visual)
        table.setCellRenderer("Color", ColorCellRenderer(factory, dataFrame))
        table.setEditable(true)
        table.setCellEditor("Threshold", object : Table.CellEditor<Int, String, Any?> {
            val decimalFormat: CPFormat<Any?> = FormatFactory.instance.createDecimalFormat()
            override fun isEditable(row: Int, column: String, value: Any?): Boolean {
                return true
            }

            override val format: CPFormat<Any?>
                get() = decimalFormat
        })
        bandsRampsRadioButtons.setSelection(PropertySingleSelection<CustomPalette.Mode>((colorMap.palette as CustomPalette).modeProperty()))
        underflowColor.setProperty(colorMap.underColorProperty)
        missingValueColor.setProperty(colorMap.nullColorProperty)
        overflowColor.setProperty(colorMap.overColorProperty)
        underColorCheckBox.setProperty(colorMap.underColorSetProperty)
        overColorCheckBox.setProperty(colorMap.overColorSetProperty)
        brightnessSlider.setProperty(colorMap.brightnessProperty)
        saturationSlider.setProperty(colorMap.saturationProperty)
    }

//    fun setStyleClass(vararg styleClasses: String?) {
//        mainPanel.setStyleClass(styleClasses)
//    }

    override val component: CPComponent
        get() = mainPanel

    private class ColorsDataFrame(
        private val palette: CustomPalette,
        private val valueType: KClass<*>,
        min: Number,
        max: Number
    ) : AbstractMutableDataFrame<Int, String, Any?>() {
        private val min: Number = min
        private val max: Number = max

        override val columnIndex: DefaultUniqueIndex<String> = DefaultUniqueIndex("Threshold", "Color")
        val rowIndexDelegate = mutableLazy { IntegerRangeUniqueIndex(0, rowCount - 1) }
        override val rowIndex by rowIndexDelegate

        init {
            palette.addPaletteListener(object : PaletteListener {
                override fun paletteChanged(event: PaletteEvent) {
                    rowIndexDelegate.forget()
                    notifyDataFrameChanged(DataFrameEvent<Int, String>(null, null, true))
                }
            })
        }

        override fun setValueAt(row: Int, column: String, value: Any?) {
            val rowIndex: Int = rowIndex.getAddress(row)
            val columnIndex: Int = columnIndex.getAddress(column)
            when (columnIndex) {
                0 -> {
                    val entry: InterpolatedPalette.Entry = getEntry(rowIndex)
                    interpolatedPalette.setColor(entry.fraction, null)
                    if (value != null && (value is Number || CPHelper.instance.isTemporalValue(value))) {
                        val value = if (CPHelper.instance.isTemporalValue(value)) {
                            CPHelper.instance.doubleTemporalValue(value)
                        } else {
                            (value as Number).toDouble()
                        }
                        val fraction = (value - min.toDouble()) / (max.toDouble() - min.toDouble())
                        if (!isNaN(entry.fraction)) {
                            interpolatedPalette.setColor(fraction, entry.getColor())
                        } else {
                            val newColor: MkColor? = interpolatedPalette.getColor(value)
                            if (newColor != null) {
                                interpolatedPalette.setColor(fraction, newColor)
                            } else {
                                interpolatedPalette.setColor(fraction, entry.getColor())
                            }
                        }
                    }
                }

                1 -> interpolatedPalette.setColor(getEntry(rowIndex).fraction, value as MkColor)
            }
        }

        override fun getRowClass(row: Int): KClass<out Any> {
            return Any::class
        }

        override fun getColumnClass(column: String): KClass<out Any> {
            return when (columnIndex.getAddress(column)) {
                0 -> valueType
                1 -> MkColor::class
                else -> Any::class
            }
        }

        override fun getValueAt(row: Int, column: String): Any? {
            val rowIndex: Int = rowIndex.getAddress(row)
            when (columnIndex.getAddress(column)) {
                0 -> {
                    val fraction: Double = getEntry(rowIndex).fraction
                    return if (!isNaN(fraction)) {
                        getAbsolute(fraction)
                    } else {
                        null
                    }
                }

                1 -> {
                    if (interpolatedPalette.getMode() === CustomPalette.Mode.Bands) {
                        if (rowIndex == interpolatedPalette.getEntries().size - 1) {
                            return CustomPalette.Entry(Double.NaN, colorOf(255, 255, 255, 0))
                        }
                    }

                    return getEntry(rowIndex).getColor()
                }

                else -> return null
            }
        }

        fun getRow(integer: Int?): Series<String, *>? {
            return null
        }

        override val rowCount: Int
            get() = interpolatedPalette.getEntries().size + 1

        override val columnCount: Int
            get() = 2

        fun isCellEditable(rowIndex: Int, columnIndex: Int): Boolean {
            if (columnIndex == 1 && interpolatedPalette.getMode() === CustomPalette.Mode.Bands) {
                if (rowIndex == interpolatedPalette.getEntries().size - 1) {
                    return false
                }
            }

            return rowIndex < interpolatedPalette.getEntries().size || columnIndex == 0
        }

        private fun getAbsolute(fraction: Double): Double {
            return if (min != null && max != null) {
                min.toDouble() + (fraction * (max.toDouble() - min.toDouble()))
            } else {
                fraction
            }
        }

        private fun getEntry(rowIndex: Int): InterpolatedPalette.Entry {
            return if (rowIndex < interpolatedPalette.getEntries().size) {
                interpolatedPalette.getEntries().toTypedArray().get(rowIndex) as InterpolatedPalette.Entry
            } else {
                InterpolatedPalette.Entry(Double.NaN, colorOf(255, 255, 255, 0))
            }
        }

        private val interpolatedPalette: CustomPalette
            get() = palette
    }
}
