/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.colormap

import com.macrofocus.common.collection.CollectionFactory
import com.macrofocus.common.collection.WeakReference

/**
 * This class provides a skeletal implementation of the Colormap interface to minimize the effort required to implement this interface.
 */
abstract class AbstractColorMap protected constructor() : ColorMap {
    private val listeners: MutableList<ColorMapListener>
    override fun addColorMapListener(listener: ColorMapListener) {
        listeners.add(listener)
    }

    override fun addWeakColorMapListener(listener: ColorMapListener) {
        listeners.add(WeakColorMapListener(listener))
    }

    override fun removeColorMapListener(listener: ColorMapListener) {
        if (listener is WeakColorMapListener) {
            val removed = listeners.remove(listener)
        } else {
            var toRemove: ColorMapListener? = null
            for (ColormapListener in listeners) {
                var comparable: ColorMapListener
                comparable = if (ColormapListener is WeakColorMapListener) {
                    ColormapListener.reference
                } else {
                    ColormapListener
                }
                if (listener.equals(comparable)) {
                    toRemove = ColormapListener
                }
            }
            if (toRemove != null) {
                val removed = listeners.remove(toRemove)
            }
        }
    }

    fun removeColormapListeners() {
        listeners.clear()
    }

    protected fun notifyColorMapChanged(event: ColorMapEvent) {
        for (listener in listeners) {
            listener.colorMapChanged(event)
        }
    }

    private class WeakColorMapListener(listener: ColorMapListener) : ColorMapListener {
        private val l_ref: WeakReference<ColorMapListener>
        override fun colorMapChanged(event: ColorMapEvent) {
            val l: ColorMapListener = reference
            if (l != null) {
                l.colorMapChanged(event)
            } else {
                event.getColorMap().removeColorMapListener(this)
            }
        }

        val reference: org.mkui.colormap.ColorMapListener
            get() = l_ref.get()

        init {
            l_ref = WeakReference<ColorMapListener>(listener)
        }
    }

    init {
        listeners = CollectionFactory.copyOnWriteArrayList()
    }
}