/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.geom

abstract class AreaOp private constructor() {
    fun calculate(left: MutableList<Curve>, right: MutableList<Curve>): MutableList<Curve> {
        val edges: MutableList<Edge> = ArrayList<Edge>()
        addEdges(edges, left, CTAG_LEFT)
        addEdges(edges, right, CTAG_RIGHT)
        val curves: MutableList<Curve> = pruneEdges(edges)
        if (false) {
            println("result: ")
            val numcurves: Int = curves.size
            val curvelist: Array<Curve> = curves.toTypedArray()
            for (i in 0 until numcurves) {
                println("curvelist[" + i + "] = " + curvelist[i])
            }
        }
        return curves
    }

    private fun pruneEdges(edges: MutableList<Edge>): MutableList<Curve> {
        val numedges: Int = edges.size
        if (numedges < 2) {
            // empty vector is expected with less than 2 edges
            return ArrayList<Curve>()
        }
        val edgelist: Array<Edge> = edges.toTypedArray()
        edgelist.sortWith(YXTopComparator)
        if (false) {
            println("pruning: ")
            for (i in 0 until numedges) {
                println("edgelist[" + i + "] = " + edgelist[i])
            }
        }
        var left = 0
        var right = 0
        var cur: Int
        var next: Int
        val yrange = DoubleArray(2)
        val subcurves: MutableList<CurveLink?> = ArrayList<CurveLink?>()
        val chains: MutableList<ChainEnd?> = ArrayList<ChainEnd?>()
        val links: MutableList<CurveLink?> = ArrayList<CurveLink?>()
        // Active edges are between left (inclusive) and right (exclusive)
        while (left < numedges) {
            var y = yrange[0]
            // Prune active edges that fall off the top of the active y range
            var e: Edge
            cur = right - 1.also { next = it }
            while (cur >= left) {
                e = edgelist[cur]
                if (e.curve.yBot > y) {
                    if (next > cur) {
                        edgelist[next] = e
                    }
                    next--
                }
                cur--
            }
            left = next + 1
            // Grab a new "top of Y range" if the active edges are empty
            if (left >= right) {
                if (right >= numedges) {
                    break
                }
                y = edgelist[right].curve.yTop
                if (y > yrange[0]) {
                    finalizeSubCurves(subcurves, chains)
                }
                yrange[0] = y
            }
            // Incorporate new active edges that enter the active y range
            while (right < numedges) {
                e = edgelist[right]
                if (e.curve.yTop > y) {
                    break
                }
                right++
            }
            // Sort the current active edges by their X values and
            // determine the maximum valid Y range where the X ordering
            // is correct
            yrange[1] = edgelist[left].curve.yBot
            if (right < numedges) {
                y = edgelist[right].curve.yTop
                if (yrange[1] > y) {
                    yrange[1] = y
                }
            }
            if (false) {
                println(
                    "current line: y = [" +
                            yrange[0] + ", " + yrange[1] + ']'
                )
                cur = left
                while (cur < right) {
                    println("  " + edgelist[cur])
                    cur++
                }
            }
            // Note: We could start at left+1, but we need to make
            // sure that edgelist[left] has its equivalence set to 0.
            var nexteq = 1
            cur = left
            while (cur < right) {
                e = edgelist[cur]
                e.equivalence = 0
                next = cur
                while (next > left) {
                    val prevedge: Edge = edgelist[next - 1]
                    val ordering: Int = e.compareTo(prevedge, yrange)
                    if (yrange[1] <= yrange[0]) {
                        throw UnsupportedOperationException(
                            "backstepping to " + yrange[1] +
                                    " from " + yrange[0]
                        )
                    }
                    if (ordering >= 0) {
                        if (ordering == 0) {
                            // If the curves are equal, mark them to be
                            // deleted later if they cancel each other
                            // out so that we avoid having extraneous
                            // curve segments.
                            var eq: Int = prevedge.equivalence
                            if (eq == 0) {
                                eq = nexteq++
                                prevedge.equivalence = eq
                            }
                            e.equivalence = eq
                        }
                        break
                    }
                    edgelist[next] = prevedge
                    next--
                }
                edgelist[next] = e
                cur++
            }
            if (false) {
                println(
                    "current sorted line: y = [" +
                            yrange[0] + ", " + yrange[1] + ']'
                )
                cur = left
                while (cur < right) {
                    println("  " + edgelist[cur])
                    cur++
                }
            }
            // Now prune the active edge list.
            // For each edge in the list, determine its classification
            // (entering shape, exiting shape, ignore - no change) and
            // record the current Y range and its classification in the
            // Edge object for use later in constructing the new outline.
            newRow()
            val ystart = yrange[0]
            val yend = yrange[1]
            cur = left
            while (cur < right) {
                e = edgelist[cur]
                var etag: Int
                val eq: Int = e.equivalence
                if (eq != 0) {
                    // Find one of the segments in the "equal" range
                    // with the right transition state and prefer an
                    // edge that was either active up until ystart
                    // or the edge that extends the furthest downward
                    // (i.e. has the most potential for continuation)
                    val origstate = state
                    etag = if (origstate == RSTAG_INSIDE) ETAG_EXIT else ETAG_ENTER
                    var activematch: Edge? = null
                    var longestmatch: Edge = e
                    var furthesty = yend
                    do {
                        // Note: classify() must be called
                        // on every edge we consume here.
                        classify(e)
                        if (activematch == null &&
                            e.isActiveFor(ystart, etag)
                        ) {
                            activematch = e
                        }
                        y = e.curve.yBot
                        if (y > furthesty) {
                            longestmatch = e
                            furthesty = y
                        }
                    } while (++cur < right &&
                        edgelist[cur].also { e = it }.equivalence === eq
                    )
                    --cur
                    if (state == origstate) {
                        etag = ETAG_IGNORE
                    } else {
                        e = (if (activematch != null) activematch else longestmatch)
                    }
                } else {
                    etag = classify(e)
                }
                if (etag != ETAG_IGNORE) {
                    e.record(yend, etag)
                    links.add(CurveLink(e.curve, ystart, yend, etag))
                }
                cur++
            }
            // assert(getState() == AreaOp.RSTAG_OUTSIDE);
            if (state != RSTAG_OUTSIDE) {
                println("Still inside at end of active edge list!")
                println("num curves = " + (right - left))
                println("num links = " + links.size)
                println("y top = " + yrange[0])
                if (right < numedges) {
                    println(
                        "y top of next curve = " +
                                edgelist[right].curve.yTop
                    )
                } else {
                    println("no more curves")
                }
                cur = left
                while (cur < right) {
                    e = edgelist[cur]
                    println(e)
                    val eq: Int = e.equivalence
                    if (eq != 0) {
                        println("  was equal to $eq...")
                    }
                    cur++
                }
            }
            if (false) {
                println("new links:")
                for (i in links.indices) {
                    val link: CurveLink = links.get(i)!!
                    println("  " + link.subCurve)
                }
            }
            resolveLinks(subcurves, chains, links)
            links.clear()
            // Finally capture the bottom of the valid Y range as the top
            // of the next Y range.
            yrange[0] = yend
        }
        finalizeSubCurves(subcurves, chains)
        val ret: MutableList<Curve> = ArrayList<Curve>()
        for (subcurve in subcurves) {
            var link: CurveLink = subcurve!!
            ret.add(link.moveto)
            var nextlink: CurveLink = link
            while (nextlink.next.also { nextlink = it!! } != null) {
                if (!link.absorb(nextlink)) {
                    ret.add(link.subCurve)
                    link = nextlink
                }
            }
            ret.add(link.subCurve)
        }
        return ret
    }

    abstract fun newRow()
    abstract fun classify(e: Edge): Int
    abstract val state: Int

    abstract class CAGOp : AreaOp() {
        var inLeft = false
        var inRight = false
        var inResult = false
        override fun newRow() {
            inLeft = false
            inRight = false
            inResult = false
        }

        override fun classify(e: Edge): Int {
            if (e.curveTag === CTAG_LEFT) {
                inLeft = !inLeft
            } else {
                inRight = !inRight
            }
            val newClass = newClassification(inLeft, inRight)
            if (inResult == newClass) {
                return ETAG_IGNORE
            }
            inResult = newClass
            return if (newClass) ETAG_ENTER else ETAG_EXIT
        }

        override val state: Int
            get() = if (inResult) RSTAG_INSIDE else RSTAG_OUTSIDE

        abstract fun newClassification(
            inLeft: Boolean,
            inRight: Boolean
        ): Boolean
    }

    class AddOp : CAGOp() {
        override fun newClassification(inLeft: Boolean, inRight: Boolean): Boolean {
            return inLeft || inRight
        }
    }

    class SubOp : CAGOp() {
        override fun newClassification(inLeft: Boolean, inRight: Boolean): Boolean {
            return inLeft && !inRight
        }
    }

    class IntOp : CAGOp() {
        override fun newClassification(inLeft: Boolean, inRight: Boolean): Boolean {
            return inLeft && inRight
        }
    }

    class XorOp : CAGOp() {
        override fun newClassification(inLeft: Boolean, inRight: Boolean): Boolean {
            return inLeft != inRight
        }
    }

    class NZWindOp : AreaOp() {
        private var count = 0
        override fun newRow() {
            count = 0
        }

        override fun classify(e: Edge): Int {
            // Note: the right curves should be an empty set with this op...
            // assert(e.getCurveTag() == CTAG_LEFT);
            var newCount = count
            val type = if (newCount == 0) ETAG_ENTER else ETAG_IGNORE
            newCount += e.curve.direction
            count = newCount
            return if (newCount == 0) ETAG_EXIT else type
        }

        override val state: Int
            get() = if (count == 0) RSTAG_OUTSIDE else RSTAG_INSIDE
    }

    class EOWindOp : AreaOp() {
        private var inside = false
        override fun newRow() {
            inside = false
        }

        override fun classify(e: Edge): Int {
            // Note: the right curves should be an empty set with this op...
            // assert(e.getCurveTag() == CTAG_LEFT);
            val newInside = !inside
            inside = newInside
            return if (newInside) ETAG_ENTER else ETAG_EXIT
        }

        override val state: Int
            get() = if (inside) RSTAG_INSIDE else RSTAG_OUTSIDE
    }

    companion object {
        /* Constants to tag the left and right curves in the edge list */
        const val CTAG_LEFT = 0
        const val CTAG_RIGHT = 1

        /* Constants to classify edges */
        const val ETAG_IGNORE = 0
        const val ETAG_ENTER = 1
        const val ETAG_EXIT = -1

        /* Constants used to classify result state */
        const val RSTAG_INSIDE = 1
        const val RSTAG_OUTSIDE = -1
        private val YXTopComparator: Comparator<Edge> =
            Comparator<Edge> { o1: Edge, o2: Edge ->
                val c1: Curve = o1.curve
                val c2: Curve = o2.curve
                var v1: Double
                var v2: Double
                if (c1.yTop.also { v1 = it } == c2.yTop.also { v2 = it }) {
                    if (c1.xTop.also { v1 = it } == c2.xTop.also { v2 = it }) {
                        return@Comparator 0
                    }
                }
                if (v1 < v2) {
                    return@Comparator -1
                }
                1
            }
        private val EmptyLinkList: Array<CurveLink?> = arrayOfNulls<CurveLink>(2)
        private val EmptyChainList: Array<ChainEnd?> = arrayOfNulls<ChainEnd>(2)
        private fun addEdges(edges: MutableList<Edge>, curves: MutableList<Curve>, curvetag: Int) {
            for (c in curves) {
                if (c.order > 0) {
                    edges.add(Edge(c, curvetag))
                }
            }
        }

        fun finalizeSubCurves(
            subcurves: MutableList<CurveLink?>,
            chains: MutableList<ChainEnd?>
        ) {
            val numchains: Int = chains.size
            if (numchains == 0) {
                return
            }
            if (numchains and 1 != 0) {
                throw UnsupportedOperationException("Odd number of chains!")
            }
            val endlist: Array<ChainEnd?> = chains.toTypedArray()
            var i = 1
            while (i < numchains) {
                val open: ChainEnd? = endlist[i - 1]
                val close: ChainEnd? = endlist[i]
                val subcurve: CurveLink? = open!!.linkTo(close)
                if (subcurve != null) {
                    subcurves.add(subcurve)
                }
                i += 2
            }
            chains.clear()
        }

        fun resolveLinks(
            subcurves: MutableList<CurveLink?>,
            chains: MutableList<ChainEnd?>,
            links: MutableList<CurveLink?>
        ) {
            val numlinks: Int = links.size
            val linklist: Array<CurveLink?>
            if (numlinks == 0) {
                linklist = EmptyLinkList
            } else {
                if (numlinks and 1 != 0) {
                    throw UnsupportedOperationException("Odd number of new curves!")
                }
                linklist = links.toTypedArray().copyOf(numlinks + 2)
            }
            val numchains: Int = chains.size
            val endlist: Array<ChainEnd?>
            if (numchains == 0) {
                endlist = EmptyChainList
            } else {
                if (numchains and 1 != 0) {
                    throw UnsupportedOperationException("Odd number of chains!")
                }
                endlist = chains.toTypedArray().copyOf(numchains + 2)
            }
            chains.clear()
            var chain: ChainEnd? = endlist[0]
            var nextchain: ChainEnd? = endlist[1]
            var link: CurveLink? = linklist[0]
            var nextlink: CurveLink? = linklist[1]
            var curlink = 0
            var curchain = 0
            while (chain != null || link != null) {
                /*
             * Strategy 1:
             * Connect chains or links if they are the only things left...
             */
                var connectchains = link == null
                var connectlinks = chain == null
                if (!connectchains && !connectlinks) {
                    // assert(link != null && chain != null);
                    /*
                 * Strategy 2:
                 * Connect chains or links if they close off an open area...
                 */
                    connectchains = curchain and 1 == 0 &&
                            chain!!.x === nextchain!!.x
                    connectlinks = curlink and 1 == 0 &&
                            link!!.x === nextlink!!.x
                    if (!connectchains && !connectlinks) {
                        /*
                     * Strategy 3:
                     * Connect chains or links if their successor is
                     * between them and their potential connectee...
                     */
                        val cx: Double = chain!!.x
                        val lx: Double = link!!.x
                        connectchains = nextchain != null && cx < lx &&
                                obstructs(nextchain.x, lx, curchain)
                        connectlinks = nextlink != null && lx < cx &&
                                obstructs(nextlink.x, cx, curlink)
                    }
                }
                if (connectchains) {
                    val subcurve: CurveLink? = chain!!.linkTo(nextchain)
                    if (subcurve != null) {
                        subcurves.add(subcurve)
                    }
                    curchain += 2
                    chain = endlist[curchain]
                    nextchain = endlist[curchain + 1]
                }
                if (connectlinks) {
                    val openend = ChainEnd(link!!, null)
                    val closeend = ChainEnd(nextlink!!, openend)
                    openend.setOtherEnd(closeend)
                    chains.add(openend)
                    chains.add(closeend)
                    curlink += 2
                    link = linklist[curlink]
                    nextlink = linklist[curlink + 1]
                }
                if (!connectchains && !connectlinks) {
                    // assert(link != null);
                    // assert(chain != null);
                    // assert(chain.getEtag() == link.getEtag());
                    chain!!.addLink(link!!)
                    chains.add(chain)
                    curchain++
                    chain = nextchain
                    nextchain = endlist[curchain + 1]
                    curlink++
                    link = nextlink
                    nextlink = linklist[curlink + 1]
                }
            }
            if (chains.size and 1 != 0) {
                println("Odd number of chains!")
            }
        }

        /*
     * Does the position of the next edge at v1 "obstruct" the
     * connectivity between current edge and the potential
     * partner edge which is positioned at v2?
     *
     * Phase tells us whether we are testing for a transition
     * into or out of the interior part of the resulting area.
     *
     * Require 4-connected continuity if this edge and the partner
     * edge are both "entering into" type edges
     * Allow 8-connected continuity for "exiting from" type edges
     */
        fun obstructs(v1: Double, v2: Double, phase: Int): Boolean {
            return if (phase and 1 == 0) v1 <= v2 else v1 < v2
        }
    }
}