/*
 * Copyright (c) 2020 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.geom

import com.macrofocus.common.collection.Arrays.toString

/**
 * Representation for different types of [Segment]s.
 *
 * @see .MOVE_TO
 *
 * @see .LINE_TO
 *
 * @see .QUAD_TO
 *
 * @see .CUBIC_TO
 */
class Segment {
    /**
     * Returns the type of this [Segment].
     *
     * @return the type of this [Segment]
     *
     * @see .MOVE_TO
     *
     * @see .LINE_TO
     *
     * @see .QUAD_TO
     *
     * @see .CUBIC_TO
     */
    var type = 0
    private var points: Array<Point2D>? = null

    constructor() {}

    /**
     * Constructs a new [Segment] of the given type. The passed-in
     * [Point2D]s are associated with this [Segment].
     *
     * @param type   The type of the new [Segment]. It is one of
     *
     *  * [.MOVE_TO]
     *  * [.LINE_TO]
     *  * [.QUAD_TO]
     *  * [.CUBIC_TO]
     *
     * @param points the [Point2D]s to associate with this [Segment]
     */
    constructor(type: Int, vararg points: Point2D) {
        when (type) {
            MOVE_TO -> require(!(points == null || points.size != 1)) {
                ("A Segment of type MOVE_TO has to be associate with exactly 1 point: new Segment("
                        + type + ", " + points + ')')
            }
            LINE_TO -> require(!(points == null || points.size != 1)) {
                ("A Segment of type LINE_TO has to be associate with exactly 1 point: new Segment("
                        + type + ", " + points + ')')
            }
            QUAD_TO -> require(!(points == null || points.size != 2)) {
                ("A Segment of type QUAD_TO has to be associate with exactly 2 points: new Segment("
                        + type + ", " + points + ')')
            }
            CUBIC_TO -> require(!(points == null || points.size != 3)) {
                ("A Segment of type CUBIC_TO has to be associate with exactly 3 point: new Segment("
                        + type + ", " + points + ')')
            }
            CLOSE -> require(!(points != null && points.size != 0)) {
                ("A Segment of type CLOSE is not to be associated with any points: new Segment("
                        + type + ", " + points + ')')
            }
            else -> throw IllegalArgumentException(
                "You can only create Segments of types MOVE_TO, LINE_TO, QUAD_TO, or CUBIC_TO: new Segment("
                        + type + ", " + points + ')'
            )
        }
        this.type = type
        this.points = arrayOf(*points)
    }

    fun getPoints(): Array<Point2D>? {
        return points
    }

    fun setPoints(points: Array<Point2D>) {
        this.points = points
    }

    override fun toString(): String {
        return "Segment{" +
                "type=" + type +
                ", points=" + toString(arrayOf(points)) +
                '}'
    }

    companion object {
        /**
         * A [.MOVE_TO] [Segment] represents a change of position
         * while piecewise building a [Path], without inserting a new
         * curve.
         *
         * @see Path.moveTo
         */
        const val MOVE_TO = 0

        /**
         * A [.LINE_TO] [Segment] represents a [Line] from the
         * previous position of a [Path] to the [Point2D] at index 0
         * associated with the [Segment].
         *
         * @see Path.lineTo
         */
        const val LINE_TO = 1

        /**
         * A [.QUAD_TO] [Segment] represents a
         * [QuadraticCurve] from the previous position of a [Path]
         * to the [Point2D] at index 1 associated with the [Segment].
         * The [Point2D] at index 0 is used as the handle [Point2D] of
         * the [QuadraticCurve].
         *
         * @see Path.quadTo
         */
        const val QUAD_TO = 2

        /**
         * A [.CUBIC_TO] [Segment] represents a [CubicCurve]
         * from the previous position of a [Path] to the [Point2D] at
         * index 2 associated with the [Segment]. The [Point2D]s at
         * indices 0 and 1 are used as the handle [Point2D]s of the
         * [CubicCurve].
         *
         * @see Path.cubicTo
         */
        const val CUBIC_TO = 3

        /**
         * A [.CLOSE] [Segment] represents the link from the current
         * position of a [Path] to the position of the last
         * [.MOVE_TO] [Segment].
         *
         * @see Path.close
         */
        const val CLOSE = 4
    }
}