package org.mkui.geom.curve

/**
 * Geom contains static methods for calculating intersections, angles and distances.
 */
internal object Geom {
    val INTERSECT = Any()
    private val PARALLEL = Any()

    /**
     * Computes the distance between a line segment (a, b) and a point (c) in n-dimensions.  Arrays a, b, and c must
     * have length greater or equal to n.  Array d must have length greater than n.  The location of the closest
     * point on the line is stored in d.  The parametric value is stored at index location n in d, and its value is
     * in the range [0, 1].
     */
    fun ptSegDistSq(a: DoubleArray?, b: DoubleArray?, c: DoubleArray?, d: DoubleArray, n: Int): Double {
        for (i in 0 until n) d[i] = b!![i] - a!![i]
        var f = 0.0
        for (i in 0 until n) f += d[i] * d[i]
        var t = 0.0
        if (f != 0.0) {
            var g = 0.0
            for (i in 0 until n) g += d[i] * (c!![i] - a!![i])
            t = g / f
        }
        if (t < 0.0) t = 0.0 else if (t > 1.0) t = 1.0
        for (i in 0 until n) d[i] = a!![i] + t * d[i]
        d[n] = t
        var distSq = 0.0
        for (i in 0 until n) {
            val h = c!![i] - d[i]
            distSq += h * h
        }
        return distSq
    }

    /**
     * Calculates the intersection location of the line segments formed by (x1, y1), (x2, y2) and (x3, y3), (x4, y4).
     * If the line segments are determined to be parallel, then Geom.PARALLEL is returned and no further computations
     * are done.  If the segments do not cross each other then null is returned and no further computations are done.
     * Otherwise the intersection location is stored in index locations 0 and 1 of the specified array.  The parametric
     * value with respect to the first line segment is stored in index location 2.  If there is an intersection, then
     * the returned value is Geom.INTERSECT.
     */
    fun getSegSegIntersection(
        x1: Double,
        y1: Double,
        x2: Double,
        y2: Double,
        x3: Double,
        y3: Double,
        x4: Double,
        y4: Double,
        result: DoubleArray?
    ): Any? {
        val bx = x2 - x1
        val by = y2 - y1
        val dx = x4 - x3
        val dy = y4 - y3
        val b_dot_d_perp = bx * dy - by * dx
        if (b_dot_d_perp == 0.0) return PARALLEL
        val cx = x3 - x1
        val cy = y3 - y1
        val t = (cx * dy - cy * dx) / b_dot_d_perp
        if (t < 0 || t > 1) return null
        val u = (cx * by - cy * bx) / b_dot_d_perp
        if (u < 0 || u > 1) return null
        if (result != null) {
            result[0] = x1 + t * bx
            result[1] = y1 + t * by
            result[2] = t
        }
        return INTERSECT
    }
}