/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.palette

import org.mkui.color.CPColor

/**
 * Palette composed of a defined list of discrete colors.
 */
open class FixedPalette(override var isInverted: Boolean, var colors: List<CPColor>) : AbstractPalette(),
    MutablePalette {

    constructor(vararg colors: Int) : this(false, *colors) {
    }

    constructor(inverted: Boolean, vararg colors: Int) : this(
        false,
        toCPColorList(colors)
    ) {
    }

    constructor(vararg colors: CPColor) : this(false, colors.toMutableList()) {
    }

    constructor(inverted: Boolean, vararg colors: CPColor) : this(inverted, colors.toMutableList()) {
    }

    constructor(colors: List<CPColor>) : this(false, colors) {
    }

    override fun getColor(fraction: Double): CPColor {
        var fraction = fraction
        var inverted = isInverted
        when (cycle) {
            Palette.Cycle.REFLECT -> {
                inverted = fraction.toInt() % 2 == 0
                fraction = fraction - fraction.toInt()
            }
            Palette.Cycle.REPEAT -> fraction = fraction - fraction.toInt()
        }
        return if (fraction < lowestFraction) {
            getColor(lowestFraction)
        } else if (fraction > highestFraction) {
            getColor(highestFraction)
        } else {
            val index: Int = getEntry(if (!inverted) fraction else 1 - fraction, colorCount, colors!!.size)
            colors!![index]
        }
    }

    protected open val lowestFraction: Double
        protected get() = 0.0
    protected open val highestFraction: Double
        protected get() = 1.0

    override fun getColorAt(index: Int): CPColor {
        return if (colorCount === colors!!.size) {
            if (index < 0) {
                getColorAt(0)
            } else if (index > colorCount - 1) {
                getColorAt(colorCount - 1)
            } else {
                colors!![index]
            }
        } else {
            getColor(colorCount / index.toDouble())
        }
    }

    override fun addPaletteListener(listener: PaletteListener) {
        // Noop
    }

    override fun removePaletteListener(listener: PaletteListener) {
        // Noop
    }

//    override fun setColorCount(binCount: Int) {
//        throw UnsupportedOperationException()
//    }

    override var cycle: Palette.Cycle?
        get() = super.cycle
        set(value) {throw UnsupportedOperationException()}

    override fun toString(): String {
        return getColor(0.0).toString() + "-" + getColor(1.0)
        //        final StringBuffer sb = new StringBuffer("FixedPalette{");
//        for (int i = 0; i < colors.size(); i++) {
//            if(i > 0) {
//                sb.append(", ");
//            }
//            CPColor color = colors.get(i);
//            sb.append("color[").append(i).append("]=").append(color.toJavaScriptColor());
//        }
//        sb.append(", inverted=").append(inverted);
//        sb.append('}');
//        return sb.toString();
    }

    companion object {
        private fun toCPColorList(colors: IntArray): List<CPColor> {
            val list: MutableList<CPColor> = ArrayList<CPColor>(colors.size)
            for (color in colors) {
                list.add(CPColor(color))
            }
            return list
        }
    }

    init {
        colorCount = if (colors != null) this.colors!!.size else 0
    }
}