/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package org.mkui.palette

import org.mkui.color.CPColor
import org.mkui.color.CPColorFactory

/**
 * Factory class for creating prebuild palettes of various types.
 */
class PaletteFactory constructor() {
    private val factory = CPColorFactory.instance
    private val negpos: Entry
    private val treemapper: Entry
    private val composite: Entry
    private val bty: Entry
    private val heated: Entry
    private val rainy: Entry
    private val linred: Entry
    private val publimedia: Entry
    private val magenta: Entry
    private val rainbow: Entry
    private val locs: Entry
    private val ocs: Entry
    private val lingrey: Entry
    private val YlOrRd: Entry
    private val PRGn: Entry
    private val PuOr: Entry
    private val RdGy: Entry
    private val Spectral: Entry
    private val Grays: Entry
    private val PuBuGn: Entry
    private val RdPu: Entry
    private val BuPu: Entry
    private val YlOrBr: Entry
    private val Greens: Entry
    private val BuGn: Entry
    private val Accents: Entry
    private val GnBu: Entry
    private val PuRd: Entry
    private val Purples: Entry
    private val RdYlGn: Entry
    private val Paired: Entry
    private val Blues: Entry
    private val RdBu: Entry
    private val Oranges: Entry
    private val RdYlBu: Entry
    private val PuBu: Entry
    private val OrRd: Entry
    private val Set3: Entry
    private val Set2: Entry
    private val Set1: Entry
    private val Reds: Entry
    private val PiYG: Entry
    private val Dark2: Entry
    private val YlGn: Entry
    private val BrBG: Entry
    private val YlGnBu: Entry
    private val Pastel2: Entry
    private val Pastel1: Entry

    /**
     * Lab-based isoluminant rainbow with constant luminance L*=60.
     * For interval data displayed with external lighting.
     */
    private val IsoL: Entry

    /**
     * Lab-based rainbow scheme with cubic-law luminance (default).
     * For interval data displayed without external lighting.
     */
    private val CubicL: Entry

    /**
     * Diverging Black-blue-cyan-white-yellow-red-black scheme.
     * For ratio data (ordered, constant scale, natural zero).
     */
    private val Edge: Entry
    private val coolwarm: Entry
    private val viridis: Entry
    private val magma: Entry
    private val plasma: Entry
    private val inferno: Entry
    private val cividis: Entry
    private val twilight: Entry
    private val entries: MutableList<Entry> = ArrayList<Entry>()
    fun getEntries(): List<Entry> {
        return entries
    }

    fun getPalette(name: String): Palette {
        return get(name)!!.getPalette()
    }

    fun add(entry: Entry) {
        entries.add(entry)
    }

    operator fun get(name: String): Entry? {
        for (entry in entries) {
            if (entry.name == name) {
                return entry
            }
        }
        return null
    }

    fun find(palette: org.mkui.palette.Palette?): Entry? {
        for (entry in entries) {
            if (entry.getPalette().equals(palette)) {
                return entry
            }
        }
        return null
    }

    fun createDefaultDivergingPalette(): FixedPalette {
        return RdYlGn.getPalette()
    }

    fun createDefaultSequentialPalette(): FixedPalette {
        return Blues.getPalette()
    }

    fun createDefaultQualititativePalette(): FixedPalette {
        return Paired.getPalette()
    }

    fun createColorGradient(name: String): MutablePalette? {
        for (entry in entries) {
            if (entry.name == name) {
                return entry.getPalette()
            }
        }
        return null
    }

//    fun createFixedPalette(vararg colors: CPColor): FixedPalette {
//        return FixedPalette(*colors)
//    }
//
//    fun createInterpolatedPalette(vararg entries: InterpolatedPalette.Entry): InterpolatedPalette {
//        return InterpolatedPalette(entries)
//    }
//
//    fun createCustomPalette(vararg entries: CustomPalette.Entry): CustomPalette {
//        return CustomPalette(factory, entries)
//    }

    enum class Type {
        SEQUENTIAL, DIVERGING, QUALITATIVE
    }

    class Entry(val name: String, val type: Type, colors: FixedPalette) {
        private val palette: FixedPalette
        fun getPalette(): FixedPalette {
            return palette
        }

        init {
            palette = colors
        }
    }

    init {
        negpos = Entry(
            "negpos", Type.DIVERGING, InterpolatedPalette(
                InterpolatedPalette.Entry(0.0, factory.createRGBColor(255, 0, 0)),
                InterpolatedPalette.Entry(0.4980, factory.createRGBColor(255, 230, 230)),
                InterpolatedPalette.Entry(0.4981, factory.createRGBColor(255, 255, 255)),
                InterpolatedPalette.Entry(0.5020, factory.createRGBColor(255, 255, 255)),
                InterpolatedPalette.Entry(0.5021, factory.createRGBColor(230, 255, 230)),
                InterpolatedPalette.Entry(1.0, factory.createRGBColor(0, 128, 0))
            )
        )
        treemapper = Entry(
            "treemapper", Type.DIVERGING, InterpolatedPalette(
                InterpolatedPalette.Entry(0.0, factory.createRGBColor(255, 0, 0)),
                InterpolatedPalette.Entry(0.5, factory.createRGBColor(255, 255, 255)),
                InterpolatedPalette.Entry(1.0, factory.createRGBColor(0, 128, 0))
            )
        )
        composite = Entry(
            "composite", Type.DIVERGING, InterpolatedPalette(
                InterpolatedPalette.Entry(0.0, factory.createRGBColor(255, 0, 0)),
                InterpolatedPalette.Entry(0.19327731092437, factory.createRGBColor(247, 52, 0)),
                InterpolatedPalette.Entry(0.428571428571429, factory.createRGBColor(83, 10, 0)),
                InterpolatedPalette.Entry(0.563025210084034, factory.createRGBColor(19, 58, 0)),
                InterpolatedPalette.Entry(1.0, factory.createRGBColor(127, 255, 0))
            )
        )
        bty = Entry(
            "bty", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    -16316418,
                    -15263748,
                    -14803206,
                    -14408456,
                    -14145289,
                    -13882123,
                    -13684749,
                    -13487374,
                    -13355792,
                    -13158417,
                    -13026834,
                    -12895252,
                    -12763669,
                    -12632086,
                    -12500503,
                    -12434713,
                    -12303130,
                    -12237339,
                    -12105756,
                    -12039965,
                    -11908382,
                    -11842591,
                    -11776799,
                    -11645216,
                    -11579425,
                    -11513634,
                    -11447843,
                    -11382051,
                    -11250468,
                    -11184677,
                    -11118886,
                    -11053094,
                    -10987303,
                    -10921512,
                    -10855720,
                    -10789929,
                    -10724138,
                    -10658346,
                    -10592555,
                    -10526763,
                    -10460972,
                    -10395180,
                    -10329389,
                    -10329390,
                    -10263598,
                    -10197807,
                    -10132015,
                    -10066224,
                    -10000432,
                    -9934640,
                    -9868849,
                    -9868849,
                    -9803058,
                    -9737266,
                    -9671475,
                    -9605683,
                    -9539892,
                    -9539892,
                    -9474100,
                    -9408309,
                    -9342517,
                    -9276726,
                    -9276726,
                    -9210934,
                    -9145143,
                    -9079351,
                    -9013560,
                    -9013560,
                    -8947768,
                    -8881977,
                    -8816185,
                    -8816185,
                    -8750394,
                    -8684602,
                    -8618810,
                    -8618811,
                    -8553019,
                    -8487227,
                    -8421436,
                    -8355644,
                    -8355645,
                    -8289853,
                    -8224061,
                    -8224062,
                    -8158270,
                    -8092478,
                    -8026687,
                    -8026687,
                    -7960895,
                    -7895104,
                    -7829312,
                    -7829312,
                    -7763521,
                    -7697729,
                    -7631937,
                    -7631938,
                    -7566146,
                    -7500354,
                    -7434563,
                    -7434563,
                    -7368771,
                    -7302980,
                    -7302980,
                    -7237188,
                    -7171397,
                    -7105605,
                    -7105605,
                    -7039814,
                    -6974022,
                    -6974022,
                    -6908231,
                    -6842439,
                    -6776647,
                    -6776648,
                    -6710856,
                    -6645064,
                    -6645065,
                    -6579273,
                    -6513482,
                    -6447690,
                    -6447690,
                    -6381899,
                    -6316107,
                    -6316107,
                    -6250316,
                    -6184524,
                    -6118732,
                    -6118733,
                    -6052941,
                    -5987150,
                    -5987150,
                    -5921358,
                    -5855567,
                    -5789775,
                    -5789776,
                    -5723984,
                    -5658192,
                    -5658193,
                    -5592401,
                    -5526610,
                    -5460818,
                    -5460819,
                    -5395027,
                    -5329235,
                    -5329236,
                    -5263444,
                    -5197653,
                    -5131861,
                    -5131862,
                    -5066070,
                    -5000279,
                    -5000279,
                    -4934488,
                    -4868696,
                    -4868697,
                    -4802905,
                    -4737114,
                    -4671322,
                    -4671323,
                    -4605531,
                    -4539740,
                    -4539740,
                    -4473949,
                    -4408157,
                    -4342366,
                    -4342366,
                    -4276575,
                    -4210783,
                    -4210784,
                    -4144993,
                    -4079201,
                    -4013410,
                    -4013410,
                    -3947619,
                    -3881827,
                    -3881828,
                    -3816037,
                    -3750245,
                    -3684454,
                    -3684455,
                    -3618663,
                    -3552872,
                    -3552873,
                    -3487081,
                    -3421290,
                    -3355499,
                    -3355499,
                    -3289708,
                    -3223917,
                    -3223918,
                    -3158126,
                    -3092335,
                    -3026544,
                    -3026545,
                    -2960753,
                    -2894962,
                    -2894963,
                    -2829172,
                    -2763381,
                    -2697590,
                    -2697590,
                    -2631799,
                    -2566008,
                    -2566009,
                    -2500218,
                    -2434427,
                    -2368636,
                    -2368637,
                    -2302846,
                    -2237055,
                    -2237056,
                    -2171265,
                    -2105474,
                    -2039683,
                    -2039684,
                    -1973893,
                    -1908102,
                    -1908103,
                    -1842313,
                    -1776522,
                    -1710731,
                    -1710732,
                    -1644942,
                    -1579151,
                    -1513360,
                    -1513362,
                    -1447571,
                    -1381781,
                    -1381782,
                    -1315992,
                    -1250201,
                    -1184411,
                    -1184412,
                    -1118622,
                    -1052832,
                    -1052834,
                    -987044,
                    -921253,
                    -855463,
                    -855466,
                    -789676,
                    -723886,
                    -658096,
                    -658099,
                    -592310,
                    -526520,
                    -526523,
                    -460735,
                    -394946,
                    -329158,
                    -329162,
                    -263375,
                    -197588,
                    -131803,
                    -131812,
                    -66035
                )
            )
        )
        heated = Entry(
            "heated", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    -10544128,
                    -10347008,
                    -10215680,
                    -10084096,
                    -10018304,
                    -9886976,
                    -9821184,
                    -9689600,
                    -9623808,
                    -9492480,
                    -9426432,
                    -9295104,
                    -9229312,
                    -9097984,
                    -8966400,
                    -8966400,
                    -8900608,
                    -8769280,
                    -8703232,
                    -8571904,
                    -8571904,
                    -8506112,
                    -8374784,
                    -8308736,
                    -8308736,
                    -8177408,
                    -8111616,
                    -7980288,
                    -7980288,
                    -7848704,
                    -7782912,
                    -7782912,
                    -7651584,
                    -7651584,
                    -7585536,
                    -7454208,
                    -7454208,
                    -7388416,
                    -7388416,
                    -7257088,
                    -7257088,
                    -7191040,
                    -7191040,
                    -7059712,
                    -7059712,
                    -6993920,
                    -6993920,
                    -6862592,
                    -6862592,
                    -6731008,
                    -6731008,
                    -6665216,
                    -6665216,
                    -6665216,
                    -6533888,
                    -6533888,
                    -6467840,
                    -6467840,
                    -6336512,
                    -6336512,
                    -6336512,
                    -6270720,
                    -6270720,
                    -6139392,
                    -6139392,
                    -6073344,
                    -6073344,
                    -5942016,
                    -5942016,
                    -5876224,
                    -5876224,
                    -5876224,
                    -5744896,
                    -5744896,
                    -5613312,
                    -5613312,
                    -5547520,
                    -5547520,
                    -5416192,
                    -5416192,
                    -5350144,
                    -5350144,
                    -5218816,
                    -5218816,
                    -5153024,
                    -5153024,
                    -5021696,
                    -5021696,
                    -4955648,
                    -4824320,
                    -4824320,
                    -4758528,
                    -4758528,
                    -4626944,
                    -4626944,
                    -4495616,
                    -4495616,
                    -4429824,
                    -4429824,
                    -4298496,
                    -4232448,
                    -4232448,
                    -4101120,
                    -4101120,
                    -4035328,
                    -3904000,
                    -3904000,
                    -3837952,
                    -3837952,
                    -3706624,
                    -3640832,
                    -3640832,
                    -3509248,
                    -3509248,
                    -3377920,
                    -3312128,
                    -3312128,
                    -3180800,
                    -3114752,
                    -3114752,
                    -2983424,
                    -2917632,
                    -2917632,
                    -2786304,
                    -2720256,
                    -2720256,
                    -2588928,
                    -2523136,
                    -2523136,
                    -2391552,
                    -2260224,
                    -2260224,
                    -2194432,
                    -2063104,
                    -2063104,
                    -1997056,
                    -1865728,
                    -1865728,
                    -1799936,
                    -1668608,
                    -1668608,
                    -1602560,
                    -1471232,
                    -1471232,
                    -1405440,
                    -1273856,
                    -1273856,
                    -1142528,
                    -1076736,
                    -945408,
                    -945408,
                    -879360,
                    -748032,
                    -748032,
                    -682240,
                    -550656,
                    -550656,
                    -484864,
                    -353536,
                    -287744,
                    -287744,
                    -156160,
                    -24832,
                    -24832,
                    -24576,
                    -24320,
                    -23808,
                    -23808,
                    -23552,
                    -23296,
                    -22784,
                    -22784,
                    -22528,
                    -22272,
                    -22272,
                    -22016,
                    -21504,
                    -21248,
                    -21248,
                    -20992,
                    -20736,
                    -20224,
                    -19968,
                    -19712,
                    -19200,
                    -19200,
                    -18944,
                    -18688,
                    -18432,
                    -17657,
                    -17398,
                    -17138,
                    -16622,
                    -16363,
                    -16103,
                    -15587,
                    -15068,
                    -14808,
                    -14293,
                    -13773,
                    -13258,
                    -12739,
                    -12479,
                    -11704,
                    -11444,
                    -10669,
                    -10149,
                    -9374,
                    -8855,
                    -8339,
                    -7820,
                    -7045,
                    -6010,
                    -5490,
                    -4715,
                    -4196,
                    -3936,
                    -3161,
                    -2386,
                    -1866,
                    -1607,
                    -831,
                    -572,
                    -52,
                    -49,
                    -45,
                    -38,
                    -34,
                    -31,
                    -27,
                    -23,
                    -20,
                    -16,
                    -12,
                    -9
                )
            )
        )
        rainy = Entry(
            "rainy",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    -16385,
                    -1592075,
                    -3167765,
                    -4743455,
                    -6319145,
                    -7894835,
                    -9470525,
                    -11046215,
                    -12621905,
                    -14197595,
                    -15773285
                )
            )
        )
        linred = Entry(
            "linred", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    -16777216,
                    -16777216,
                    -16777216,
                    -16777216,
                    -16777216,
                    -16777216,
                    -16777216,
                    -16711680,
                    -16711680,
                    -16711680,
                    -16711680,
                    -16711680,
                    -16711680,
                    -16711680,
                    -16711680,
                    -16711680,
                    -16646144,
                    -16646144,
                    -16646144,
                    -16646144,
                    -16646144,
                    -16646144,
                    -16646144,
                    -16580608,
                    -16580608,
                    -16580608,
                    -16580608,
                    -16580608,
                    -16580608,
                    -16580608,
                    -16515072,
                    -16515072,
                    -16515072,
                    -16515072,
                    -16515072,
                    -16449536,
                    -16449536,
                    -16449536,
                    -16449536,
                    -16449536,
                    -16384000,
                    -16384000,
                    -16384000,
                    -16384000,
                    -16384000,
                    -16318464,
                    -16318464,
                    -16318464,
                    -16318464,
                    -16318464,
                    -16252928,
                    -16252928,
                    -16187392,
                    -16187392,
                    -16187392,
                    -16187392,
                    -16121856,
                    -16121856,
                    -16121856,
                    -16121856,
                    -16121856,
                    -16056320,
                    -16056320,
                    -15990784,
                    -15990784,
                    -15990784,
                    -15925248,
                    -15925248,
                    -15859712,
                    -15859712,
                    -15794176,
                    -15794176,
                    -15794176,
                    -15728640,
                    -15728640,
                    -15663104,
                    -15663104,
                    -15597568,
                    -15597568,
                    -15532032,
                    -15532032,
                    -15532032,
                    -15532032,
                    -15532032,
                    -15466496,
                    -15466496,
                    -15335424,
                    -15335424,
                    -15335424,
                    -15269888,
                    -15269888,
                    -15204352,
                    -15204352,
                    -15073280,
                    -15073280,
                    -15073280,
                    -15007744,
                    -15007744,
                    -14876672,
                    -14876672,
                    -14811136,
                    -14811136,
                    -14680064,
                    -14680064,
                    -14680064,
                    -14680064,
                    -14680064,
                    -14548992,
                    -14548992,
                    -14483456,
                    -14483456,
                    -14483456,
                    -14352384,
                    -14352384,
                    -14221312,
                    -14221312,
                    -14090240,
                    -14090240,
                    -14090240,
                    -13959168,
                    -13959168,
                    -13828096,
                    -13828096,
                    -13762560,
                    -13762560,
                    -13762560,
                    -13697024,
                    -13697024,
                    -13565952,
                    -13565952,
                    -13434880,
                    -13434880,
                    -13369344,
                    -13369344,
                    -13369344,
                    -13238272,
                    -13238272,
                    -13107200,
                    -13107200,
                    -12910592,
                    -12910592,
                    -12910592,
                    -12779520,
                    -12779520,
                    -12582912,
                    -12582912,
                    -12386304,
                    -12386304,
                    -12386304,
                    -12255232,
                    -12255232,
                    -12058624,
                    -12058624,
                    -11862016,
                    -11862016,
                    -11796480,
                    -11796480,
                    -11796480,
                    -11665408,
                    -11665408,
                    -11468800,
                    -11468800,
                    -11272192,
                    -11272192,
                    -11272192,
                    -11075584,
                    -11075584,
                    -10813440,
                    -10813440,
                    -10616832,
                    -10616832,
                    -10616832,
                    -10420224,
                    -10420224,
                    -10158080,
                    -10158080,
                    -9961472,
                    -9961472,
                    -9764864,
                    -9764864,
                    -9764864,
                    -9699328,
                    -9699328,
                    -9437184,
                    -9437184,
                    -9175040,
                    -9175040,
                    -9175040,
                    -8912896,
                    -8912896,
                    -8650752,
                    -8650752,
                    -8388608,
                    -8388608,
                    -8388608,
                    -8126464,
                    -8126464,
                    -7864320,
                    -7864320,
                    -7536640,
                    -7536640,
                    -7274496,
                    -7274496,
                    -7274496,
                    -7143424,
                    -7143424,
                    -6946816,
                    -6946816,
                    -6684672,
                    -6684672,
                    -6684672,
                    -6356992,
                    -6356992,
                    -6029312,
                    -6029312,
                    -5701632,
                    -5701632,
                    -5701632,
                    -5373952,
                    -5373952,
                    -5046272,
                    -5046272,
                    -4653056,
                    -4653056,
                    -4325376,
                    -4325376,
                    -4325376,
                    -3997696,
                    -3997696,
                    -3997696,
                    -3997696,
                    -3604480,
                    -3604480,
                    -3604480,
                    -3211264,
                    -3211264,
                    -2883584,
                    -2883584,
                    -2490368,
                    -2490368,
                    -2490368,
                    -2097152,
                    -2097152,
                    -1703936,
                    -1703936,
                    -1245184,
                    -1245184,
                    -851968,
                    -851968,
                    -851968,
                    -458752,
                    -458752,
                    -262144,
                    -262144,
                    -262144,
                    -65536
                )
            )
        )
        publimedia = Entry(
            "publimedia",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    -1,
                    -3634808,
                    -3778215,
                    -4182484,
                    -6087908,
                    -8907752,
                    -11333360,
                    -16777216,
                    -16777216,
                    -16777216,
                    -16777216
                )
            )
        )
        magenta = Entry(
            "magenta", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    -16777216,
                    -14155776,
                    -13107196,
                    -12779511,
                    -12582900,
                    -12451826,
                    -12255215,
                    -11993068,
                    -11927530,
                    -11665383,
                    -11599845,
                    -11337698,
                    -11206625,
                    -11141087,
                    -10878940,
                    -10813402,
                    -10682329,
                    -10551255,
                    -10485717,
                    -10223570,
                    -10092497,
                    -10026959,
                    -9895885,
                    -9764812,
                    -9699274,
                    -9568201,
                    -9437127,
                    -9437127,
                    -9371590,
                    -9240516,
                    -9109442,
                    -8978369,
                    -8912831,
                    -8781758,
                    -8650684,
                    -8585146,
                    -8454073,
                    -8322999,
                    -8322999,
                    -8257462,
                    -8126388,
                    -7995314,
                    -7864241,
                    -7798703,
                    -7667630,
                    -7536556,
                    -7471018,
                    -7339945,
                    -7208871,
                    -7143334,
                    -7012260,
                    -6881186,
                    -6881186,
                    -6750113,
                    -6684575,
                    -6553502,
                    -6422428,
                    -6356890,
                    -6225817,
                    -6094743,
                    -6029206,
                    -5898132,
                    -5767059,
                    -5635985,
                    -5570447,
                    -5439374,
                    -5308300,
                    -5242763,
                    -5111689,
                    -4980615,
                    -4980615,
                    -4915078,
                    -4784004,
                    -4652931,
                    -4521857,
                    -4456319,
                    -4325246,
                    -4194172,
                    -4128635,
                    -3997561,
                    -3866487,
                    -3800950,
                    -3669876,
                    -3538803,
                    -3407729,
                    -3407729,
                    -3342191,
                    -3211118,
                    -3080044,
                    -3014507,
                    -2883433,
                    -2752359,
                    -2686822,
                    -2555748,
                    -2424675,
                    -2293601,
                    -2228064,
                    -2228064,
                    -2096990,
                    -1965916,
                    -1900379,
                    -1769305,
                    -1638232,
                    -1572694,
                    -1441620,
                    -1310547,
                    -1179473,
                    -1179473,
                    -1113936,
                    -982862,
                    -851788,
                    -786251,
                    -655177,
                    -523592,
                    -457542,
                    -457542,
                    -457542,
                    -325956,
                    -325956,
                    -194115,
                    -194115,
                    -62529,
                    -62529,
                    -62016,
                    -62016,
                    -62016,
                    -61246,
                    -60732,
                    -60219,
                    -60219,
                    -59449,
                    -58936,
                    -58422,
                    -57652,
                    -57139,
                    -56625,
                    -55856,
                    -55856,
                    -55342,
                    -54829,
                    -54059,
                    -53545,
                    -53032,
                    -52262,
                    -51749,
                    -51749,
                    -51235,
                    -50721,
                    -49952,
                    -49438,
                    -48925,
                    -48155,
                    -48155,
                    -47641,
                    -47128,
                    -46358,
                    -45845,
                    -45331,
                    -44561,
                    -44561,
                    -44048,
                    -43534,
                    -42765,
                    -42251,
                    -41737,
                    -40968,
                    -40968,
                    -40454,
                    -39941,
                    -39171,
                    -38657,
                    -38145,
                    -38145,
                    -37633,
                    -36865,
                    -36353,
                    -35841,
                    -35841,
                    -35073,
                    -34561,
                    -34049,
                    -34049,
                    -33281,
                    -32769,
                    -32257,
                    -32257,
                    -31489,
                    -30977,
                    -30465,
                    -30465,
                    -29697,
                    -29185,
                    -28673,
                    -28673,
                    -27905,
                    -27393,
                    -26881,
                    -26881,
                    -26113,
                    -25601,
                    -25601,
                    -25089,
                    -24577,
                    -24577,
                    -23809,
                    -23297,
                    -23297,
                    -22785,
                    -22017,
                    -21505,
                    -21505,
                    -20993,
                    -20225,
                    -20225,
                    -19713,
                    -19201,
                    -19201,
                    -18433,
                    -17921,
                    -17921,
                    -17409,
                    -16641,
                    -16641,
                    -16129,
                    -15617,
                    -14849,
                    -14849,
                    -14337,
                    -13825,
                    -13057,
                    -13057,
                    -12545,
                    -12033,
                    -11521,
                    -11521,
                    -10753,
                    -10241,
                    -9729,
                    -9729,
                    -8961,
                    -8449,
                    -7937,
                    -7169,
                    -6657,
                    -6657,
                    -6145,
                    -5377,
                    -4865,
                    -4353,
                    -3585,
                    -3073,
                    -3073,
                    -2561,
                    -1793,
                    -1281,
                    -1
                )
            )
        )
        rainbow = Entry(
            "rainbow", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    -16777216,
                    -13828060,
                    -13107154,
                    -12845007,
                    -12386250,
                    -12189637,
                    -12124099,
                    -11861948,
                    -11927479,
                    -11927475,
                    -11993007,
                    -12124073,
                    -12254886,
                    -12320162,
                    -12450975,
                    -12646810,
                    -12777622,
                    -12973459,
                    -13104015,
                    -13299852,
                    -13626761,
                    -13691783,
                    -13887620,
                    -14083200,
                    -14148479,
                    -14344060,
                    -14539642,
                    -14865527,
                    -15125366,
                    -15189621,
                    -15188339,
                    -15187826,
                    -15253106,
                    -15252081,
                    -15251570,
                    -15251570,
                    -15251058,
                    -15250546,
                    -15250546,
                    -15249778,
                    -15249266,
                    -15249266,
                    -15248755,
                    -15247987,
                    -15247987,
                    -15247476,
                    -15247476,
                    -15181172,
                    -15181172,
                    -15180405,
                    -15180405,
                    -15180405,
                    -15180405,
                    -15179381,
                    -15179381,
                    -15112822,
                    -15112822,
                    -15112055,
                    -15112055,
                    -15112055,
                    -15045495,
                    -15045495,
                    -14979192,
                    -14979192,
                    -14979192,
                    -14978169,
                    -14978169,
                    -14911866,
                    -14911866,
                    -14845307,
                    -14845307,
                    -14845307,
                    -14845307,
                    -14844540,
                    -14844540,
                    -14778237,
                    -14778237,
                    -14711934,
                    -14711934,
                    -14711934,
                    -14645632,
                    -14645632,
                    -14579329,
                    -14579329,
                    -14579329,
                    -14513027,
                    -14513027,
                    -14446981,
                    -14446981,
                    -14380679,
                    -14380679,
                    -14380679,
                    -14314634,
                    -14314634,
                    -14248332,
                    -14248332,
                    -14116751,
                    -14116751,
                    -14050705,
                    -14050705,
                    -13984660,
                    -13984660,
                    -13918614,
                    -13918614,
                    -13918614,
                    -13787032,
                    -13787032,
                    -13720987,
                    -13720987,
                    -13589405,
                    -13589405,
                    -13457823,
                    -13457823,
                    -13391777,
                    -13260195,
                    -13260195,
                    -13260195,
                    -13128613,
                    -13128613,
                    -12997032,
                    -12997032,
                    -12865706,
                    -12668588,
                    -12537262,
                    -12537262,
                    -12340144,
                    -12340144,
                    -12208817,
                    -12011699,
                    -12011699,
                    -12011699,
                    -11814836,
                    -11683254,
                    -11486391,
                    -11289272,
                    -11026872,
                    -10764473,
                    -10502074,
                    -10239674,
                    -9977274,
                    -9714874,
                    -9452474,
                    -9452474,
                    -9190330,
                    -8927930,
                    -8665786,
                    -8206777,
                    -8009913,
                    -7747768,
                    -7551160,
                    -7354551,
                    -7354551,
                    -7092407,
                    -6895798,
                    -6699190,
                    -6502581,
                    -6240180,
                    -5781426,
                    -5584817,
                    -5388209,
                    -5388209,
                    -5126064,
                    -4929455,
                    -4733102,
                    -4536494,
                    -4274349,
                    -3881131,
                    -3684779,
                    -3684779,
                    -3422634,
                    -3226281,
                    -2833063,
                    -2636710,
                    -2440357,
                    -2047394,
                    -2047394,
                    -1654432,
                    -1458078,
                    -1261980,
                    -1131160,
                    -1000342,
                    -1000342,
                    -869522,
                    -738958,
                    -673676,
                    -542856,
                    -477573,
                    -477573,
                    -346755,
                    -281472,
                    -215934,
                    -150395,
                    -150395,
                    -84858,
                    -85110,
                    -19570,
                    -19567,
                    -19567,
                    -19560,
                    -19295,
                    -19292,
                    -19289,
                    -19289,
                    -19031,
                    -19030,
                    -18771,
                    -18512,
                    -18512,
                    -18253,
                    -17997,
                    -17994,
                    -17738,
                    -17738,
                    -17479,
                    -17223,
                    -16964,
                    -16964,
                    -16708,
                    -16449,
                    -16193,
                    -15678,
                    -15678,
                    -14907,
                    -14650,
                    -14136,
                    -13879,
                    -13879,
                    -13622,
                    -13365,
                    -12851,
                    -12594,
                    -12594,
                    -12080,
                    -11823,
                    -11309,
                    -10281,
                    -10024,
                    -10024,
                    -9510,
                    -9253,
                    -8739,
                    -8225,
                    -7454,
                    -6940,
                    -6426,
                    -6426,
                    -5912,
                    -5141,
                    -4627,
                    -3856,
                    -3085,
                    -2314,
                    -1543,
                    -1029,
                    -515,
                    -1
                )
            )
        )
        locs = Entry(
            "locs", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    -16777216,
                    -16777216,
                    -16777216,
                    -16711680,
                    -16646144,
                    -16646144,
                    -16580608,
                    -16580608,
                    -16515072,
                    -16449536,
                    -16449536,
                    -16384000,
                    -16318464,
                    -16318464,
                    -16252928,
                    -16187392,
                    -16187392,
                    -16121856,
                    -16056320,
                    -15990784,
                    -15925248,
                    -15859712,
                    -15794176,
                    -15728640,
                    -15663104,
                    -15597568,
                    -15532032,
                    -15466496,
                    -15400960,
                    -15335424,
                    -15269888,
                    -15138816,
                    -15073280,
                    -15007744,
                    -14942208,
                    -14811136,
                    -14745600,
                    -14614528,
                    -14548992,
                    -14483456,
                    -14352384,
                    -14221312,
                    -14155776,
                    -13959168,
                    -13828096,
                    -13762560,
                    -13565952,
                    -13434880,
                    -13303808,
                    -13238272,
                    -13107200,
                    -12976128,
                    -12845056,
                    -12713984,
                    -12582912,
                    -12386304,
                    -12255232,
                    -12124160,
                    -11927552,
                    -11796480,
                    -11534336,
                    -11468800,
                    -11272192,
                    -11141120,
                    -10944512,
                    -10747904,
                    -10616832,
                    -10420224,
                    -10223616,
                    -10027008,
                    -9830400,
                    -9633792,
                    -9437184,
                    -9240576,
                    -9109504,
                    -8781824,
                    -8519680,
                    -8388608,
                    -8192000,
                    -7929856,
                    -7929856,
                    -7929600,
                    -7929344,
                    -7929088,
                    -7928832,
                    -7928320,
                    -7928320,
                    -7927808,
                    -7927552,
                    -7927296,
                    -7927040,
                    -7926528,
                    -7926528,
                    -7926016,
                    -7925504,
                    -7925504,
                    -7924992,
                    -7924480,
                    -7924224,
                    -7923968,
                    -7923456,
                    -7923200,
                    -7922944,
                    -7922432,
                    -7921920,
                    -7921664,
                    -7921408,
                    -7920896,
                    -7920640,
                    -7920128,
                    -7919616,
                    -7919104,
                    -7918592,
                    -7918080,
                    -7917824,
                    -7917312,
                    -7916800,
                    -7916544,
                    -7916032,
                    -7915520,
                    -7915264,
                    -7914752,
                    -7913984,
                    -7913728,
                    -7913216,
                    -7912704,
                    -7912192,
                    -7911424,
                    -7911168,
                    -7910400,
                    -7909888,
                    -7909376,
                    -7908864,
                    -7908352,
                    -7907584,
                    -7907328,
                    -7906816,
                    -7906048,
                    -7905536,
                    -7904768,
                    -7904000,
                    -7903488,
                    -7902720,
                    -7902464,
                    -7901696,
                    -7900928,
                    -7900416,
                    -7899648,
                    -7898880,
                    -7898112,
                    -7897344,
                    -7896832,
                    -7895808,
                    -7895296,
                    -7894528,
                    -7893760,
                    -7892992,
                    -7891968,
                    -7891456,
                    -7890176,
                    -7889664,
                    -7888896,
                    -7888128,
                    -7887360,
                    -7886336,
                    -7885312,
                    -7884544,
                    -7883776,
                    -7882752,
                    -7881728,
                    -7880704,
                    -7879936,
                    -7878656,
                    -7877888,
                    -7877376,
                    -7876096,
                    -7875072,
                    -7874048,
                    -7873024,
                    -7872000,
                    -7870720,
                    -7869440,
                    -7868672,
                    -7867392,
                    -7866112,
                    -7864832,
                    -7864575,
                    -7864571,
                    -7864566,
                    -7864561,
                    -7864556,
                    -7864553,
                    -7864548,
                    -7864543,
                    -7864538,
                    -7864533,
                    -7864531,
                    -7864527,
                    -7864522,
                    -7864517,
                    -7864511,
                    -7864506,
                    -7864502,
                    -7864496,
                    -7864492,
                    -7864486,
                    -7864481,
                    -7864478,
                    -7864472,
                    -7864466,
                    -7864460,
                    -7864456,
                    -7864451,
                    -7864445,
                    -7864439,
                    -7864432,
                    -7864427,
                    -7864422,
                    -7864418,
                    -7864411,
                    -7864404,
                    -7864397,
                    -7864390,
                    -7864385,
                    -7864378,
                    -7864373,
                    -7864365,
                    -7864360,
                    -7864352,
                    -7864344,
                    -7864336,
                    -7864328,
                    -7864322,
                    -7864321,
                    -7536641,
                    -7143425,
                    -6684673,
                    -6488065,
                    -6160385,
                    -5701633,
                    -5439489,
                    -5111809,
                    -4784129,
                    -4325377,
                    -4128769,
                    -3670017,
                    -3342337,
                    -2949121,
                    -2621441,
                    -2293761,
                    -1966081,
                    -1507329,
                    -1245185,
                    -983041,
                    -458753,
                    -1
                )
            )
        )
        ocs = Entry(
            "ocs", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    -16777216,
                    -16515072,
                    -16252928,
                    -15990784,
                    -15728640,
                    -15466496,
                    -15204352,
                    -14942208,
                    -14680064,
                    -14417920,
                    -14155776,
                    -13893632,
                    -13631488,
                    -13369344,
                    -13107200,
                    -12845056,
                    -12582912,
                    -12320768,
                    -12058624,
                    -11796480,
                    -11534336,
                    -11272192,
                    -11010048,
                    -10747904,
                    -10485760,
                    -10223616,
                    -9961472,
                    -9699328,
                    -9437184,
                    -9175040,
                    -8912128,
                    -8911616,
                    -8910848,
                    -8910336,
                    -8909568,
                    -8909056,
                    -8908288,
                    -8907776,
                    -8907008,
                    -8906496,
                    -8905728,
                    -8905216,
                    -8904448,
                    -8903936,
                    -8903168,
                    -8902656,
                    -8901888,
                    -8901376,
                    -8900608,
                    -8900096,
                    -8899328,
                    -8898816,
                    -8898048,
                    -8897536,
                    -8896768,
                    -8896256,
                    -8895488,
                    -8894976,
                    -8894208,
                    -8893696,
                    -8892928,
                    -8892416,
                    -8891648,
                    -8891136,
                    -8890368,
                    -8889856,
                    -8889088,
                    -8888576,
                    -8887808,
                    -8887296,
                    -8886528,
                    -8886016,
                    -8885248,
                    -8884736,
                    -8883968,
                    -8883456,
                    -8882688,
                    -8882176,
                    -8881408,
                    -8880893,
                    -8880123,
                    -8879608,
                    -8878838,
                    -8878323,
                    -8877553,
                    -8877038,
                    -8876268,
                    -8875753,
                    -8809447,
                    -8743396,
                    -8677090,
                    -8611039,
                    -8544733,
                    -8478682,
                    -8412376,
                    -8346325,
                    -8280019,
                    -8213968,
                    -8147662,
                    -8081611,
                    -8080841,
                    -8080326,
                    -8079556,
                    -8079041,
                    -8078271,
                    -8077756,
                    -8076986,
                    -8076471,
                    -8075701,
                    -8075186,
                    -8074416,
                    -8073901,
                    -8073131,
                    -8072616,
                    -8071846,
                    -8071331,
                    -8070561,
                    -8070046,
                    -8069276,
                    -8068761,
                    -8067991,
                    -8067476,
                    -8066706,
                    -8066191,
                    -8065421,
                    -8064906,
                    -8064136,
                    -8063621,
                    -8062851,
                    -8062336,
                    -8061566,
                    -8061051,
                    -8061049,
                    -8061046,
                    -8061044,
                    -8061041,
                    -8061039,
                    -8061036,
                    -8061034,
                    -8061031,
                    -8061029,
                    -8061026,
                    -8061024,
                    -8061021,
                    -8061019,
                    -8061016,
                    -8061014,
                    -8061011,
                    -8061009,
                    -8061006,
                    -8061004,
                    -8061001,
                    -8060999,
                    -8060996,
                    -8060994,
                    -8060991,
                    -8060989,
                    -8060986,
                    -8060984,
                    -8060981,
                    -8060979,
                    -8060976,
                    -8060974,
                    -8060971,
                    -8060969,
                    -8060966,
                    -8060964,
                    -8060961,
                    -8060959,
                    -8060956,
                    -8060954,
                    -8060951,
                    -8060949,
                    -8060946,
                    -8060944,
                    -8060941,
                    -8060939,
                    -8060936,
                    -8060934,
                    -8060931,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -8060929,
                    -7864321,
                    -7733249,
                    -7536641,
                    -7340033,
                    -7208961,
                    -7012353,
                    -6881281,
                    -6684673,
                    -6553601,
                    -6356993,
                    -6225921,
                    -6029313,
                    -5898241,
                    -5701633,
                    -5570561,
                    -5373953,
                    -5242881,
                    -5046273,
                    -4915201,
                    -4718593,
                    -4587521,
                    -4390913,
                    -4259841,
                    -4063233,
                    -3932161,
                    -3735553,
                    -3604481,
                    -3407873,
                    -3276801,
                    -3080193,
                    -2949121,
                    -2752513,
                    -2621441,
                    -2424833,
                    -2293761,
                    -2097153,
                    -1966081,
                    -1769473,
                    -1638401,
                    -1441793,
                    -1310721,
                    -1114113,
                    -983041,
                    -786433,
                    -655361,
                    -458753,
                    -327681,
                    -131073,
                    -1,
                    -1,
                    -1
                )
            )
        )
        lingrey = Entry(
            "lingrey", Type.SEQUENTIAL, InterpolatedPalette(
                FixedPalette(
                    -16777216,
                    -16777216,
                    -16777216,
                    -16777216,
                    -16777216,
                    -16777216,
                    -16777216,
                    -16711423,
                    -16711423,
                    -16711423,
                    -16711423,
                    -16711423,
                    -16711423,
                    -16711423,
                    -16711423,
                    -16711423,
                    -16645630,
                    -16645630,
                    -16645630,
                    -16645630,
                    -16645630,
                    -16645630,
                    -16645630,
                    -16579837,
                    -16579837,
                    -16579837,
                    -16579837,
                    -16579837,
                    -16579837,
                    -16579837,
                    -16514044,
                    -16514044,
                    -16514044,
                    -16514044,
                    -16514044,
                    -16448251,
                    -16448251,
                    -16448251,
                    -16448251,
                    -16448251,
                    -16382458,
                    -16382458,
                    -16382458,
                    -16382458,
                    -16382458,
                    -16316665,
                    -16316665,
                    -16316665,
                    -16316665,
                    -16316665,
                    -16250872,
                    -16250872,
                    -16185079,
                    -16185079,
                    -16185079,
                    -16185079,
                    -16119286,
                    -16119286,
                    -16119286,
                    -16119286,
                    -16119286,
                    -16053493,
                    -16053493,
                    -15987700,
                    -15987700,
                    -15987700,
                    -15921907,
                    -15921907,
                    -15856114,
                    -15856114,
                    -15790321,
                    -15790321,
                    -15790321,
                    -15724528,
                    -15724528,
                    -15658735,
                    -15658735,
                    -15592942,
                    -15592942,
                    -15527149,
                    -15527149,
                    -15527149,
                    -15527149,
                    -15527149,
                    -15461356,
                    -15461356,
                    -15329770,
                    -15329770,
                    -15329770,
                    -15263977,
                    -15263977,
                    -15198184,
                    -15198184,
                    -15066598,
                    -15066598,
                    -15066598,
                    -15000805,
                    -15000805,
                    -14869219,
                    -14869219,
                    -14803426,
                    -14803426,
                    -14671840,
                    -14671840,
                    -14671840,
                    -14671840,
                    -14671840,
                    -14540254,
                    -14540254,
                    -14474461,
                    -14474461,
                    -14474461,
                    -14342875,
                    -14342875,
                    -14211289,
                    -14211289,
                    -14079703,
                    -14079703,
                    -14079703,
                    -13948117,
                    -13948117,
                    -13816531,
                    -13816531,
                    -13750738,
                    -13750738,
                    -13750738,
                    -13684945,
                    -13684945,
                    -13553359,
                    -13553359,
                    -13421773,
                    -13421773,
                    -13355980,
                    -13355980,
                    -13355980,
                    -13224394,
                    -13224394,
                    -13092808,
                    -13092808,
                    -12895429,
                    -12895429,
                    -12895429,
                    -12763843,
                    -12763843,
                    -12566464,
                    -12566464,
                    -12369085,
                    -12369085,
                    -12369085,
                    -12237499,
                    -12237499,
                    -12040120,
                    -12040120,
                    -11842741,
                    -11842741,
                    -11776948,
                    -11776948,
                    -11776948,
                    -11645362,
                    -11645362,
                    -11447983,
                    -11447983,
                    -11250604,
                    -11250604,
                    -11250604,
                    -11053225,
                    -11053225,
                    -10790053,
                    -10790053,
                    -10592674,
                    -10592674,
                    -10592674,
                    -10395295,
                    -10395295,
                    -10132123,
                    -10132123,
                    -9934744,
                    -9934744,
                    -9737365,
                    -9737365,
                    -9737365,
                    -9671572,
                    -9671572,
                    -9408400,
                    -9408400,
                    -9145228,
                    -9145228,
                    -9145228,
                    -8882056,
                    -8882056,
                    -8618884,
                    -8618884,
                    -8355712,
                    -8355712,
                    -8355712,
                    -8092540,
                    -8092540,
                    -7829368,
                    -7829368,
                    -7500403,
                    -7500403,
                    -7237231,
                    -7237231,
                    -7237231,
                    -7105645,
                    -7105645,
                    -6908266,
                    -6908266,
                    -6645094,
                    -6645094,
                    -6645094,
                    -6316129,
                    -6316129,
                    -5987164,
                    -5987164,
                    -5658199,
                    -5658199,
                    -5658199,
                    -5329234,
                    -5329234,
                    -5000269,
                    -5000269,
                    -4605511,
                    -4605511,
                    -4276546,
                    -4276546,
                    -4276546,
                    -3947581,
                    -3947581,
                    -3947581,
                    -3947581,
                    -3552823,
                    -3552823,
                    -3552823,
                    -3158065,
                    -3158065,
                    -2829100,
                    -2829100,
                    -2434342,
                    -2434342,
                    -2434342,
                    -2039584,
                    -2039584,
                    -1644826,
                    -1644826,
                    -1184275,
                    -1184275,
                    -789517,
                    -789517,
                    -789517,
                    -394759,
                    -394759,
                    -197380,
                    -197380,
                    -197380,
                    -1
                )
            )
        )
        YlOrRd = Entry(
            "YlOrRd",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -52,
                    -78,
                    -4704,
                    -75402,
                    -78756,
                    -85428,
                    -160452,
                    -242134,
                    -1033440,
                    -1893860,
                    -4390874,
                    -5177306,
                    -8388570
                )
            )
        )
        PRGn = Entry(
            "PRGn",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    -12582837,
                    -9033085,
                    -8703340,
                    -6721365,
                    -5272125,
                    -4020785,
                    -1583896,
                    -526345,
                    -2494253,
                    -5842016,
                    -8405125,
                    -10834335,
                    -16742345,
                    -14976969,
                    -16759781
                )
            )
        )
        PuOr = Entry(
            "PuOr",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    -8439032,
                    -5023738,
                    -1679103,
                    -2063852,
                    -941248,
                    -149405,
                    -73546,
                    -526345,
                    -2565397,
                    -5067822,
                    -6713661,
                    -8358996,
                    -10601319,
                    -11262072,
                    -13828021
                )
            )
        )
        RdGy = Entry(
            "RdGy",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    -10026977,
                    -5105621,
                    -3538912,
                    -2727859,
                    -1078686,
                    -744062,
                    -140345,
                    -1,
                    -2039584,
                    -4539718,
                    -6710887,
                    -7895161,
                    -12566464,
                    -11711155,
                    -15066598
                )
            )
        )
        Spectral = Entry(
            "Spectral",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    -6422206,
                    -2802097,
                    -2680548,
                    -758461,
                    -225959,
                    -151967,
                    -73589,
                    -65,
                    -1641064,
                    -5513820,
                    -6695532,
                    -10042715,
                    -13925446,
                    -13465411,
                    -10596446
                )
            )
        )
        Grays = Entry(
            "Grays",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -1,
                    -526345,
                    -986896,
                    -2500135,
                    -3355444,
                    -4342339,
                    -6908266,
                    -9211021,
                    -10263709,
                    -11382190,
                    -14342875,
                    -14342875,
                    -16777216
                )
            )
        )
        PuBuGn = Entry(
            "PuBuGn",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -2053,
                    -593929,
                    -1252624,
                    -3091994,
                    -4339231,
                    -5849637,
                    -9131569,
                    -13201216,
                    -14905191,
                    -16612982,
                    -16683943,
                    -16686000,
                    -16693706
                )
            )
        )
        RdPu = Entry(
            "RdPu",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -2061,
                    -70686,
                    -139043,
                    -211520,
                    -281415,
                    -352331,
                    -563039,
                    -2280297,
                    -3859574,
                    -5373570,
                    -8781449,
                    -8781449,
                    -11992982
                )
            )
        )
        BuPu = Entry(
            "BuPu",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -525059,
                    -1181445,
                    -2036492,
                    -4205594,
                    -4993565,
                    -6374182,
                    -7563578,
                    -7574607,
                    -7842137,
                    -7847523,
                    -8319108,
                    -9567893,
                    -11730869
                )
            )
        )
        YlOrBr = Entry(
            "YlOrBr",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -27,
                    -44,
                    -2116,
                    -72815,
                    -75378,
                    -80817,
                    -91863,
                    -1282028,
                    -2531570,
                    -3388414,
                    -6736892,
                    -7590652,
                    -10083066
                )
            )
        )
        Greens = Entry(
            "Greens",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -525067,
                    -1181463,
                    -1706528,
                    -3675712,
                    -4528973,
                    -6170213,
                    -9124746,
                    -12473507,
                    -13524140,
                    -14447803,
                    -16749268,
                    -16754126,
                    -16759781
                )
            )
        )
        BuGn = Entry(
            "BuGn",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -525059,
                    -1181445,
                    -1706503,
                    -3347226,
                    -5053726,
                    -6694711,
                    -10042716,
                    -12472714,
                    -13852065,
                    -14447803,
                    -16749268,
                    -16754652,
                    -16759781
                )
            )
        )
        Accents = Entry(
            "Accents",
            Type.QUALITATIVE,
            FixedPalette(-8402561, -4280620, -147322, -103, -13079376, -1047937, -4236521, -10066330)
        )
        GnBu = Entry(
            "GnBu",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -525072,
                    -984600,
                    -2034725,
                    -3347515,
                    -4528964,
                    -5710411,
                    -8663868,
                    -11619373,
                    -12344630,
                    -13923138,
                    -16226132,
                    -16230242,
                    -16236415
                )
            )
        )
        PuRd = Entry(
            "PuRd",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -527111,
                    -921866,
                    -1580561,
                    -2836006,
                    -2640424,
                    -3566393,
                    -2136656,
                    -1627766,
                    -2286473,
                    -3272106,
                    -6815677,
                    -7274433,
                    -10026977
                )
            )
        )
        Purples = Entry(
            "Purples",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -197635,
                    -855817,
                    -1053195,
                    -2434325,
                    -3421726,
                    -4407844,
                    -6382904,
                    -8356422,
                    -9081935,
                    -9809501,
                    -11262065,
                    -11922298,
                    -12648323
                )
            )
        )
        RdYlGn = Entry(
            "RdYlGn",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    -5963738,
                    -2674649,
                    -2680548,
                    -758461,
                    -225959,
                    -151967,
                    -73589,
                    -65,
                    -2494581,
                    -5842582,
                    -7221408,
                    -10044061,
                    -15034815,
                    -15034288,
                    -16750537
                )
            )
        )
        Paired = Entry(
            "Paired",
            Type.QUALITATIVE,
            FixedPalette(
                -5845277,
                -14714700,
                -5054582,
                -13393876,
                -288103,
                -1893860,
                -147601,
                -33024,
                -3493162,
                -9814630,
                -103,
                -5154520
            )
        )
        Blues = Entry(
            "Blues",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -525313,
                    -1051649,
                    -2167817,
                    -3744785,
                    -4335641,
                    -6370591,
                    -9720106,
                    -12414266,
                    -13532483,
                    -14585419,
                    -16232036,
                    -16235116,
                    -16240533
                )
            )
        )
        RdBu = Entry(
            "RdBu",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    -10026977,
                    -5105621,
                    -3538912,
                    -2727859,
                    -1078686,
                    -744062,
                    -140345,
                    -526345,
                    -3021328,
                    -7158306,
                    -9983537,
                    -12348477,
                    -16420432,
                    -14588244,
                    -16437151
                )
            )
        )
        Oranges = Entry(
            "Oranges",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -2581,
                    -70178,
                    -71986,
                    -143198,
                    -147835,
                    -151957,
                    -160452,
                    -956141,
                    -1682163,
                    -2537471,
                    -5884413,
                    -7590652,
                    -8444156
                )
            )
        )
        RdYlBu = Entry(
            "RdYlBu",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    -5963738,
                    -2674649,
                    -2680548,
                    -758461,
                    -225959,
                    -151967,
                    -73584,
                    -65,
                    -2034696,
                    -5514775,
                    -7225381,
                    -9130543,
                    -13861962,
                    -12225100,
                    -13551979
                )
            )
        )
        PuBu = Entry(
            "PuBu",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -2053,
                    -921866,
                    -1251342,
                    -3091994,
                    -4339231,
                    -5849637,
                    -9983537,
                    -13201216,
                    -13923138,
                    -16420688,
                    -16491891,
                    -16560517,
                    -16631720
                )
            )
        )
        OrRd = Entry(
            "OrRd",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -2068,
                    -69415,
                    -71480,
                    -142178,
                    -144246,
                    -148604,
                    -225959,
                    -1088184,
                    -1881549,
                    -2674657,
                    -5046272,
                    -6750208,
                    -8454144
                )
            )
        )
        Set3 = Entry(
            "Set3",
            Type.QUALITATIVE,
            FixedPalette(
                -7482425,
                -77,
                -4277542,
                -294798,
                -8343085,
                -150430,
                -4989335,
                -209435,
                -2500135,
                -4423491,
                -3347515,
                -4753
            )
        )
        Set2 = Entry(
            "Set2",
            Type.QUALITATIVE,
            FixedPalette(-10042715, -225950, -7495477, -1602877, -5842860, -9937, -1719148, -5000269)
        )
        Set1 = Entry(
            "Set1",
            Type.QUALITATIVE,
            FixedPalette(-1828324, -13140296, -11686070, -6795613, -33024, -205, -5876184, -556609, -6710887)
        )
        Reds = Entry(
            "Reds",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -2576,
                    -72231,
                    -73518,
                    -214111,
                    -217455,
                    -224654,
                    -300470,
                    -1098964,
                    -2216666,
                    -3467235,
                    -5959915,
                    -6750195,
                    -10026995
                )
            )
        )
        PiYG = Entry(
            "PiYG",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    -7470766,
                    -3859587,
                    -3138421,
                    -2197586,
                    -1465399,
                    -936230,
                    -139025,
                    -526345,
                    -1641008,
                    -4660858,
                    -6170774,
                    -8405951,
                    -11686874,
                    -11693535,
                    -14195687
                )
            )
        )
        Dark2 = Entry(
            "Dark2",
            Type.QUALITATIVE,
            FixedPalette(-14967177, -2531582, -9080653, -1627766, -10050018, -1660158, -5868003, -10066330)
        )
        YlGn = Entry(
            "YlGn",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -27,
                    -52,
                    -525127,
                    -2494301,
                    -4004199,
                    -5382770,
                    -8862087,
                    -12473507,
                    -13524140,
                    -14449597,
                    -16750537,
                    -16754126,
                    -16759511
                )
            )
        )
        BrBG = Entry(
            "BrBG",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    -11259899,
                    -7581430,
                    -5873382,
                    -4226771,
                    -2575515,
                    -2112899,
                    -595773,
                    -657931,
                    -3675419,
                    -8335935,
                    -10832724,
                    -13265009,
                    -16677519,
                    -16685474,
                    -16761808
                )
            )
        )
        YlGnBu = Entry(
            "YlGnBu",
            Type.SEQUENTIAL,
            InterpolatedPalette(
                FixedPalette(
                    -39,
                    -52,
                    -1181519,
                    -3675724,
                    -6169932,
                    -8401477,
                    -12470615,
                    -14839360,
                    -13860936,
                    -14524248,
                    -14338924,
                    -15979388,
                    -16245416
                )
            )
        )
        Pastel2 = Entry(
            "Pastel2",
            Type.QUALITATIVE,
            FixedPalette(-4988211, -143956, -3418648, -734492, -1641015, -3410, -924980, -3355444)
        )
        Pastel1 = Entry(
            "Pastel1",
            Type.QUALITATIVE,
            FixedPalette(-281426, -4993565, -3347515, -2176028, -75354, -52, -1713987, -140564, -855310)
        )
        IsoL = Entry(
            "IsoL",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    CPColor(0.9102f, 0.2236f, 0.8997f),
                    CPColor(0.4027f, 0.3711f, 1f),
                    CPColor(0.0422f, 0.5904f, 0.5899f),
                    CPColor(0.0386f, 0.6206f, 0.0201f),
                    CPColor(0.5441f, 0.5428f, 0.011f),
                    CPColor(1f, 0.2288f, 0.1631f)
                )
            )
        )
        CubicL = Entry(
            "CubicL",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    CPColor(0.4706f, 0f, 0.5216f),
                    CPColor(0.5137f, 0.0527f, 0.7096f),
                    CPColor(0.4942f, 0.2507f, 0.8781f),
                    CPColor(0.4296f, 0.3858f, 0.9922f),
                    CPColor(0.3691f, 0.5172f, 0.9495f),
                    CPColor(0.2963f, 0.6191f, 0.8515f),
                    CPColor(0.2199f, 0.7134f, 0.7225f),
                    CPColor(0.2643f, 0.7836f, 0.5756f),
                    CPColor(0.3094f, 0.8388f, 0.4248f),
                    CPColor(0.3623f, 0.8917f, 0.2858f),
                    CPColor(0.52f, 0.921f, 0.3137f),
                    CPColor(0.68f, 0.9255f, 0.3386f),
                    CPColor(0.8f, 0.9255f, 0.3529f),
                    CPColor(0.8706f, 0.8549f, 0.3608f),
                    CPColor(0.9514f, 0.7466f, 0.3686f),
                    CPColor(0.9765f, 0.5887f, 0.3569f)
                )
            )
        )
        Edge = Entry(
            "Edge",
            Type.DIVERGING,
            InterpolatedPalette(
                FixedPalette(
                    CPColor(0f, 0f, 0f),
                    CPColor(0f, 0f, 1f),
                    CPColor(0f, 1f, 1f),
                    CPColor(1f, 1f, 1f),
                    CPColor(1f, 1f, 0f),
                    CPColor(1f, 0f, 0f),
                    CPColor(0f, 0f, 0f)
                )
            )
        )
        coolwarm = Entry(
            "coolwarm", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    CPColor(0.2298057f, 0.298717966f, 0.753683153f),
                    CPColor(0.23437707945098038f, 0.3055417303294118f, 0.7596795275882353f),
                    CPColor(0.2389484589019608f, 0.3123654946588235f, 0.7656759021764705f),
                    CPColor(0.24351983835294116f, 0.3191892589882353f, 0.7716722767647058f),
                    CPColor(0.24809121780392157f, 0.3260130233176471f, 0.7776686513529412f),
                    CPColor(0.2526625972549019f, 0.3328367876470588f, 0.7836650259411765f),
                    CPColor(0.25723397670588233f, 0.3396605519764706f, 0.7896614005294117f),
                    CPColor(0.26180535615686273f, 0.3464843163058824f, 0.795657775117647f),
                    CPColor(0.26638146835294113f, 0.35330440842352945f, 0.8016373194980392f),
                    CPColor(0.2711042956470588f, 0.3600106619764706f, 0.8070951274352941f),
                    CPColor(0.27582712294117645f, 0.36671691552941177f, 0.812552935372549f),
                    CPColor(0.2805499502352941f, 0.37342316908235296f, 0.818010743309804f),
                    CPColor(0.28527277752941177f, 0.38012942263529415f, 0.8234685512470589f),
                    CPColor(0.2899956048235294f, 0.3868356761882353f, 0.8289263591843137f),
                    CPColor(0.29471843211764703f, 0.39354192974117647f, 0.8343841671215686f),
                    CPColor(0.2994412594117647f, 0.40024818329411765f, 0.8398419750588235f),
                    CPColor(0.3041742870039216f, 0.40694488283921565f, 0.8452627266980393f),
                    CPColor(0.30906031906666664f, 0.41349827226666663f, 0.8501276338666667f),
                    CPColor(0.31394635112941177f, 0.4200516616941176f, 0.8549925410352941f),
                    CPColor(0.31883238319215684f, 0.4266050511215686f, 0.8598574482039216f),
                    CPColor(0.32371841525490197f, 0.4331584405490196f, 0.864722355372549f),
                    CPColor(0.32860444731764704f, 0.43971182997647057f, 0.8695872625411765f),
                    CPColor(0.33349047938039217f, 0.44626521940392155f, 0.8744521697098039f),
                    CPColor(0.3383765114431373f, 0.45281860883137254f, 0.8793170768784313f),
                    CPColor(0.34327752343529416f, 0.45935363472941176f, 0.8841219216235294f),
                    CPColor(0.34832334141176474f, 0.4657111465098039f, 0.8883461629411764f),
                    CPColor(0.3533691593882353f, 0.47206865829019606f, 0.8925704042588235f),
                    CPColor(0.3584149773647059f, 0.47842617007058824f, 0.8967946455764706f),
                    CPColor(0.3634607953411765f, 0.4847836818509804f, 0.9010188868941177f),
                    CPColor(0.36850661331764706f, 0.49114119363137254f, 0.9052431282117647f),
                    CPColor(0.37355243129411764f, 0.4974987054117647f, 0.9094673695294118f),
                    CPColor(0.3785982492705882f, 0.5038562171921569f, 0.9136916108470589f),
                    CPColor(0.383662065772549f, 0.5101834172862746f, 0.9178306732313726f),
                    CPColor(0.38885187195294113f, 0.5162984355764706f, 0.9213734830823529f),
                    CPColor(0.3940416781333333f, 0.5224134538666667f, 0.9249162929333333f),
                    CPColor(0.39923148431372546f, 0.5285284721568628f, 0.9284591027843138f),
                    CPColor(0.4044212904941176f, 0.5346434904470588f, 0.9320019126352941f),
                    CPColor(0.4096110966745098f, 0.540758508737255f, 0.9355447224862745f),
                    CPColor(0.41480090285490195f, 0.5468735270274511f, 0.939087532337255f),
                    CPColor(0.4199907090352941f, 0.5529885453176471f, 0.9426303421882353f),
                    CPColor(0.42519897019607844f, 0.559058179764706f, 0.9460614570784314f),
                    CPColor(0.4305068882509804f, 0.5648827414588236f, 0.9488894191803922f),
                    CPColor(0.4358148063058824f, 0.5707073031529412f, 0.951717381282353f),
                    CPColor(0.4411227243607843f, 0.5765318648470589f, 0.9545453433843137f),
                    CPColor(0.4464306424156863f, 0.5823564265411765f, 0.9573733054862745f),
                    CPColor(0.4517385604705882f, 0.5881809882352941f, 0.9602012675882352f),
                    CPColor(0.4570464785254902f, 0.5940055499294118f, 0.963029229690196f),
                    CPColor(0.4623543965803922f, 0.5998301116235294f, 0.9658571917921568f),
                    CPColor(0.46767809468235294f, 0.6055912316235293f, 0.9685462810941176f),
                    CPColor(0.4730701729882353f, 0.6110774376156862f, 0.970633588262745f),
                    CPColor(0.4784622512941176f, 0.6165636436078431f, 0.9727208954313725f),
                    CPColor(0.48385432959999997f, 0.6220498496f, 0.9748082026f),
                    CPColor(0.48924640790588236f, 0.6275360555921569f, 0.9768955097686275f),
                    CPColor(0.49463848621176465f, 0.6330222615843136f, 0.9789828169372549f),
                    CPColor(0.500030564517647f, 0.6385084675764706f, 0.9810701241058823f),
                    CPColor(0.5054226428235293f, 0.6439946735686275f, 0.9831574312745098f),
                    CPColor(0.5108243242509803f, 0.6493966148235294f, 0.9850787763764707f),
                    CPColor(0.5162603025411764f, 0.6544976105882353f, 0.9864073998117647f),
                    CPColor(0.5216962808313725f, 0.6595986063529412f, 0.9877360232470589f),
                    CPColor(0.5271322591215686f, 0.664699602117647f, 0.9890646466823529f),
                    CPColor(0.5325682374117646f, 0.6698005978823529f, 0.9903932701176471f),
                    CPColor(0.5380042157019607f, 0.6749015936470587f, 0.9917218935529412f),
                    CPColor(0.5434401939921568f, 0.6800025894117647f, 0.9930505169882353f),
                    CPColor(0.5488761722823529f, 0.6851035851764705f, 0.9943791404235295f),
                    CPColor(0.5543118699137254f, 0.6900970112156862f, 0.9955155482352941f),
                    CPColor(0.5597467255686274f, 0.6947677280784313f, 0.9960753091764706f),
                    CPColor(0.5651815812235294f, 0.6994384449411764f, 0.9966350701176471f),
                    CPColor(0.5706164368784313f, 0.7041091618039216f, 0.9971948310588236f),
                    CPColor(0.5760512925333333f, 0.7087798786666667f, 0.997754592f),
                    CPColor(0.5814861481882353f, 0.7134505955294117f, 0.9983143529411764f),
                    CPColor(0.5869210038431373f, 0.7181213123921568f, 0.9988741138823529f),
                    CPColor(0.5923558594980393f, 0.7227920292549019f, 0.9994338748235294f),
                    CPColor(0.5977767754941177f, 0.7273297248823529f, 0.9997767317764705f),
                    CPColor(0.603162067917647f, 0.7315274773529412f, 0.9995652785372549f),
                    CPColor(0.6085473603411764f, 0.7357252298235294f, 0.9993538252980392f),
                    CPColor(0.6139326527647059f, 0.7399229822941177f, 0.9991423720588235f),
                    CPColor(0.6193179451882354f, 0.7441207347647059f, 0.9989309188196078f),
                    CPColor(0.6247032376117647f, 0.7483184872352941f, 0.9987194655803922f),
                    CPColor(0.6300885300352941f, 0.7525162397058823f, 0.9985080123411765f),
                    CPColor(0.6354738224588236f, 0.7567139921764706f, 0.9982965591019608f),
                    CPColor(0.640827782372549f, 0.7607515064117647f, 0.9978457748823529f),
                    CPColor(0.6461128107647058f, 0.7644364965294117f, 0.9968684625058823f),
                    CPColor(0.6513978391568627f, 0.7681214866470587f, 0.9958911501294118f),
                    CPColor(0.6566828675490196f, 0.7718064767647058f, 0.9949138377529412f),
                    CPColor(0.6619678959411764f, 0.7754914668823529f, 0.9939365253764706f),
                    CPColor(0.6672529243333334f, 0.7791764569999999f, 0.992959213f),
                    CPColor(0.6725379527254902f, 0.782861447117647f, 0.9919819006235294f),
                    CPColor(0.677822981117647f, 0.786546437235294f, 0.9910045882470588f),
                    CPColor(0.6830556815607843f, 0.790042626890196f, 0.9897684281843138f),
                    CPColor(0.6881884831921569f, 0.7931783792980391f, 0.9880381043568628f),
                    CPColor(0.6933212848235294f, 0.7963141317058823f, 0.9863077805294118f),
                    CPColor(0.6984540864549019f, 0.7994498841137254f, 0.9845774567019608f),
                    CPColor(0.7035868880862746f, 0.8025856365215686f, 0.9828471328745098f),
                    CPColor(0.7087196897176471f, 0.8057213889294117f, 0.9811168090470588f),
                    CPColor(0.7138524913490196f, 0.8088571413372548f, 0.9793864852196078f),
                    CPColor(0.7189852929803922f, 0.811992893745098f, 0.9776561613921568f),
                    CPColor(0.724041371882353f, 0.8149103926470588f, 0.9756509706470589f),
                    CPColor(0.7289695795686274f, 0.8174641357058824f, 0.973187668372549f),
                    CPColor(0.7338977872549018f, 0.8200178787647059f, 0.9707243660980392f),
                    CPColor(0.7388259949411764f, 0.8225716218235294f, 0.9682610638235294f),
                    CPColor(0.743754202627451f, 0.8251253648823529f, 0.9657977615490196f),
                    CPColor(0.7486824103137254f, 0.8276791079411765f, 0.9633344592745098f),
                    CPColor(0.753610618f, 0.830232851f, 0.960871157f),
                    CPColor(0.7585388256862745f, 0.8327865940588235f, 0.9584078547254902f),
                    CPColor(0.7633627801019607f, 0.8350922218196078f, 0.9556576765568627f),
                    CPColor(0.7680343643529411f, 0.8370352195294117f, 0.9524882182352941f),
                    CPColor(0.7727059486039215f, 0.8389782172392156f, 0.9493187599137255f),
                    CPColor(0.777377532854902f, 0.8409212149490196f, 0.9461493015921568f),
                    CPColor(0.7820491171058823f, 0.8428642126588235f, 0.9429798432705883f),
                    CPColor(0.7867207013568628f, 0.8448072103686275f, 0.9398103849490196f),
                    CPColor(0.7913922856078431f, 0.8467502080784314f, 0.9366409266274509f),
                    CPColor(0.7960638698588236f, 0.8486932057882353f, 0.9334714683058823f),
                    CPColor(0.8006008472941177f, 0.8503583215607843f, 0.9300075603921568f),
                    CPColor(0.8049647588235295f, 0.8516661605568627f, 0.9261650744313725f),
                    CPColor(0.8093286703529411f, 0.8529739995529412f, 0.9223225884705882f),
                    CPColor(0.8136925818823529f, 0.8542818385490196f, 0.9184801025098039f),
                    CPColor(0.8180564934117647f, 0.8555896775450981f, 0.9146376165490196f),
                    CPColor(0.8224204049411765f, 0.8568975165411765f, 0.9107951305882354f),
                    CPColor(0.8267843164705883f, 0.8582053555372549f, 0.906952644627451f),
                    CPColor(0.831148228f, 0.8595131945333333f, 0.9031101586666667f),
                    CPColor(0.8353447113529412f, 0.8605139972941176f, 0.8989704099411765f),
                    CPColor(0.839351442772549f, 0.861166825654902f, 0.8944937634156863f),
                    CPColor(0.8433581741921568f, 0.8618196540156863f, 0.8900171168901961f),
                    CPColor(0.8473649056117647f, 0.8624724823764706f, 0.885540470364706f),
                    CPColor(0.8513716370313725f, 0.8631253107372548f, 0.8810638238392158f),
                    CPColor(0.8553783684509804f, 0.8637781390980391f, 0.8765871773137255f),
                    CPColor(0.8593850998705882f, 0.8644309674588235f, 0.8721105307882353f),
                    CPColor(0.863391831290196f, 0.8650837958196078f, 0.8676338842627451f),
                    CPColor(0.8674276350862745f, 0.864376599772549f, 0.8626024620196079f),
                    CPColor(0.8714925112588235f, 0.8623093793176471f, 0.8570162640588236f),
                    CPColor(0.8755573874313726f, 0.860242158862745f, 0.8514300660980393f),
                    CPColor(0.8796222636039216f, 0.8581749384078431f, 0.845843868137255f),
                    CPColor(0.8836871397764705f, 0.8561077179529412f, 0.8402576701764708f),
                    CPColor(0.8877520159490196f, 0.8540404974980391f, 0.8346714722156863f),
                    CPColor(0.8918168921215687f, 0.8519732770431372f, 0.829085274254902f),
                    CPColor(0.8958817682941177f, 0.8499060565882353f, 0.8234990762941177f),
                    CPColor(0.8995432066000001f, 0.8475002359999999f, 0.8177890744f),
                    CPColor(0.9028486703176472f, 0.8447956505882352f, 0.8119698337411765f),
                    CPColor(0.9061541340352941f, 0.8420910651764706f, 0.8061505930823529f),
                    CPColor(0.9094595977529412f, 0.8393864797647058f, 0.8003313524235294f),
                    CPColor(0.9127650614705882f, 0.8366818943529412f, 0.7945121117647058f),
                    CPColor(0.9160705251882353f, 0.8339773089411765f, 0.7886928711058824f),
                    CPColor(0.9193759889058823f, 0.8312727235294118f, 0.7828736304470588f),
                    CPColor(0.9226814526235294f, 0.8285681381176471f, 0.7770543897882353f),
                    CPColor(0.925563423f, 0.8255172980705883f, 0.7711363078117647f),
                    CPColor(0.9281160096666666f, 0.8221971488627451f, 0.765141349254902f),
                    CPColor(0.9306685963333333f, 0.818876999654902f, 0.7591463906980392f),
                    CPColor(0.933221183f, 0.8155568504470588f, 0.7531514321411764f),
                    CPColor(0.9357737696666666f, 0.8122367012392158f, 0.7471564735843139f),
                    CPColor(0.9383263563333333f, 0.8089165520313726f, 0.741161515027451f),
                    CPColor(0.940878943f, 0.8055964028235294f, 0.7351665564705883f),
                    CPColor(0.9434315296666667f, 0.8022762536156862f, 0.7291715979137255f),
                    CPColor(0.945540298909804f, 0.7986057405333333f, 0.7231054172980392f),
                    CPColor(0.9473454036f, 0.7946955048f, 0.7169905058f),
                    CPColor(0.9491505082901961f, 0.7907852690666667f, 0.7108755943019608f),
                    CPColor(0.9509556129803922f, 0.7868750333333333f, 0.7047606828039216f),
                    CPColor(0.9527607176705882f, 0.7829647976f, 0.6986457713058823f),
                    CPColor(0.9545658223607844f, 0.7790545618666667f, 0.6925308598078431f),
                    CPColor(0.9563709270509804f, 0.7751443261333334f, 0.6864159483098039f),
                    CPColor(0.9581760317411765f, 0.7712340904f, 0.6803010368117647f),
                    CPColor(0.9595176584705882f, 0.7669728545098039f, 0.6741447150392157f),
                    CPColor(0.9605811984235294f, 0.7625010185254902f, 0.6679635471019607f),
                    CPColor(0.9616447383764706f, 0.7580291825411765f, 0.6617823791647058f),
                    CPColor(0.9627082783294117f, 0.7535573465568628f, 0.655601211227451f),
                    CPColor(0.9637718182823529f, 0.7490855105725491f, 0.6494200432901962f),
                    CPColor(0.9648353582352941f, 0.7446136745882352f, 0.6432388753529412f),
                    CPColor(0.9658988981882353f, 0.7401418386039216f, 0.6370577074156862f),
                    CPColor(0.9669624381411764f, 0.7356700026196078f, 0.6308765394784314f),
                    CPColor(0.9675442976352941f, 0.7308497161882352f, 0.6246854782352941f),
                    CPColor(0.9678738483176471f, 0.7258469080941177f, 0.6184892347843137f),
                    CPColor(0.968203399f, 0.7208441f, 0.6122929913333334f),
                    CPColor(0.9685329496823529f, 0.7158412919058823f, 0.6060967478823529f),
                    CPColor(0.9688625003647059f, 0.7108384838117647f, 0.5999005044313725f),
                    CPColor(0.9691920510470589f, 0.705835675717647f, 0.5937042609803921f),
                    CPColor(0.9695216017294117f, 0.7008328676235294f, 0.5875080175294117f),
                    CPColor(0.9698511524117647f, 0.6958300595294117f, 0.5813117740784314f),
                    CPColor(0.9696829796666666f, 0.6904839307372549f, 0.5751383613647059f),
                    CPColor(0.9692885689999999f, 0.6849817470823529f, 0.5689753262588235f),
                    CPColor(0.9688941583333334f, 0.679479563427451f, 0.5628122911529412f),
                    CPColor(0.9684997476666667f, 0.673977379772549f, 0.5566492560470588f),
                    CPColor(0.968105337f, 0.6684751961176472f, 0.5504862209411766f),
                    CPColor(0.9677109263333333f, 0.6629730124627451f, 0.5443231858352942f),
                    CPColor(0.9673165156666667f, 0.6574708288078431f, 0.5381601507294118f),
                    CPColor(0.966922105f, 0.6519686451529412f, 0.5319971156235295f),
                    CPColor(0.9660167198392157f, 0.6461297415882352f, 0.5258903482588235f),
                    CPColor(0.9649113881372549f, 0.6401590780588234f, 0.5198055987058824f),
                    CPColor(0.963806056435294f, 0.6341884145294118f, 0.5137208491529413f),
                    CPColor(0.9627007247333333f, 0.628217751f, 0.5076360996f),
                    CPColor(0.9615953930313725f, 0.6222470874705882f, 0.5015513500470589f),
                    CPColor(0.9604900613294117f, 0.6162764239411764f, 0.49546660049411767f),
                    CPColor(0.9593847296274509f, 0.6103057604117648f, 0.4893818509411765f),
                    CPColor(0.9582793979254902f, 0.604335096882353f, 0.48329710138823534f),
                    CPColor(0.9566532109764706f, 0.598033822717647f, 0.4773022923529412f),
                    CPColor(0.9548534056117647f, 0.5916223450078432f, 0.4713374634901961f),
                    CPColor(0.9530536002470588f, 0.5852108672980392f, 0.465372634627451f),
                    CPColor(0.951253794882353f, 0.5787993895882353f, 0.4594078057647059f),
                    CPColor(0.9494539895176471f, 0.5723879118784315f, 0.453442976901961f),
                    CPColor(0.9476541841529411f, 0.5659764341686274f, 0.4474781480392157f),
                    CPColor(0.9458543787882353f, 0.5595649564588235f, 0.44151331917647063f),
                    CPColor(0.9440545734235294f, 0.5531534787490197f, 0.4355484903137255f),
                    CPColor(0.9417279298235294f, 0.5464134770196079f, 0.429707070372549f),
                    CPColor(0.9392537715176471f, 0.5395814885647059f, 0.4239002049294118f),
                    CPColor(0.9367796132117647f, 0.5327495001098039f, 0.41809333948627453f),
                    CPColor(0.9343054549058823f, 0.525917511654902f, 0.4122864740431373f),
                    CPColor(0.9318312966f, 0.5190855232f, 0.4064796086f),
                    CPColor(0.9293571382941177f, 0.512253534745098f, 0.40067274315686274f),
                    CPColor(0.9268829799882353f, 0.5054215462901961f, 0.3948658777137255f),
                    CPColor(0.9244088216823529f, 0.49858955783529413f, 0.38905901227058826f),
                    CPColor(0.921406221227451f, 0.49142041718431373f, 0.38340843537647057f),
                    CPColor(0.9182816725843137f, 0.48417347218039214f, 0.37779392507058823f),
                    CPColor(0.9151571239411764f, 0.4769265271764706f, 0.3721794147647059f),
                    CPColor(0.9120325752980393f, 0.469679582172549f, 0.36656490445882356f),
                    CPColor(0.908908026654902f, 0.46243263716862765f, 0.36095039415294133f),
                    CPColor(0.9057834780117647f, 0.4551856921647059f, 0.35533588384705883f),
                    CPColor(0.9026589293686275f, 0.4479387471607843f, 0.3497213735411765f),
                    CPColor(0.8995343807254902f, 0.4406918021568627f, 0.34410686323529416f),
                    CPColor(0.8958845948352941f, 0.43307455670588235f, 0.3386806345176471f),
                    CPColor(0.8921375427882353f, 0.4253887370980392f, 0.33328927276078435f),
                    CPColor(0.8883904907411765f, 0.41770291749019606f, 0.32789791100392157f),
                    CPColor(0.8846434386941177f, 0.41001709788235297f, 0.32250654924705885f),
                    CPColor(0.8808963866470588f, 0.4023312782745098f, 0.3171151874901961f),
                    CPColor(0.8771493346f, 0.39464545866666667f, 0.31172382573333335f),
                    CPColor(0.8734022825529412f, 0.3869596390588235f, 0.3063324639764706f),
                    CPColor(0.8696552305058823f, 0.37927381945098043f, 0.30094110221960785f),
                    CPColor(0.8653913329372549f, 0.3711276720470588f, 0.2957689564156863f),
                    CPColor(0.8610536002941176f, 0.3629157635294118f, 0.2906281271764706f),
                    CPColor(0.8567158676509804f, 0.3547038550117647f, 0.2854872979372549f),
                    CPColor(0.8523781350078431f, 0.34649194649411763f, 0.2803464686980392f),
                    CPColor(0.848040402364706f, 0.3382800379764708f, 0.2752056394588237f),
                    CPColor(0.8437026697215686f, 0.3300681294588235f, 0.2700648102196078f),
                    CPColor(0.8393649370784314f, 0.32185622094117644f, 0.26492398098039216f),
                    CPColor(0.8350272044352941f, 0.3136443124235294f, 0.25978315174117644f),
                    CPColor(0.8301865219490197f, 0.30473276355294115f, 0.25489142806666665f),
                    CPColor(0.8252938101686275f, 0.2957488380941176f, 0.2500254739333333f),
                    CPColor(0.8204010983882353f, 0.2867649126352941f, 0.2451595198f),
                    CPColor(0.8155083866078432f, 0.2777809871764706f, 0.24029356566666665f),
                    CPColor(0.810615674827451f, 0.26879706171764706f, 0.23542761153333333f),
                    CPColor(0.8057229630470588f, 0.2598131362588235f, 0.2305616574f),
                    CPColor(0.8008302512666666f, 0.2508292108f, 0.22569570326666666f),
                    CPColor(0.7959375394862745f, 0.24184528534117647f, 0.22082974913333334f),
                    CPColor(0.7905615319411765f, 0.23139699905882352f, 0.21624203829411764f),
                    CPColor(0.7851533046784314f, 0.2208510887215686f, 0.21167287700784312f),
                    CPColor(0.7797450774156863f, 0.21030517838431373f, 0.20710371572156863f),
                    CPColor(0.7743368501529412f, 0.19975926804705882f, 0.2025345544352941f),
                    CPColor(0.7689286228901963f, 0.18921335770980421f, 0.19796539314901973f),
                    CPColor(0.763520395627451f, 0.17866744737254903f, 0.1933962318627451f),
                    CPColor(0.758112168364706f, 0.1681215370352941f, 0.18882707057647058f),
                    CPColor(0.7527039411019608f, 0.1575756266980392f, 0.1842579092901961f),
                    CPColor(0.7468380122117647f, 0.14002101948235293f, 0.17999609695686275f),
                    CPColor(0.7409573187529412f, 0.12224032527058823f, 0.17574419910588235f),
                    CPColor(0.7350766252941177f, 0.10445963105882351f, 0.17149230125490195f),
                    CPColor(0.7291959318352941f, 0.08667893684705881f, 0.16724040340392157f),
                    CPColor(0.7233152383764706f, 0.06889824263529411f, 0.16298850555294117f),
                    CPColor(0.717434544917647f, 0.05111754842352939f, 0.15873660770196077f),
                    CPColor(0.7115538514588235f, 0.03333685421176469f, 0.1544847098509804f),
                    CPColor(0.705673158f, 0.01555616f, 0.150232812f)
                )
            )
        )
        magma = Entry(
            "magma", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    CPColor(0.001462f, 0.000466f, 0.013866f),
                    CPColor(0.002258f, 0.001295f, 0.018331f),
                    CPColor(0.003279f, 0.002305f, 0.023708f),
                    CPColor(0.004512f, 0.00349f, 0.029965f),
                    CPColor(0.00595f, 0.004843f, 0.03713f),
                    CPColor(0.007588f, 0.006356f, 0.044973f),
                    CPColor(0.009426f, 0.008022f, 0.052844f),
                    CPColor(0.011465f, 0.009828f, 0.06075f),
                    CPColor(0.013708f, 0.011771f, 0.068667f),
                    CPColor(0.016156f, 0.01384f, 0.076603f),
                    CPColor(0.018815f, 0.016026f, 0.084584f),
                    CPColor(0.021692f, 0.01832f, 0.09261f),
                    CPColor(0.024792f, 0.020715f, 0.100676f),
                    CPColor(0.028123f, 0.023201f, 0.108787f),
                    CPColor(0.031696f, 0.025765f, 0.116965f),
                    CPColor(0.03552f, 0.028397f, 0.125209f),
                    CPColor(0.039608f, 0.03109f, 0.133515f),
                    CPColor(0.04383f, 0.03383f, 0.141886f),
                    CPColor(0.048062f, 0.036607f, 0.150327f),
                    CPColor(0.05232f, 0.039407f, 0.158841f),
                    CPColor(0.056615f, 0.04216f, 0.167446f),
                    CPColor(0.060949f, 0.044794f, 0.176129f),
                    CPColor(0.06533f, 0.047318f, 0.184892f),
                    CPColor(0.069764f, 0.049726f, 0.193735f),
                    CPColor(0.074257f, 0.052017f, 0.20266f),
                    CPColor(0.078815f, 0.054184f, 0.211667f),
                    CPColor(0.083446f, 0.056225f, 0.220755f),
                    CPColor(0.088155f, 0.058133f, 0.229922f),
                    CPColor(0.092949f, 0.059904f, 0.239164f),
                    CPColor(0.097833f, 0.061531f, 0.248477f),
                    CPColor(0.102815f, 0.06301f, 0.257854f),
                    CPColor(0.107899f, 0.064335f, 0.267289f),
                    CPColor(0.113094f, 0.065492f, 0.276784f),
                    CPColor(0.118405f, 0.066479f, 0.286321f),
                    CPColor(0.123833f, 0.067295f, 0.295879f),
                    CPColor(0.12938f, 0.067935f, 0.305443f),
                    CPColor(0.135053f, 0.068391f, 0.315f),
                    CPColor(0.140858f, 0.068654f, 0.324538f),
                    CPColor(0.146785f, 0.068738f, 0.334011f),
                    CPColor(0.152839f, 0.068637f, 0.343404f),
                    CPColor(0.159018f, 0.068354f, 0.352688f),
                    CPColor(0.165308f, 0.067911f, 0.361816f),
                    CPColor(0.171713f, 0.067305f, 0.370771f),
                    CPColor(0.178212f, 0.066576f, 0.379497f),
                    CPColor(0.184801f, 0.065732f, 0.387973f),
                    CPColor(0.19146f, 0.064818f, 0.396152f),
                    CPColor(0.198177f, 0.063862f, 0.404009f),
                    CPColor(0.204935f, 0.062907f, 0.411514f),
                    CPColor(0.211718f, 0.061992f, 0.418647f),
                    CPColor(0.218512f, 0.061158f, 0.425392f),
                    CPColor(0.225302f, 0.060445f, 0.431742f),
                    CPColor(0.232077f, 0.059889f, 0.437695f),
                    CPColor(0.238826f, 0.059517f, 0.443256f),
                    CPColor(0.245543f, 0.059352f, 0.448436f),
                    CPColor(0.25222f, 0.059415f, 0.453248f),
                    CPColor(0.258857f, 0.059706f, 0.45771f),
                    CPColor(0.265447f, 0.060237f, 0.46184f),
                    CPColor(0.271994f, 0.060994f, 0.46566f),
                    CPColor(0.278493f, 0.061978f, 0.46919f),
                    CPColor(0.284951f, 0.063168f, 0.472451f),
                    CPColor(0.291366f, 0.064553f, 0.475462f),
                    CPColor(0.29774f, 0.066117f, 0.478243f),
                    CPColor(0.304081f, 0.067835f, 0.480812f),
                    CPColor(0.310382f, 0.069702f, 0.483186f),
                    CPColor(0.316654f, 0.07169f, 0.48538f),
                    CPColor(0.322899f, 0.073782f, 0.487408f),
                    CPColor(0.329114f, 0.075972f, 0.489287f),
                    CPColor(0.335308f, 0.078236f, 0.491024f),
                    CPColor(0.341482f, 0.080564f, 0.492631f),
                    CPColor(0.347636f, 0.082946f, 0.494121f),
                    CPColor(0.353773f, 0.085373f, 0.495501f),
                    CPColor(0.359898f, 0.087831f, 0.496778f),
                    CPColor(0.366012f, 0.090314f, 0.49796f),
                    CPColor(0.372116f, 0.092816f, 0.499053f),
                    CPColor(0.378211f, 0.095332f, 0.500067f),
                    CPColor(0.384299f, 0.097855f, 0.501002f),
                    CPColor(0.390384f, 0.100379f, 0.501864f),
                    CPColor(0.396467f, 0.102902f, 0.502658f),
                    CPColor(0.402548f, 0.10542f, 0.503386f),
                    CPColor(0.408629f, 0.10793f, 0.504052f),
                    CPColor(0.414709f, 0.110431f, 0.504662f),
                    CPColor(0.420791f, 0.11292f, 0.505215f),
                    CPColor(0.426877f, 0.115395f, 0.505714f),
                    CPColor(0.432967f, 0.117855f, 0.50616f),
                    CPColor(0.439062f, 0.120298f, 0.506555f),
                    CPColor(0.445163f, 0.122724f, 0.506901f),
                    CPColor(0.451271f, 0.125132f, 0.507198f),
                    CPColor(0.457386f, 0.127522f, 0.507448f),
                    CPColor(0.463508f, 0.129893f, 0.507652f),
                    CPColor(0.46964f, 0.132245f, 0.507809f),
                    CPColor(0.47578f, 0.134577f, 0.507921f),
                    CPColor(0.481929f, 0.136891f, 0.507989f),
                    CPColor(0.488088f, 0.139186f, 0.508011f),
                    CPColor(0.494258f, 0.141462f, 0.507988f),
                    CPColor(0.500438f, 0.143719f, 0.50792f),
                    CPColor(0.506629f, 0.145958f, 0.507806f),
                    CPColor(0.512831f, 0.148179f, 0.507648f),
                    CPColor(0.519045f, 0.150383f, 0.507443f),
                    CPColor(0.52527f, 0.152569f, 0.507192f),
                    CPColor(0.531507f, 0.154739f, 0.506895f),
                    CPColor(0.537755f, 0.156894f, 0.506551f),
                    CPColor(0.544015f, 0.159033f, 0.506159f),
                    CPColor(0.550287f, 0.161158f, 0.505719f),
                    CPColor(0.556571f, 0.163269f, 0.50523f),
                    CPColor(0.562866f, 0.165368f, 0.504692f),
                    CPColor(0.569172f, 0.167454f, 0.504105f),
                    CPColor(0.57549f, 0.16953f, 0.503466f),
                    CPColor(0.581819f, 0.171596f, 0.502777f),
                    CPColor(0.588158f, 0.173652f, 0.502035f),
                    CPColor(0.594508f, 0.175701f, 0.501241f),
                    CPColor(0.600868f, 0.177743f, 0.500394f),
                    CPColor(0.607238f, 0.179779f, 0.499492f),
                    CPColor(0.613617f, 0.181811f, 0.498536f),
                    CPColor(0.620005f, 0.18384f, 0.497524f),
                    CPColor(0.626401f, 0.185867f, 0.496456f),
                    CPColor(0.632805f, 0.187893f, 0.495332f),
                    CPColor(0.639216f, 0.189921f, 0.49415f),
                    CPColor(0.645633f, 0.191952f, 0.49291f),
                    CPColor(0.652056f, 0.193986f, 0.491611f),
                    CPColor(0.658483f, 0.196027f, 0.490253f),
                    CPColor(0.664915f, 0.198075f, 0.488836f),
                    CPColor(0.671349f, 0.200133f, 0.487358f),
                    CPColor(0.677786f, 0.202203f, 0.485819f),
                    CPColor(0.684224f, 0.204286f, 0.484219f),
                    CPColor(0.690661f, 0.206384f, 0.482558f),
                    CPColor(0.697098f, 0.208501f, 0.480835f),
                    CPColor(0.703532f, 0.210638f, 0.479049f),
                    CPColor(0.709962f, 0.212797f, 0.477201f),
                    CPColor(0.716387f, 0.214982f, 0.47529f),
                    CPColor(0.722805f, 0.217194f, 0.473316f),
                    CPColor(0.729216f, 0.219437f, 0.471279f),
                    CPColor(0.735616f, 0.221713f, 0.46918f),
                    CPColor(0.742004f, 0.224025f, 0.467018f),
                    CPColor(0.748378f, 0.226377f, 0.464794f),
                    CPColor(0.754737f, 0.228772f, 0.462509f),
                    CPColor(0.761077f, 0.231214f, 0.460162f),
                    CPColor(0.767398f, 0.233705f, 0.457755f),
                    CPColor(0.773695f, 0.236249f, 0.455289f),
                    CPColor(0.779968f, 0.238851f, 0.452765f),
                    CPColor(0.786212f, 0.241514f, 0.450184f),
                    CPColor(0.792427f, 0.244242f, 0.447543f),
                    CPColor(0.798608f, 0.24704f, 0.444848f),
                    CPColor(0.804752f, 0.249911f, 0.442102f),
                    CPColor(0.810855f, 0.252861f, 0.439305f),
                    CPColor(0.816914f, 0.255895f, 0.436461f),
                    CPColor(0.822926f, 0.259016f, 0.433573f),
                    CPColor(0.828886f, 0.262229f, 0.430644f),
                    CPColor(0.834791f, 0.26554f, 0.427671f),
                    CPColor(0.840636f, 0.268953f, 0.424666f),
                    CPColor(0.846416f, 0.272473f, 0.421631f),
                    CPColor(0.852126f, 0.276106f, 0.418573f),
                    CPColor(0.857763f, 0.279857f, 0.415496f),
                    CPColor(0.86332f, 0.283729f, 0.412403f),
                    CPColor(0.868793f, 0.287728f, 0.409303f),
                    CPColor(0.874176f, 0.291859f, 0.406205f),
                    CPColor(0.879464f, 0.296125f, 0.403118f),
                    CPColor(0.884651f, 0.30053f, 0.400047f),
                    CPColor(0.889731f, 0.305079f, 0.397002f),
                    CPColor(0.8947f, 0.309773f, 0.393995f),
                    CPColor(0.899552f, 0.314616f, 0.391037f),
                    CPColor(0.904281f, 0.31961f, 0.388137f),
                    CPColor(0.908884f, 0.324755f, 0.385308f),
                    CPColor(0.913354f, 0.330052f, 0.382563f),
                    CPColor(0.917689f, 0.3355f, 0.379915f),
                    CPColor(0.921884f, 0.341098f, 0.377376f),
                    CPColor(0.925937f, 0.346844f, 0.374959f),
                    CPColor(0.929845f, 0.352734f, 0.372677f),
                    CPColor(0.933606f, 0.358764f, 0.370541f),
                    CPColor(0.937221f, 0.364929f, 0.368567f),
                    CPColor(0.940687f, 0.371224f, 0.366762f),
                    CPColor(0.944006f, 0.377643f, 0.365136f),
                    CPColor(0.94718f, 0.384178f, 0.363701f),
                    CPColor(0.95021f, 0.39082f, 0.362468f),
                    CPColor(0.953099f, 0.397563f, 0.361438f),
                    CPColor(0.955849f, 0.4044f, 0.360619f),
                    CPColor(0.958464f, 0.411324f, 0.360014f),
                    CPColor(0.960949f, 0.418323f, 0.35963f),
                    CPColor(0.96331f, 0.42539f, 0.359469f),
                    CPColor(0.965549f, 0.432519f, 0.359529f),
                    CPColor(0.967671f, 0.439703f, 0.35981f),
                    CPColor(0.96968f, 0.446936f, 0.360311f),
                    CPColor(0.971582f, 0.45421f, 0.36103f),
                    CPColor(0.973381f, 0.46152f, 0.361965f),
                    CPColor(0.975082f, 0.468861f, 0.363111f),
                    CPColor(0.97669f, 0.476226f, 0.364466f),
                    CPColor(0.97821f, 0.483612f, 0.366025f),
                    CPColor(0.979645f, 0.491014f, 0.367783f),
                    CPColor(0.981f, 0.498428f, 0.369734f),
                    CPColor(0.982279f, 0.505851f, 0.371874f),
                    CPColor(0.983485f, 0.51328f, 0.374198f),
                    CPColor(0.984622f, 0.520713f, 0.376698f),
                    CPColor(0.985693f, 0.528148f, 0.379371f),
                    CPColor(0.9867f, 0.535582f, 0.38221f),
                    CPColor(0.987646f, 0.543015f, 0.38521f),
                    CPColor(0.988533f, 0.550446f, 0.388365f),
                    CPColor(0.989363f, 0.557873f, 0.391671f),
                    CPColor(0.990138f, 0.565296f, 0.395122f),
                    CPColor(0.990871f, 0.572706f, 0.398714f),
                    CPColor(0.991558f, 0.580107f, 0.402441f),
                    CPColor(0.992196f, 0.587502f, 0.406299f),
                    CPColor(0.992785f, 0.594891f, 0.410283f),
                    CPColor(0.993326f, 0.602275f, 0.41439f),
                    CPColor(0.993834f, 0.609644f, 0.418613f),
                    CPColor(0.994309f, 0.616999f, 0.42295f),
                    CPColor(0.994738f, 0.62435f, 0.427397f),
                    CPColor(0.995122f, 0.631696f, 0.431951f),
                    CPColor(0.99548f, 0.639027f, 0.436607f),
                    CPColor(0.99581f, 0.646344f, 0.441361f),
                    CPColor(0.996096f, 0.653659f, 0.446213f),
                    CPColor(0.996341f, 0.660969f, 0.45116f),
                    CPColor(0.99658f, 0.668256f, 0.456192f),
                    CPColor(0.996775f, 0.675541f, 0.461314f),
                    CPColor(0.996925f, 0.682828f, 0.466526f),
                    CPColor(0.997077f, 0.690088f, 0.471811f),
                    CPColor(0.997186f, 0.697349f, 0.477182f),
                    CPColor(0.997254f, 0.704611f, 0.482635f),
                    CPColor(0.997325f, 0.711848f, 0.488154f),
                    CPColor(0.997351f, 0.719089f, 0.493755f),
                    CPColor(0.997351f, 0.726324f, 0.499428f),
                    CPColor(0.997341f, 0.733545f, 0.505167f),
                    CPColor(0.997285f, 0.740772f, 0.510983f),
                    CPColor(0.997228f, 0.747981f, 0.516859f),
                    CPColor(0.997138f, 0.75519f, 0.522806f),
                    CPColor(0.997019f, 0.762398f, 0.528821f),
                    CPColor(0.996898f, 0.769591f, 0.534892f),
                    CPColor(0.996727f, 0.776795f, 0.541039f),
                    CPColor(0.996571f, 0.783977f, 0.547233f),
                    CPColor(0.996369f, 0.791167f, 0.553499f),
                    CPColor(0.996162f, 0.798348f, 0.55982f),
                    CPColor(0.995932f, 0.805527f, 0.566202f),
                    CPColor(0.99568f, 0.812706f, 0.572645f),
                    CPColor(0.995424f, 0.819875f, 0.57914f),
                    CPColor(0.995131f, 0.827052f, 0.585701f),
                    CPColor(0.994851f, 0.834213f, 0.592307f),
                    CPColor(0.994524f, 0.841387f, 0.598983f),
                    CPColor(0.994222f, 0.84854f, 0.605696f),
                    CPColor(0.993866f, 0.855711f, 0.612482f),
                    CPColor(0.993545f, 0.862859f, 0.619299f),
                    CPColor(0.99317f, 0.870024f, 0.626189f),
                    CPColor(0.992831f, 0.877168f, 0.633109f),
                    CPColor(0.99244f, 0.88433f, 0.640099f),
                    CPColor(0.992089f, 0.89147f, 0.647116f),
                    CPColor(0.991688f, 0.898627f, 0.654202f),
                    CPColor(0.991332f, 0.905763f, 0.661309f),
                    CPColor(0.99093f, 0.912915f, 0.668481f),
                    CPColor(0.99057f, 0.920049f, 0.675675f),
                    CPColor(0.990175f, 0.927196f, 0.682926f),
                    CPColor(0.989815f, 0.934329f, 0.690198f),
                    CPColor(0.989434f, 0.94147f, 0.697519f),
                    CPColor(0.989077f, 0.948604f, 0.704863f),
                    CPColor(0.988717f, 0.955742f, 0.712242f),
                    CPColor(0.988367f, 0.962878f, 0.719649f),
                    CPColor(0.988033f, 0.970012f, 0.727077f),
                    CPColor(0.987691f, 0.977154f, 0.734536f),
                    CPColor(0.987387f, 0.984288f, 0.742002f),
                    CPColor(0.987053f, 0.991438f, 0.749504f)
                )
            )
        )
        inferno = Entry(
            "inferno", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    CPColor(0.001462f, 0.000466f, 0.013866f),
                    CPColor(0.002267f, 0.00127f, 0.01857f),
                    CPColor(0.003299f, 0.002249f, 0.024239f),
                    CPColor(0.004547f, 0.003392f, 0.030909f),
                    CPColor(0.006006f, 0.004692f, 0.038558f),
                    CPColor(0.007676f, 0.006136f, 0.046836f),
                    CPColor(0.009561f, 0.007713f, 0.055143f),
                    CPColor(0.011663f, 0.009417f, 0.06346f),
                    CPColor(0.013995f, 0.011225f, 0.071862f),
                    CPColor(0.016561f, 0.013136f, 0.080282f),
                    CPColor(0.019373f, 0.015133f, 0.088767f),
                    CPColor(0.022447f, 0.017199f, 0.097327f),
                    CPColor(0.025793f, 0.019331f, 0.10593f),
                    CPColor(0.029432f, 0.021503f, 0.114621f),
                    CPColor(0.033385f, 0.023702f, 0.123397f),
                    CPColor(0.037668f, 0.025921f, 0.132232f),
                    CPColor(0.042253f, 0.028139f, 0.141141f),
                    CPColor(0.046915f, 0.030324f, 0.150164f),
                    CPColor(0.051644f, 0.032474f, 0.159254f),
                    CPColor(0.056449f, 0.034569f, 0.168414f),
                    CPColor(0.06134f, 0.03659f, 0.177642f),
                    CPColor(0.066331f, 0.038504f, 0.186962f),
                    CPColor(0.071429f, 0.040294f, 0.196354f),
                    CPColor(0.076637f, 0.041905f, 0.205799f),
                    CPColor(0.081962f, 0.043328f, 0.215289f),
                    CPColor(0.087411f, 0.044556f, 0.224813f),
                    CPColor(0.09299f, 0.045583f, 0.234358f),
                    CPColor(0.098702f, 0.046402f, 0.243904f),
                    CPColor(0.104551f, 0.047008f, 0.25343f),
                    CPColor(0.110536f, 0.047399f, 0.262912f),
                    CPColor(0.116656f, 0.047574f, 0.272321f),
                    CPColor(0.122908f, 0.047536f, 0.281624f),
                    CPColor(0.129285f, 0.047293f, 0.290788f),
                    CPColor(0.135778f, 0.046856f, 0.299776f),
                    CPColor(0.142378f, 0.046242f, 0.308553f),
                    CPColor(0.149073f, 0.045468f, 0.317085f),
                    CPColor(0.15585f, 0.044559f, 0.325338f),
                    CPColor(0.162689f, 0.043554f, 0.333277f),
                    CPColor(0.169575f, 0.042489f, 0.340874f),
                    CPColor(0.176493f, 0.041402f, 0.348111f),
                    CPColor(0.183429f, 0.040329f, 0.354971f),
                    CPColor(0.190367f, 0.039309f, 0.361447f),
                    CPColor(0.197297f, 0.0384f, 0.367535f),
                    CPColor(0.204209f, 0.037632f, 0.373238f),
                    CPColor(0.211095f, 0.03703f, 0.378563f),
                    CPColor(0.217949f, 0.036615f, 0.383522f),
                    CPColor(0.224763f, 0.036405f, 0.388129f),
                    CPColor(0.231538f, 0.036405f, 0.3924f),
                    CPColor(0.238273f, 0.036621f, 0.396353f),
                    CPColor(0.244967f, 0.037055f, 0.400007f),
                    CPColor(0.25162f, 0.037705f, 0.403378f),
                    CPColor(0.258234f, 0.038571f, 0.406485f),
                    CPColor(0.26481f, 0.039647f, 0.409345f),
                    CPColor(0.271347f, 0.040922f, 0.411976f),
                    CPColor(0.27785f, 0.042353f, 0.414392f),
                    CPColor(0.284321f, 0.043933f, 0.416608f),
                    CPColor(0.290763f, 0.045644f, 0.418637f),
                    CPColor(0.297178f, 0.04747f, 0.420491f),
                    CPColor(0.303568f, 0.049396f, 0.422182f),
                    CPColor(0.309935f, 0.051407f, 0.423721f),
                    CPColor(0.316282f, 0.05349f, 0.425116f),
                    CPColor(0.32261f, 0.055634f, 0.426377f),
                    CPColor(0.328921f, 0.057827f, 0.427511f),
                    CPColor(0.335217f, 0.06006f, 0.428524f),
                    CPColor(0.3415f, 0.062325f, 0.429425f),
                    CPColor(0.347771f, 0.064616f, 0.430217f),
                    CPColor(0.354032f, 0.066925f, 0.430906f),
                    CPColor(0.360284f, 0.069247f, 0.431497f),
                    CPColor(0.366529f, 0.071579f, 0.431994f),
                    CPColor(0.372768f, 0.073915f, 0.4324f),
                    CPColor(0.379001f, 0.076253f, 0.432719f),
                    CPColor(0.385228f, 0.078591f, 0.432955f),
                    CPColor(0.391453f, 0.080927f, 0.433109f),
                    CPColor(0.397674f, 0.083257f, 0.433183f),
                    CPColor(0.403894f, 0.08558f, 0.433179f),
                    CPColor(0.410113f, 0.087896f, 0.433098f),
                    CPColor(0.416331f, 0.090203f, 0.432943f),
                    CPColor(0.422549f, 0.092501f, 0.432714f),
                    CPColor(0.428768f, 0.09479f, 0.432412f),
                    CPColor(0.434987f, 0.097069f, 0.432039f),
                    CPColor(0.441207f, 0.099338f, 0.431594f),
                    CPColor(0.447428f, 0.101597f, 0.43108f),
                    CPColor(0.453651f, 0.103848f, 0.430498f),
                    CPColor(0.459875f, 0.106089f, 0.429846f),
                    CPColor(0.4661f, 0.108322f, 0.429125f),
                    CPColor(0.472328f, 0.110547f, 0.428334f),
                    CPColor(0.478558f, 0.112764f, 0.427475f),
                    CPColor(0.484789f, 0.114974f, 0.426548f),
                    CPColor(0.491022f, 0.117179f, 0.425552f),
                    CPColor(0.497257f, 0.119379f, 0.424488f),
                    CPColor(0.503493f, 0.121575f, 0.423356f),
                    CPColor(0.50973f, 0.123769f, 0.422156f),
                    CPColor(0.515967f, 0.12596f, 0.420887f),
                    CPColor(0.522206f, 0.12815f, 0.419549f),
                    CPColor(0.528444f, 0.130341f, 0.418142f),
                    CPColor(0.534683f, 0.132534f, 0.416667f),
                    CPColor(0.54092f, 0.134729f, 0.415123f),
                    CPColor(0.547157f, 0.136929f, 0.413511f),
                    CPColor(0.553392f, 0.139134f, 0.411829f),
                    CPColor(0.559624f, 0.141346f, 0.410078f),
                    CPColor(0.565854f, 0.143567f, 0.408258f),
                    CPColor(0.572081f, 0.145797f, 0.406369f),
                    CPColor(0.578304f, 0.148039f, 0.404411f),
                    CPColor(0.584521f, 0.150294f, 0.402385f),
                    CPColor(0.590734f, 0.152563f, 0.40029f),
                    CPColor(0.59694f, 0.154848f, 0.398125f),
                    CPColor(0.603139f, 0.157151f, 0.395891f),
                    CPColor(0.60933f, 0.159474f, 0.393589f),
                    CPColor(0.615513f, 0.161817f, 0.391219f),
                    CPColor(0.621685f, 0.164184f, 0.388781f),
                    CPColor(0.627847f, 0.166575f, 0.386276f),
                    CPColor(0.633998f, 0.168992f, 0.383704f),
                    CPColor(0.640135f, 0.171438f, 0.381065f),
                    CPColor(0.64626f, 0.173914f, 0.378359f),
                    CPColor(0.652369f, 0.176421f, 0.375586f),
                    CPColor(0.658463f, 0.178962f, 0.372748f),
                    CPColor(0.66454f, 0.181539f, 0.369846f),
                    CPColor(0.670599f, 0.184153f, 0.366879f),
                    CPColor(0.676638f, 0.186807f, 0.363849f),
                    CPColor(0.682656f, 0.189501f, 0.360757f),
                    CPColor(0.688653f, 0.192239f, 0.357603f),
                    CPColor(0.694627f, 0.195021f, 0.354388f),
                    CPColor(0.700576f, 0.197851f, 0.351113f),
                    CPColor(0.7065f, 0.200728f, 0.347777f),
                    CPColor(0.712396f, 0.203656f, 0.344383f),
                    CPColor(0.718264f, 0.206636f, 0.340931f),
                    CPColor(0.724103f, 0.20967f, 0.337424f),
                    CPColor(0.729909f, 0.212759f, 0.333861f),
                    CPColor(0.735683f, 0.215906f, 0.330245f),
                    CPColor(0.741423f, 0.219112f, 0.326576f),
                    CPColor(0.747127f, 0.222378f, 0.322856f),
                    CPColor(0.752794f, 0.225706f, 0.319085f),
                    CPColor(0.758422f, 0.229097f, 0.315266f),
                    CPColor(0.76401f, 0.232554f, 0.311399f),
                    CPColor(0.769556f, 0.236077f, 0.307485f),
                    CPColor(0.775059f, 0.239667f, 0.303526f),
                    CPColor(0.780517f, 0.243327f, 0.299523f),
                    CPColor(0.785929f, 0.247056f, 0.295477f),
                    CPColor(0.791293f, 0.250856f, 0.29139f),
                    CPColor(0.796607f, 0.254728f, 0.287264f),
                    CPColor(0.801871f, 0.258674f, 0.283099f),
                    CPColor(0.807082f, 0.262692f, 0.278898f),
                    CPColor(0.812239f, 0.266786f, 0.274661f),
                    CPColor(0.817341f, 0.270954f, 0.27039f),
                    CPColor(0.822386f, 0.275197f, 0.266085f),
                    CPColor(0.827372f, 0.279517f, 0.26175f),
                    CPColor(0.832299f, 0.283913f, 0.257383f),
                    CPColor(0.837165f, 0.288385f, 0.252988f),
                    CPColor(0.841969f, 0.292933f, 0.248564f),
                    CPColor(0.846709f, 0.297559f, 0.244113f),
                    CPColor(0.851384f, 0.30226f, 0.239636f),
                    CPColor(0.855992f, 0.307038f, 0.235133f),
                    CPColor(0.860533f, 0.311892f, 0.230606f),
                    CPColor(0.865006f, 0.316822f, 0.226055f),
                    CPColor(0.869409f, 0.321827f, 0.221482f),
                    CPColor(0.873741f, 0.326906f, 0.216886f),
                    CPColor(0.878001f, 0.33206f, 0.212268f),
                    CPColor(0.882188f, 0.337287f, 0.207628f),
                    CPColor(0.886302f, 0.342586f, 0.202968f),
                    CPColor(0.890341f, 0.347957f, 0.198286f),
                    CPColor(0.894305f, 0.353399f, 0.193584f),
                    CPColor(0.898192f, 0.358911f, 0.18886f),
                    CPColor(0.902003f, 0.364492f, 0.184116f),
                    CPColor(0.905735f, 0.37014f, 0.17935f),
                    CPColor(0.90939f, 0.375856f, 0.174563f),
                    CPColor(0.912966f, 0.381636f, 0.169755f),
                    CPColor(0.916462f, 0.387481f, 0.164924f),
                    CPColor(0.919879f, 0.393389f, 0.16007f),
                    CPColor(0.923215f, 0.399359f, 0.155193f),
                    CPColor(0.92647f, 0.405389f, 0.150292f),
                    CPColor(0.929644f, 0.411479f, 0.145367f),
                    CPColor(0.932737f, 0.417627f, 0.140417f),
                    CPColor(0.935747f, 0.423831f, 0.13544f),
                    CPColor(0.938675f, 0.430091f, 0.130438f),
                    CPColor(0.941521f, 0.436405f, 0.125409f),
                    CPColor(0.944285f, 0.442772f, 0.120354f),
                    CPColor(0.946965f, 0.449191f, 0.115272f),
                    CPColor(0.949562f, 0.45566f, 0.110164f),
                    CPColor(0.952075f, 0.462178f, 0.105031f),
                    CPColor(0.954506f, 0.468744f, 0.099874f),
                    CPColor(0.956852f, 0.475356f, 0.094695f),
                    CPColor(0.959114f, 0.482014f, 0.089499f),
                    CPColor(0.961293f, 0.488716f, 0.084289f),
                    CPColor(0.963387f, 0.495462f, 0.079073f),
                    CPColor(0.965397f, 0.502249f, 0.073859f),
                    CPColor(0.967322f, 0.509078f, 0.068659f),
                    CPColor(0.969163f, 0.515946f, 0.063488f),
                    CPColor(0.970919f, 0.522853f, 0.058367f),
                    CPColor(0.97259f, 0.529798f, 0.053324f),
                    CPColor(0.974176f, 0.53678f, 0.048392f),
                    CPColor(0.975677f, 0.543798f, 0.043618f),
                    CPColor(0.977092f, 0.55085f, 0.03905f),
                    CPColor(0.978422f, 0.557937f, 0.034931f),
                    CPColor(0.979666f, 0.565057f, 0.031409f),
                    CPColor(0.980824f, 0.572209f, 0.028508f),
                    CPColor(0.981895f, 0.579392f, 0.02625f),
                    CPColor(0.982881f, 0.586606f, 0.024661f),
                    CPColor(0.983779f, 0.593849f, 0.02377f),
                    CPColor(0.984591f, 0.601122f, 0.023606f),
                    CPColor(0.985315f, 0.608422f, 0.024202f),
                    CPColor(0.985952f, 0.61575f, 0.025592f),
                    CPColor(0.986502f, 0.623105f, 0.027814f),
                    CPColor(0.986964f, 0.630485f, 0.030908f),
                    CPColor(0.987337f, 0.63789f, 0.034916f),
                    CPColor(0.987622f, 0.64532f, 0.039886f),
                    CPColor(0.987819f, 0.652773f, 0.045581f),
                    CPColor(0.987926f, 0.66025f, 0.05175f),
                    CPColor(0.987945f, 0.667748f, 0.058329f),
                    CPColor(0.987874f, 0.675267f, 0.065257f),
                    CPColor(0.987714f, 0.682807f, 0.072489f),
                    CPColor(0.987464f, 0.690366f, 0.07999f),
                    CPColor(0.987124f, 0.697944f, 0.087731f),
                    CPColor(0.986694f, 0.70554f, 0.095694f),
                    CPColor(0.986175f, 0.713153f, 0.103863f),
                    CPColor(0.985566f, 0.720782f, 0.112229f),
                    CPColor(0.984865f, 0.728427f, 0.120785f),
                    CPColor(0.984075f, 0.736087f, 0.129527f),
                    CPColor(0.983196f, 0.743758f, 0.138453f),
                    CPColor(0.982228f, 0.751442f, 0.147565f),
                    CPColor(0.981173f, 0.759135f, 0.156863f),
                    CPColor(0.980032f, 0.766837f, 0.166353f),
                    CPColor(0.978806f, 0.774545f, 0.176037f),
                    CPColor(0.977497f, 0.782258f, 0.185923f),
                    CPColor(0.976108f, 0.789974f, 0.196018f),
                    CPColor(0.974638f, 0.797692f, 0.206332f),
                    CPColor(0.973088f, 0.805409f, 0.216877f),
                    CPColor(0.971468f, 0.813122f, 0.227658f),
                    CPColor(0.969783f, 0.820825f, 0.238686f),
                    CPColor(0.968041f, 0.828515f, 0.249972f),
                    CPColor(0.966243f, 0.836191f, 0.261534f),
                    CPColor(0.964394f, 0.843848f, 0.273391f),
                    CPColor(0.962517f, 0.851476f, 0.285546f),
                    CPColor(0.960626f, 0.859069f, 0.29801f),
                    CPColor(0.95872f, 0.866624f, 0.31082f),
                    CPColor(0.956834f, 0.874129f, 0.323974f),
                    CPColor(0.954997f, 0.881569f, 0.337475f),
                    CPColor(0.953215f, 0.888942f, 0.351369f),
                    CPColor(0.951546f, 0.896226f, 0.365627f),
                    CPColor(0.950018f, 0.903409f, 0.380271f),
                    CPColor(0.948683f, 0.910473f, 0.395289f),
                    CPColor(0.947594f, 0.917399f, 0.410665f),
                    CPColor(0.946809f, 0.924168f, 0.426373f),
                    CPColor(0.946392f, 0.930761f, 0.442367f),
                    CPColor(0.946403f, 0.937159f, 0.458592f),
                    CPColor(0.946903f, 0.943348f, 0.47497f),
                    CPColor(0.947937f, 0.949318f, 0.491426f),
                    CPColor(0.949545f, 0.955063f, 0.50786f),
                    CPColor(0.95174f, 0.960587f, 0.524203f),
                    CPColor(0.954529f, 0.965896f, 0.540361f),
                    CPColor(0.957896f, 0.971003f, 0.556275f),
                    CPColor(0.961812f, 0.975924f, 0.571925f),
                    CPColor(0.966249f, 0.980678f, 0.587206f),
                    CPColor(0.971162f, 0.985282f, 0.602154f),
                    CPColor(0.976511f, 0.989753f, 0.61676f),
                    CPColor(0.982257f, 0.994109f, 0.631017f),
                    CPColor(0.988362f, 0.998364f, 0.644924f)
                )
            )
        )
        plasma = Entry(
            "plasma", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    CPColor(0.050383f, 0.029803f, 0.527975f),
                    CPColor(0.063536f, 0.028426f, 0.533124f),
                    CPColor(0.075353f, 0.027206f, 0.538007f),
                    CPColor(0.086222f, 0.026125f, 0.542658f),
                    CPColor(0.096379f, 0.025165f, 0.547103f),
                    CPColor(0.10598f, 0.024309f, 0.551368f),
                    CPColor(0.115124f, 0.023556f, 0.555468f),
                    CPColor(0.123903f, 0.022878f, 0.559423f),
                    CPColor(0.132381f, 0.022258f, 0.56325f),
                    CPColor(0.140603f, 0.021687f, 0.566959f),
                    CPColor(0.148607f, 0.021154f, 0.570562f),
                    CPColor(0.156421f, 0.020651f, 0.574065f),
                    CPColor(0.16407f, 0.020171f, 0.577478f),
                    CPColor(0.171574f, 0.019706f, 0.580806f),
                    CPColor(0.17895f, 0.019252f, 0.584054f),
                    CPColor(0.186213f, 0.018803f, 0.587228f),
                    CPColor(0.193374f, 0.018354f, 0.59033f),
                    CPColor(0.200445f, 0.017902f, 0.593364f),
                    CPColor(0.207435f, 0.017442f, 0.596333f),
                    CPColor(0.21435f, 0.016973f, 0.599239f),
                    CPColor(0.221197f, 0.016497f, 0.602083f),
                    CPColor(0.227983f, 0.016007f, 0.604867f),
                    CPColor(0.234715f, 0.015502f, 0.607592f),
                    CPColor(0.241396f, 0.014979f, 0.610259f),
                    CPColor(0.248032f, 0.014439f, 0.612868f),
                    CPColor(0.254627f, 0.013882f, 0.615419f),
                    CPColor(0.261183f, 0.013308f, 0.617911f),
                    CPColor(0.267703f, 0.012716f, 0.620346f),
                    CPColor(0.274191f, 0.012109f, 0.622722f),
                    CPColor(0.280648f, 0.011488f, 0.625038f),
                    CPColor(0.287076f, 0.010855f, 0.627295f),
                    CPColor(0.293478f, 0.010213f, 0.62949f),
                    CPColor(0.299855f, 0.009561f, 0.631624f),
                    CPColor(0.30621f, 0.008902f, 0.633694f),
                    CPColor(0.312543f, 0.008239f, 0.6357f),
                    CPColor(0.318856f, 0.007576f, 0.63764f),
                    CPColor(0.32515f, 0.006915f, 0.639512f),
                    CPColor(0.331426f, 0.006261f, 0.641316f),
                    CPColor(0.337683f, 0.005618f, 0.643049f),
                    CPColor(0.343925f, 0.004991f, 0.64471f),
                    CPColor(0.35015f, 0.004382f, 0.646298f),
                    CPColor(0.356359f, 0.003798f, 0.64781f),
                    CPColor(0.362553f, 0.003243f, 0.649245f),
                    CPColor(0.368733f, 0.002724f, 0.650601f),
                    CPColor(0.374897f, 0.002245f, 0.651876f),
                    CPColor(0.381047f, 0.001814f, 0.653068f),
                    CPColor(0.387183f, 0.001434f, 0.654177f),
                    CPColor(0.393304f, 0.001114f, 0.655199f),
                    CPColor(0.399411f, 0.000859f, 0.656133f),
                    CPColor(0.405503f, 0.000678f, 0.656977f),
                    CPColor(0.41158f, 0.000577f, 0.65773f),
                    CPColor(0.417642f, 0.000564f, 0.65839f),
                    CPColor(0.423689f, 0.000646f, 0.658956f),
                    CPColor(0.429719f, 0.000831f, 0.659425f),
                    CPColor(0.435734f, 0.001127f, 0.659797f),
                    CPColor(0.441732f, 0.00154f, 0.660069f),
                    CPColor(0.447714f, 0.00208f, 0.66024f),
                    CPColor(0.453677f, 0.002755f, 0.66031f),
                    CPColor(0.459623f, 0.003574f, 0.660277f),
                    CPColor(0.46555f, 0.004545f, 0.660139f),
                    CPColor(0.471457f, 0.005678f, 0.659897f),
                    CPColor(0.477344f, 0.00698f, 0.659549f),
                    CPColor(0.48321f, 0.00846f, 0.659095f),
                    CPColor(0.489055f, 0.010127f, 0.658534f),
                    CPColor(0.494877f, 0.01199f, 0.657865f),
                    CPColor(0.500678f, 0.014055f, 0.657088f),
                    CPColor(0.506454f, 0.016333f, 0.656202f),
                    CPColor(0.512206f, 0.018833f, 0.655209f),
                    CPColor(0.517933f, 0.021563f, 0.654109f),
                    CPColor(0.523633f, 0.024532f, 0.652901f),
                    CPColor(0.529306f, 0.027747f, 0.651586f),
                    CPColor(0.534952f, 0.031217f, 0.650165f),
                    CPColor(0.54057f, 0.03495f, 0.64864f),
                    CPColor(0.546157f, 0.038954f, 0.64701f),
                    CPColor(0.551715f, 0.043136f, 0.645277f),
                    CPColor(0.557243f, 0.047331f, 0.643443f),
                    CPColor(0.562738f, 0.051545f, 0.641509f),
                    CPColor(0.568201f, 0.055778f, 0.639477f),
                    CPColor(0.573632f, 0.060028f, 0.637349f),
                    CPColor(0.579029f, 0.064296f, 0.635126f),
                    CPColor(0.584391f, 0.068579f, 0.632812f),
                    CPColor(0.589719f, 0.072878f, 0.630408f),
                    CPColor(0.595011f, 0.07719f, 0.627917f),
                    CPColor(0.600266f, 0.081516f, 0.625342f),
                    CPColor(0.605485f, 0.085854f, 0.622686f),
                    CPColor(0.610667f, 0.090204f, 0.619951f),
                    CPColor(0.615812f, 0.094564f, 0.61714f),
                    CPColor(0.620919f, 0.098934f, 0.614257f),
                    CPColor(0.625987f, 0.103312f, 0.611305f),
                    CPColor(0.631017f, 0.107699f, 0.608287f),
                    CPColor(0.636008f, 0.112092f, 0.605205f),
                    CPColor(0.640959f, 0.116492f, 0.602065f),
                    CPColor(0.645872f, 0.120898f, 0.598867f),
                    CPColor(0.650746f, 0.125309f, 0.595617f),
                    CPColor(0.65558f, 0.129725f, 0.592317f),
                    CPColor(0.660374f, 0.134144f, 0.588971f),
                    CPColor(0.665129f, 0.138566f, 0.585582f),
                    CPColor(0.669845f, 0.142992f, 0.582154f),
                    CPColor(0.674522f, 0.147419f, 0.578688f),
                    CPColor(0.67916f, 0.151848f, 0.575189f),
                    CPColor(0.683758f, 0.156278f, 0.57166f),
                    CPColor(0.688318f, 0.160709f, 0.568103f),
                    CPColor(0.69284f, 0.165141f, 0.564522f),
                    CPColor(0.697324f, 0.169573f, 0.560919f),
                    CPColor(0.701769f, 0.174005f, 0.557296f),
                    CPColor(0.706178f, 0.178437f, 0.553657f),
                    CPColor(0.710549f, 0.182868f, 0.550004f),
                    CPColor(0.714883f, 0.187299f, 0.546338f),
                    CPColor(0.719181f, 0.191729f, 0.542663f),
                    CPColor(0.723444f, 0.196158f, 0.538981f),
                    CPColor(0.72767f, 0.200586f, 0.535293f),
                    CPColor(0.731862f, 0.205013f, 0.531601f),
                    CPColor(0.736019f, 0.209439f, 0.527908f),
                    CPColor(0.740143f, 0.213864f, 0.524216f),
                    CPColor(0.744232f, 0.218288f, 0.520524f),
                    CPColor(0.748289f, 0.222711f, 0.516834f),
                    CPColor(0.752312f, 0.227133f, 0.513149f),
                    CPColor(0.756304f, 0.231555f, 0.509468f),
                    CPColor(0.760264f, 0.235976f, 0.505794f),
                    CPColor(0.764193f, 0.240396f, 0.502126f),
                    CPColor(0.76809f, 0.244817f, 0.498465f),
                    CPColor(0.771958f, 0.249237f, 0.494813f),
                    CPColor(0.775796f, 0.253658f, 0.491171f),
                    CPColor(0.779604f, 0.258078f, 0.487539f),
                    CPColor(0.783383f, 0.2625f, 0.483918f),
                    CPColor(0.787133f, 0.266922f, 0.480307f),
                    CPColor(0.790855f, 0.271345f, 0.476706f),
                    CPColor(0.794549f, 0.27577f, 0.473117f),
                    CPColor(0.798216f, 0.280197f, 0.469538f),
                    CPColor(0.801855f, 0.284626f, 0.465971f),
                    CPColor(0.805467f, 0.289057f, 0.462415f),
                    CPColor(0.809052f, 0.293491f, 0.45887f),
                    CPColor(0.812612f, 0.297928f, 0.455338f),
                    CPColor(0.816144f, 0.302368f, 0.451816f),
                    CPColor(0.819651f, 0.306812f, 0.448306f),
                    CPColor(0.823132f, 0.311261f, 0.444806f),
                    CPColor(0.826588f, 0.315714f, 0.441316f),
                    CPColor(0.830018f, 0.320172f, 0.437836f),
                    CPColor(0.833422f, 0.324635f, 0.434366f),
                    CPColor(0.836801f, 0.329105f, 0.430905f),
                    CPColor(0.840155f, 0.33358f, 0.427455f),
                    CPColor(0.843484f, 0.338062f, 0.424013f),
                    CPColor(0.846788f, 0.342551f, 0.420579f),
                    CPColor(0.850066f, 0.347048f, 0.417153f),
                    CPColor(0.853319f, 0.351553f, 0.413734f),
                    CPColor(0.856547f, 0.356066f, 0.410322f),
                    CPColor(0.85975f, 0.360588f, 0.406917f),
                    CPColor(0.862927f, 0.365119f, 0.403519f),
                    CPColor(0.866078f, 0.36966f, 0.400126f),
                    CPColor(0.869203f, 0.374212f, 0.396738f),
                    CPColor(0.872303f, 0.378774f, 0.393355f),
                    CPColor(0.875376f, 0.383347f, 0.389976f),
                    CPColor(0.878423f, 0.387932f, 0.3866f),
                    CPColor(0.881443f, 0.392529f, 0.383229f),
                    CPColor(0.884436f, 0.397139f, 0.37986f),
                    CPColor(0.887402f, 0.401762f, 0.376494f),
                    CPColor(0.89034f, 0.406398f, 0.37313f),
                    CPColor(0.89325f, 0.411048f, 0.369768f),
                    CPColor(0.896131f, 0.415712f, 0.366407f),
                    CPColor(0.898984f, 0.420392f, 0.363047f),
                    CPColor(0.901807f, 0.425087f, 0.359688f),
                    CPColor(0.904601f, 0.429797f, 0.356329f),
                    CPColor(0.907365f, 0.434524f, 0.35297f),
                    CPColor(0.910098f, 0.439268f, 0.34961f),
                    CPColor(0.9128f, 0.444029f, 0.346251f),
                    CPColor(0.915471f, 0.448807f, 0.34289f),
                    CPColor(0.918109f, 0.453603f, 0.339529f),
                    CPColor(0.920714f, 0.458417f, 0.336166f),
                    CPColor(0.923287f, 0.463251f, 0.332801f),
                    CPColor(0.925825f, 0.468103f, 0.329435f),
                    CPColor(0.928329f, 0.472975f, 0.326067f),
                    CPColor(0.930798f, 0.477867f, 0.322697f),
                    CPColor(0.933232f, 0.48278f, 0.319325f),
                    CPColor(0.93563f, 0.487712f, 0.315952f),
                    CPColor(0.93799f, 0.492667f, 0.312575f),
                    CPColor(0.940313f, 0.497642f, 0.309197f),
                    CPColor(0.942598f, 0.502639f, 0.305816f),
                    CPColor(0.944844f, 0.507658f, 0.302433f),
                    CPColor(0.947051f, 0.512699f, 0.299049f),
                    CPColor(0.949217f, 0.517763f, 0.295662f),
                    CPColor(0.951344f, 0.52285f, 0.292275f),
                    CPColor(0.953428f, 0.52796f, 0.288883f),
                    CPColor(0.95547f, 0.533093f, 0.28549f),
                    CPColor(0.957469f, 0.53825f, 0.282096f),
                    CPColor(0.959424f, 0.543431f, 0.278701f),
                    CPColor(0.961336f, 0.548636f, 0.275305f),
                    CPColor(0.963203f, 0.553865f, 0.271909f),
                    CPColor(0.965024f, 0.559118f, 0.268513f),
                    CPColor(0.966798f, 0.564396f, 0.265118f),
                    CPColor(0.968526f, 0.5697f, 0.261721f),
                    CPColor(0.970205f, 0.575028f, 0.258325f),
                    CPColor(0.971835f, 0.580382f, 0.254931f),
                    CPColor(0.973416f, 0.585761f, 0.25154f),
                    CPColor(0.974947f, 0.591165f, 0.248151f),
                    CPColor(0.976428f, 0.596595f, 0.244767f),
                    CPColor(0.977856f, 0.602051f, 0.241387f),
                    CPColor(0.979233f, 0.607532f, 0.238013f),
                    CPColor(0.980556f, 0.613039f, 0.234646f),
                    CPColor(0.981826f, 0.618572f, 0.231287f),
                    CPColor(0.983041f, 0.624131f, 0.227937f),
                    CPColor(0.984199f, 0.629718f, 0.224595f),
                    CPColor(0.985301f, 0.63533f, 0.221265f),
                    CPColor(0.986345f, 0.640969f, 0.217948f),
                    CPColor(0.987332f, 0.646633f, 0.214648f),
                    CPColor(0.98826f, 0.652325f, 0.211364f),
                    CPColor(0.989128f, 0.658043f, 0.2081f),
                    CPColor(0.989935f, 0.663787f, 0.204859f),
                    CPColor(0.990681f, 0.669558f, 0.201642f),
                    CPColor(0.991365f, 0.675355f, 0.198453f),
                    CPColor(0.991985f, 0.681179f, 0.195295f),
                    CPColor(0.992541f, 0.68703f, 0.19217f),
                    CPColor(0.993032f, 0.692907f, 0.189084f),
                    CPColor(0.993456f, 0.69881f, 0.186041f),
                    CPColor(0.993814f, 0.704741f, 0.183043f),
                    CPColor(0.994103f, 0.710698f, 0.180097f),
                    CPColor(0.994324f, 0.716681f, 0.177208f),
                    CPColor(0.994474f, 0.722691f, 0.174381f),
                    CPColor(0.994553f, 0.728728f, 0.171622f),
                    CPColor(0.994561f, 0.734791f, 0.168938f),
                    CPColor(0.994495f, 0.74088f, 0.166335f),
                    CPColor(0.994355f, 0.746995f, 0.163821f),
                    CPColor(0.994141f, 0.753137f, 0.161404f),
                    CPColor(0.993851f, 0.759304f, 0.159092f),
                    CPColor(0.993482f, 0.765499f, 0.156891f),
                    CPColor(0.993033f, 0.77172f, 0.154808f),
                    CPColor(0.992505f, 0.777967f, 0.152855f),
                    CPColor(0.991897f, 0.784239f, 0.151042f),
                    CPColor(0.991209f, 0.790537f, 0.149377f),
                    CPColor(0.990439f, 0.796859f, 0.14787f),
                    CPColor(0.989587f, 0.803205f, 0.146529f),
                    CPColor(0.988648f, 0.809579f, 0.145357f),
                    CPColor(0.987621f, 0.815978f, 0.144363f),
                    CPColor(0.986509f, 0.822401f, 0.143557f),
                    CPColor(0.985314f, 0.828846f, 0.142945f),
                    CPColor(0.984031f, 0.835315f, 0.142528f),
                    CPColor(0.982653f, 0.841812f, 0.142303f),
                    CPColor(0.98119f, 0.848329f, 0.142279f),
                    CPColor(0.979644f, 0.854866f, 0.142453f),
                    CPColor(0.977995f, 0.861432f, 0.142808f),
                    CPColor(0.976265f, 0.868016f, 0.143351f),
                    CPColor(0.974443f, 0.874622f, 0.144061f),
                    CPColor(0.97253f, 0.88125f, 0.144923f),
                    CPColor(0.970533f, 0.887896f, 0.145919f),
                    CPColor(0.968443f, 0.894564f, 0.147014f),
                    CPColor(0.966271f, 0.901249f, 0.14818f),
                    CPColor(0.964021f, 0.90795f, 0.14937f),
                    CPColor(0.961681f, 0.914672f, 0.15052f),
                    CPColor(0.959276f, 0.921407f, 0.151566f),
                    CPColor(0.956808f, 0.928152f, 0.152409f),
                    CPColor(0.954287f, 0.934908f, 0.152921f),
                    CPColor(0.951726f, 0.941671f, 0.152925f),
                    CPColor(0.949151f, 0.948435f, 0.152178f),
                    CPColor(0.946602f, 0.95519f, 0.150328f),
                    CPColor(0.944152f, 0.961916f, 0.146861f),
                    CPColor(0.941896f, 0.96859f, 0.140956f),
                    CPColor(0.940015f, 0.975158f, 0.131326f)
                )
            )
        )
        viridis = Entry(
            "viridis", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    CPColor(0.267004f, 0.004874f, 0.329415f),
                    CPColor(0.26851f, 0.009605f, 0.335427f),
                    CPColor(0.269944f, 0.014625f, 0.341379f),
                    CPColor(0.271305f, 0.019942f, 0.347269f),
                    CPColor(0.272594f, 0.025563f, 0.353093f),
                    CPColor(0.273809f, 0.031497f, 0.358853f),
                    CPColor(0.274952f, 0.037752f, 0.364543f),
                    CPColor(0.276022f, 0.044167f, 0.370164f),
                    CPColor(0.277018f, 0.050344f, 0.375715f),
                    CPColor(0.277941f, 0.056324f, 0.381191f),
                    CPColor(0.278791f, 0.062145f, 0.386592f),
                    CPColor(0.279566f, 0.067836f, 0.391917f),
                    CPColor(0.280267f, 0.073417f, 0.397163f),
                    CPColor(0.280894f, 0.078907f, 0.402329f),
                    CPColor(0.281446f, 0.08432f, 0.407414f),
                    CPColor(0.281924f, 0.089666f, 0.412415f),
                    CPColor(0.282327f, 0.094955f, 0.417331f),
                    CPColor(0.282656f, 0.100196f, 0.42216f),
                    CPColor(0.28291f, 0.105393f, 0.426902f),
                    CPColor(0.283091f, 0.110553f, 0.431554f),
                    CPColor(0.283197f, 0.11568f, 0.436115f),
                    CPColor(0.283229f, 0.120777f, 0.440584f),
                    CPColor(0.283187f, 0.125848f, 0.44496f),
                    CPColor(0.283072f, 0.130895f, 0.449241f),
                    CPColor(0.282884f, 0.13592f, 0.453427f),
                    CPColor(0.282623f, 0.140926f, 0.457517f),
                    CPColor(0.28229f, 0.145912f, 0.46151f),
                    CPColor(0.281887f, 0.150881f, 0.465405f),
                    CPColor(0.281412f, 0.155834f, 0.469201f),
                    CPColor(0.280868f, 0.160771f, 0.472899f),
                    CPColor(0.280255f, 0.165693f, 0.476498f),
                    CPColor(0.279574f, 0.170599f, 0.479997f),
                    CPColor(0.278826f, 0.17549f, 0.483397f),
                    CPColor(0.278012f, 0.180367f, 0.486697f),
                    CPColor(0.277134f, 0.185228f, 0.489898f),
                    CPColor(0.276194f, 0.190074f, 0.493001f),
                    CPColor(0.275191f, 0.194905f, 0.496005f),
                    CPColor(0.274128f, 0.199721f, 0.498911f),
                    CPColor(0.273006f, 0.20452f, 0.501721f),
                    CPColor(0.271828f, 0.209303f, 0.504434f),
                    CPColor(0.270595f, 0.214069f, 0.507052f),
                    CPColor(0.269308f, 0.218818f, 0.509577f),
                    CPColor(0.267968f, 0.223549f, 0.512008f),
                    CPColor(0.26658f, 0.228262f, 0.514349f),
                    CPColor(0.265145f, 0.232956f, 0.516599f),
                    CPColor(0.263663f, 0.237631f, 0.518762f),
                    CPColor(0.262138f, 0.242286f, 0.520837f),
                    CPColor(0.260571f, 0.246922f, 0.522828f),
                    CPColor(0.258965f, 0.251537f, 0.524736f),
                    CPColor(0.257322f, 0.25613f, 0.526563f),
                    CPColor(0.255645f, 0.260703f, 0.528312f),
                    CPColor(0.253935f, 0.265254f, 0.529983f),
                    CPColor(0.252194f, 0.269783f, 0.531579f),
                    CPColor(0.250425f, 0.27429f, 0.533103f),
                    CPColor(0.248629f, 0.278775f, 0.534556f),
                    CPColor(0.246811f, 0.283237f, 0.535941f),
                    CPColor(0.244972f, 0.287675f, 0.53726f),
                    CPColor(0.243113f, 0.292092f, 0.538516f),
                    CPColor(0.241237f, 0.296485f, 0.539709f),
                    CPColor(0.239346f, 0.300855f, 0.540844f),
                    CPColor(0.237441f, 0.305202f, 0.541921f),
                    CPColor(0.235526f, 0.309527f, 0.542944f),
                    CPColor(0.233603f, 0.313828f, 0.543914f),
                    CPColor(0.231674f, 0.318106f, 0.544834f),
                    CPColor(0.229739f, 0.322361f, 0.545706f),
                    CPColor(0.227802f, 0.326594f, 0.546532f),
                    CPColor(0.225863f, 0.330805f, 0.547314f),
                    CPColor(0.223925f, 0.334994f, 0.548053f),
                    CPColor(0.221989f, 0.339161f, 0.548752f),
                    CPColor(0.220057f, 0.343307f, 0.549413f),
                    CPColor(0.21813f, 0.347432f, 0.550038f),
                    CPColor(0.21621f, 0.351535f, 0.550627f),
                    CPColor(0.214298f, 0.355619f, 0.551184f),
                    CPColor(0.212395f, 0.359683f, 0.55171f),
                    CPColor(0.210503f, 0.363727f, 0.552206f),
                    CPColor(0.208623f, 0.367752f, 0.552675f),
                    CPColor(0.206756f, 0.371758f, 0.553117f),
                    CPColor(0.204903f, 0.375746f, 0.553533f),
                    CPColor(0.203063f, 0.379716f, 0.553925f),
                    CPColor(0.201239f, 0.38367f, 0.554294f),
                    CPColor(0.19943f, 0.387607f, 0.554642f),
                    CPColor(0.197636f, 0.391528f, 0.554969f),
                    CPColor(0.19586f, 0.395433f, 0.555276f),
                    CPColor(0.1941f, 0.399323f, 0.555565f),
                    CPColor(0.192357f, 0.403199f, 0.555836f),
                    CPColor(0.190631f, 0.407061f, 0.556089f),
                    CPColor(0.188923f, 0.41091f, 0.556326f),
                    CPColor(0.187231f, 0.414746f, 0.556547f),
                    CPColor(0.185556f, 0.41857f, 0.556753f),
                    CPColor(0.183898f, 0.422383f, 0.556944f),
                    CPColor(0.182256f, 0.426184f, 0.55712f),
                    CPColor(0.180629f, 0.429975f, 0.557282f),
                    CPColor(0.179019f, 0.433756f, 0.55743f),
                    CPColor(0.177423f, 0.437527f, 0.557565f),
                    CPColor(0.175841f, 0.44129f, 0.557685f),
                    CPColor(0.174274f, 0.445044f, 0.557792f),
                    CPColor(0.172719f, 0.448791f, 0.557885f),
                    CPColor(0.171176f, 0.45253f, 0.557965f),
                    CPColor(0.169646f, 0.456262f, 0.55803f),
                    CPColor(0.168126f, 0.459988f, 0.558082f),
                    CPColor(0.166617f, 0.463708f, 0.558119f),
                    CPColor(0.165117f, 0.467423f, 0.558141f),
                    CPColor(0.163625f, 0.471133f, 0.558148f),
                    CPColor(0.162142f, 0.474838f, 0.55814f),
                    CPColor(0.160665f, 0.47854f, 0.558115f),
                    CPColor(0.159194f, 0.482237f, 0.558073f),
                    CPColor(0.157729f, 0.485932f, 0.558013f),
                    CPColor(0.15627f, 0.489624f, 0.557936f),
                    CPColor(0.154815f, 0.493313f, 0.55784f),
                    CPColor(0.153364f, 0.497f, 0.557724f),
                    CPColor(0.151918f, 0.500685f, 0.557587f),
                    CPColor(0.150476f, 0.504369f, 0.55743f),
                    CPColor(0.149039f, 0.508051f, 0.55725f),
                    CPColor(0.147607f, 0.511733f, 0.557049f),
                    CPColor(0.14618f, 0.515413f, 0.556823f),
                    CPColor(0.144759f, 0.519093f, 0.556572f),
                    CPColor(0.143343f, 0.522773f, 0.556295f),
                    CPColor(0.141935f, 0.526453f, 0.555991f),
                    CPColor(0.140536f, 0.530132f, 0.555659f),
                    CPColor(0.139147f, 0.533812f, 0.555298f),
                    CPColor(0.13777f, 0.537492f, 0.554906f),
                    CPColor(0.136408f, 0.541173f, 0.554483f),
                    CPColor(0.135066f, 0.544853f, 0.554029f),
                    CPColor(0.133743f, 0.548535f, 0.553541f),
                    CPColor(0.132444f, 0.552216f, 0.553018f),
                    CPColor(0.131172f, 0.555899f, 0.552459f),
                    CPColor(0.129933f, 0.559582f, 0.551864f),
                    CPColor(0.128729f, 0.563265f, 0.551229f),
                    CPColor(0.127568f, 0.566949f, 0.550556f),
                    CPColor(0.126453f, 0.570633f, 0.549841f),
                    CPColor(0.125394f, 0.574318f, 0.549086f),
                    CPColor(0.124395f, 0.578002f, 0.548287f),
                    CPColor(0.123463f, 0.581687f, 0.547445f),
                    CPColor(0.122606f, 0.585371f, 0.546557f),
                    CPColor(0.121831f, 0.589055f, 0.545623f),
                    CPColor(0.121148f, 0.592739f, 0.544641f),
                    CPColor(0.120565f, 0.596422f, 0.543611f),
                    CPColor(0.120092f, 0.600104f, 0.54253f),
                    CPColor(0.119738f, 0.603785f, 0.5414f),
                    CPColor(0.119512f, 0.607464f, 0.540218f),
                    CPColor(0.119423f, 0.611141f, 0.538982f),
                    CPColor(0.119483f, 0.614817f, 0.537692f),
                    CPColor(0.119699f, 0.61849f, 0.536347f),
                    CPColor(0.120081f, 0.622161f, 0.534946f),
                    CPColor(0.120638f, 0.625828f, 0.533488f),
                    CPColor(0.12138f, 0.629492f, 0.531973f),
                    CPColor(0.122312f, 0.633153f, 0.530398f),
                    CPColor(0.123444f, 0.636809f, 0.528763f),
                    CPColor(0.12478f, 0.640461f, 0.527068f),
                    CPColor(0.126326f, 0.644107f, 0.525311f),
                    CPColor(0.128087f, 0.647749f, 0.523491f),
                    CPColor(0.130067f, 0.651384f, 0.521608f),
                    CPColor(0.132268f, 0.655014f, 0.519661f),
                    CPColor(0.134692f, 0.658636f, 0.517649f),
                    CPColor(0.137339f, 0.662252f, 0.515571f),
                    CPColor(0.14021f, 0.665859f, 0.513427f),
                    CPColor(0.143303f, 0.669459f, 0.511215f),
                    CPColor(0.146616f, 0.67305f, 0.508936f),
                    CPColor(0.150148f, 0.676631f, 0.506589f),
                    CPColor(0.153894f, 0.680203f, 0.504172f),
                    CPColor(0.157851f, 0.683765f, 0.501686f),
                    CPColor(0.162016f, 0.687316f, 0.499129f),
                    CPColor(0.166383f, 0.690856f, 0.496502f),
                    CPColor(0.170948f, 0.694384f, 0.493803f),
                    CPColor(0.175707f, 0.6979f, 0.491033f),
                    CPColor(0.180653f, 0.701402f, 0.488189f),
                    CPColor(0.185783f, 0.704891f, 0.485273f),
                    CPColor(0.19109f, 0.708366f, 0.482284f),
                    CPColor(0.196571f, 0.711827f, 0.479221f),
                    CPColor(0.202219f, 0.715272f, 0.476084f),
                    CPColor(0.20803f, 0.718701f, 0.472873f),
                    CPColor(0.214f, 0.722114f, 0.469588f),
                    CPColor(0.220124f, 0.725509f, 0.466226f),
                    CPColor(0.226397f, 0.728888f, 0.462789f),
                    CPColor(0.232815f, 0.732247f, 0.459277f),
                    CPColor(0.239374f, 0.735588f, 0.455688f),
                    CPColor(0.24607f, 0.73891f, 0.452024f),
                    CPColor(0.252899f, 0.742211f, 0.448284f),
                    CPColor(0.259857f, 0.745492f, 0.444467f),
                    CPColor(0.266941f, 0.748751f, 0.440573f),
                    CPColor(0.274149f, 0.751988f, 0.436601f),
                    CPColor(0.281477f, 0.755203f, 0.432552f),
                    CPColor(0.288921f, 0.758394f, 0.428426f),
                    CPColor(0.296479f, 0.761561f, 0.424223f),
                    CPColor(0.304148f, 0.764704f, 0.419943f),
                    CPColor(0.311925f, 0.767822f, 0.415586f),
                    CPColor(0.319809f, 0.770914f, 0.411152f),
                    CPColor(0.327796f, 0.77398f, 0.40664f),
                    CPColor(0.335885f, 0.777018f, 0.402049f),
                    CPColor(0.344074f, 0.780029f, 0.397381f),
                    CPColor(0.35236f, 0.783011f, 0.392636f),
                    CPColor(0.360741f, 0.785964f, 0.387814f),
                    CPColor(0.369214f, 0.788888f, 0.382914f),
                    CPColor(0.377779f, 0.791781f, 0.377939f),
                    CPColor(0.386433f, 0.794644f, 0.372886f),
                    CPColor(0.395174f, 0.797475f, 0.367757f),
                    CPColor(0.404001f, 0.800275f, 0.362552f),
                    CPColor(0.412913f, 0.803041f, 0.357269f),
                    CPColor(0.421908f, 0.805774f, 0.35191f),
                    CPColor(0.430983f, 0.808473f, 0.346476f),
                    CPColor(0.440137f, 0.811138f, 0.340967f),
                    CPColor(0.449368f, 0.813768f, 0.335384f),
                    CPColor(0.458674f, 0.816363f, 0.329727f),
                    CPColor(0.468053f, 0.818921f, 0.323998f),
                    CPColor(0.477504f, 0.821444f, 0.318195f),
                    CPColor(0.487026f, 0.823929f, 0.312321f),
                    CPColor(0.496615f, 0.826376f, 0.306377f),
                    CPColor(0.506271f, 0.828786f, 0.300362f),
                    CPColor(0.515992f, 0.831158f, 0.294279f),
                    CPColor(0.525776f, 0.833491f, 0.288127f),
                    CPColor(0.535621f, 0.835785f, 0.281908f),
                    CPColor(0.545524f, 0.838039f, 0.275626f),
                    CPColor(0.555484f, 0.840254f, 0.269281f),
                    CPColor(0.565498f, 0.84243f, 0.262877f),
                    CPColor(0.575563f, 0.844566f, 0.256415f),
                    CPColor(0.585678f, 0.846661f, 0.249897f),
                    CPColor(0.595839f, 0.848717f, 0.243329f),
                    CPColor(0.606045f, 0.850733f, 0.236712f),
                    CPColor(0.616293f, 0.852709f, 0.230052f),
                    CPColor(0.626579f, 0.854645f, 0.223353f),
                    CPColor(0.636902f, 0.856542f, 0.21662f),
                    CPColor(0.647257f, 0.8584f, 0.209861f),
                    CPColor(0.657642f, 0.860219f, 0.203082f),
                    CPColor(0.668054f, 0.861999f, 0.196293f),
                    CPColor(0.678489f, 0.863742f, 0.189503f),
                    CPColor(0.688944f, 0.865448f, 0.182725f),
                    CPColor(0.699415f, 0.867117f, 0.175971f),
                    CPColor(0.709898f, 0.868751f, 0.169257f),
                    CPColor(0.720391f, 0.87035f, 0.162603f),
                    CPColor(0.730889f, 0.871916f, 0.156029f),
                    CPColor(0.741388f, 0.873449f, 0.149561f),
                    CPColor(0.751884f, 0.874951f, 0.143228f),
                    CPColor(0.762373f, 0.876424f, 0.137064f),
                    CPColor(0.772852f, 0.877868f, 0.131109f),
                    CPColor(0.783315f, 0.879285f, 0.125405f),
                    CPColor(0.79376f, 0.880678f, 0.120005f),
                    CPColor(0.804182f, 0.882046f, 0.114965f),
                    CPColor(0.814576f, 0.883393f, 0.110347f),
                    CPColor(0.82494f, 0.88472f, 0.106217f),
                    CPColor(0.83527f, 0.886029f, 0.102646f),
                    CPColor(0.845561f, 0.887322f, 0.099702f),
                    CPColor(0.85581f, 0.888601f, 0.097452f),
                    CPColor(0.866013f, 0.889868f, 0.095953f),
                    CPColor(0.876168f, 0.891125f, 0.09525f),
                    CPColor(0.886271f, 0.892374f, 0.095374f),
                    CPColor(0.89632f, 0.893616f, 0.096335f),
                    CPColor(0.906311f, 0.894855f, 0.098125f),
                    CPColor(0.916242f, 0.896091f, 0.100717f),
                    CPColor(0.926106f, 0.89733f, 0.104071f),
                    CPColor(0.935904f, 0.89857f, 0.108131f),
                    CPColor(0.945636f, 0.899815f, 0.112838f),
                    CPColor(0.9553f, 0.901065f, 0.118128f),
                    CPColor(0.964894f, 0.902323f, 0.123941f),
                    CPColor(0.974417f, 0.90359f, 0.130215f),
                    CPColor(0.983868f, 0.904867f, 0.136897f),
                    CPColor(0.993248f, 0.906157f, 0.143936f)
                )
            )
        )
        cividis = Entry(
            "cividis", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    CPColor(0.0f, 0.135112f, 0.304751f),
                    CPColor(0.0f, 0.138068f, 0.311105f),
                    CPColor(0.0f, 0.141013f, 0.317579f),
                    CPColor(0.0f, 0.143951f, 0.323982f),
                    CPColor(0.0f, 0.146877f, 0.330479f),
                    CPColor(0.0f, 0.149791f, 0.337065f),
                    CPColor(0.0f, 0.152673f, 0.343704f),
                    CPColor(0.0f, 0.155377f, 0.3505f),
                    CPColor(0.0f, 0.157932f, 0.357521f),
                    CPColor(0.0f, 0.160495f, 0.364534f),
                    CPColor(0.0f, 0.163058f, 0.371608f),
                    CPColor(0.0f, 0.165621f, 0.378769f),
                    CPColor(0.0f, 0.168204f, 0.385902f),
                    CPColor(0.0f, 0.1708f, 0.3931f),
                    CPColor(0.0f, 0.17342f, 0.400353f),
                    CPColor(0.0f, 0.176082f, 0.407577f),
                    CPColor(0.0f, 0.178802f, 0.414764f),
                    CPColor(0.0f, 0.18161f, 0.421859f),
                    CPColor(0.0f, 0.18455f, 0.428802f),
                    CPColor(0.0f, 0.186915f, 0.435532f),
                    CPColor(0.0f, 0.188769f, 0.439563f),
                    CPColor(0.0f, 0.19095f, 0.441085f),
                    CPColor(0.0f, 0.193366f, 0.441561f),
                    CPColor(0.003602f, 0.195911f, 0.441564f),
                    CPColor(0.017852f, 0.198528f, 0.441248f),
                    CPColor(0.03211f, 0.201199f, 0.440785f),
                    CPColor(0.046205f, 0.203903f, 0.440196f),
                    CPColor(0.058378f, 0.206629f, 0.439531f),
                    CPColor(0.068968f, 0.209372f, 0.438863f),
                    CPColor(0.078624f, 0.212122f, 0.438105f),
                    CPColor(0.087465f, 0.214879f, 0.437342f),
                    CPColor(0.095645f, 0.217643f, 0.436593f),
                    CPColor(0.103401f, 0.220406f, 0.43579f),
                    CPColor(0.110658f, 0.22317f, 0.435067f),
                    CPColor(0.117612f, 0.225935f, 0.434308f),
                    CPColor(0.124291f, 0.228697f, 0.433547f),
                    CPColor(0.130669f, 0.231458f, 0.43284f),
                    CPColor(0.13683f, 0.234216f, 0.432148f),
                    CPColor(0.142852f, 0.236972f, 0.431404f),
                    CPColor(0.148638f, 0.239724f, 0.430752f),
                    CPColor(0.154261f, 0.242475f, 0.43012f),
                    CPColor(0.159733f, 0.245221f, 0.429528f),
                    CPColor(0.165113f, 0.247965f, 0.428908f),
                    CPColor(0.170362f, 0.250707f, 0.428325f),
                    CPColor(0.17549f, 0.253444f, 0.42779f),
                    CPColor(0.180503f, 0.25618f, 0.427299f),
                    CPColor(0.185453f, 0.258914f, 0.426788f),
                    CPColor(0.190303f, 0.261644f, 0.426329f),
                    CPColor(0.195057f, 0.264372f, 0.425924f),
                    CPColor(0.199764f, 0.267099f, 0.425497f),
                    CPColor(0.204385f, 0.269823f, 0.425126f),
                    CPColor(0.208926f, 0.272546f, 0.424809f),
                    CPColor(0.213431f, 0.275266f, 0.42448f),
                    CPColor(0.217863f, 0.277985f, 0.424206f),
                    CPColor(0.222264f, 0.280702f, 0.423914f),
                    CPColor(0.226598f, 0.283419f, 0.423678f),
                    CPColor(0.230871f, 0.286134f, 0.423498f),
                    CPColor(0.23512f, 0.288848f, 0.423304f),
                    CPColor(0.239312f, 0.291562f, 0.423167f),
                    CPColor(0.243485f, 0.294274f, 0.423014f),
                    CPColor(0.247605f, 0.296986f, 0.422917f),
                    CPColor(0.251675f, 0.299698f, 0.422873f),
                    CPColor(0.255731f, 0.302409f, 0.422814f),
                    CPColor(0.25974f, 0.30512f, 0.42281f),
                    CPColor(0.263738f, 0.307831f, 0.422789f),
                    CPColor(0.267693f, 0.310542f, 0.422821f),
                    CPColor(0.271639f, 0.313253f, 0.422837f),
                    CPColor(0.275513f, 0.315965f, 0.422979f),
                    CPColor(0.279411f, 0.318677f, 0.423031f),
                    CPColor(0.28324f, 0.32139f, 0.423211f),
                    CPColor(0.287065f, 0.324103f, 0.423373f),
                    CPColor(0.290884f, 0.326816f, 0.423517f),
                    CPColor(0.294669f, 0.329531f, 0.423716f),
                    CPColor(0.298421f, 0.332247f, 0.423973f),
                    CPColor(0.302169f, 0.334963f, 0.424213f),
                    CPColor(0.305886f, 0.337681f, 0.424512f),
                    CPColor(0.309601f, 0.340399f, 0.42479f),
                    CPColor(0.313287f, 0.34312f, 0.42512f),
                    CPColor(0.316941f, 0.345842f, 0.425512f),
                    CPColor(0.320595f, 0.348565f, 0.425889f),
                    CPColor(0.32425f, 0.351289f, 0.42625f),
                    CPColor(0.327875f, 0.354016f, 0.42667f),
                    CPColor(0.331474f, 0.356744f, 0.427144f),
                    CPColor(0.335073f, 0.359474f, 0.427605f),
                    CPColor(0.338673f, 0.362206f, 0.428053f),
                    CPColor(0.342246f, 0.364939f, 0.428559f),
                    CPColor(0.345793f, 0.367676f, 0.429127f),
                    CPColor(0.349341f, 0.370414f, 0.429685f),
                    CPColor(0.352892f, 0.373153f, 0.430226f),
                    CPColor(0.356418f, 0.375896f, 0.430823f),
                    CPColor(0.359916f, 0.378641f, 0.431501f),
                    CPColor(0.363446f, 0.381388f, 0.432075f),
                    CPColor(0.366923f, 0.384139f, 0.432796f),
                    CPColor(0.37043f, 0.38689f, 0.433428f),
                    CPColor(0.373884f, 0.389646f, 0.434209f),
                    CPColor(0.377371f, 0.392404f, 0.43489f),
                    CPColor(0.38083f, 0.395164f, 0.435653f),
                    CPColor(0.384268f, 0.397928f, 0.436475f),
                    CPColor(0.387705f, 0.400694f, 0.437305f),
                    CPColor(0.391151f, 0.403464f, 0.438096f),
                    CPColor(0.394568f, 0.406236f, 0.438986f),
                    CPColor(0.397991f, 0.409011f, 0.439848f),
                    CPColor(0.401418f, 0.41179f, 0.440708f),
                    CPColor(0.40482f, 0.414572f, 0.441642f),
                    CPColor(0.408226f, 0.417357f, 0.44257f),
                    CPColor(0.411607f, 0.420145f, 0.443577f),
                    CPColor(0.414992f, 0.422937f, 0.444578f),
                    CPColor(0.418383f, 0.425733f, 0.44556f),
                    CPColor(0.421748f, 0.428531f, 0.44664f),
                    CPColor(0.42512f, 0.431334f, 0.447692f),
                    CPColor(0.428462f, 0.43414f, 0.448864f),
                    CPColor(0.431817f, 0.43695f, 0.449982f),
                    CPColor(0.435168f, 0.439763f, 0.451134f),
                    CPColor(0.438504f, 0.44258f, 0.452341f),
                    CPColor(0.44181f, 0.445402f, 0.453659f),
                    CPColor(0.445148f, 0.448226f, 0.454885f),
                    CPColor(0.448447f, 0.451053f, 0.456264f),
                    CPColor(0.451759f, 0.453887f, 0.457582f),
                    CPColor(0.455072f, 0.456718f, 0.458976f),
                    CPColor(0.458366f, 0.459552f, 0.460457f),
                    CPColor(0.461616f, 0.462405f, 0.461969f),
                    CPColor(0.464947f, 0.465241f, 0.463395f),
                    CPColor(0.468254f, 0.468083f, 0.464908f),
                    CPColor(0.471501f, 0.47096f, 0.466357f),
                    CPColor(0.474812f, 0.473832f, 0.467681f),
                    CPColor(0.478186f, 0.476699f, 0.468845f),
                    CPColor(0.481622f, 0.479573f, 0.469767f),
                    CPColor(0.485141f, 0.482451f, 0.470384f),
                    CPColor(0.488697f, 0.485318f, 0.471008f),
                    CPColor(0.492278f, 0.488198f, 0.471453f),
                    CPColor(0.495913f, 0.491076f, 0.471751f),
                    CPColor(0.499552f, 0.49396f, 0.472032f),
                    CPColor(0.503185f, 0.496851f, 0.472305f),
                    CPColor(0.506866f, 0.499743f, 0.472432f),
                    CPColor(0.51054f, 0.502643f, 0.47255f),
                    CPColor(0.514226f, 0.505546f, 0.47264f),
                    CPColor(0.51792f, 0.508454f, 0.472707f),
                    CPColor(0.521643f, 0.511367f, 0.472639f),
                    CPColor(0.525348f, 0.514285f, 0.47266f),
                    CPColor(0.529086f, 0.517207f, 0.472543f),
                    CPColor(0.532829f, 0.520135f, 0.472401f),
                    CPColor(0.536553f, 0.523067f, 0.472352f),
                    CPColor(0.540307f, 0.526005f, 0.472163f),
                    CPColor(0.544069f, 0.528948f, 0.471947f),
                    CPColor(0.54784f, 0.531895f, 0.471704f),
                    CPColor(0.551612f, 0.534849f, 0.471439f),
                    CPColor(0.555393f, 0.537807f, 0.471147f),
                    CPColor(0.559181f, 0.540771f, 0.470829f),
                    CPColor(0.562972f, 0.543741f, 0.470488f),
                    CPColor(0.566802f, 0.546715f, 0.469988f),
                    CPColor(0.570607f, 0.549695f, 0.469593f),
                    CPColor(0.574417f, 0.552682f, 0.469172f),
                    CPColor(0.578236f, 0.555673f, 0.468724f),
                    CPColor(0.582087f, 0.55867f, 0.468118f),
                    CPColor(0.585916f, 0.561674f, 0.467618f),
                    CPColor(0.589753f, 0.564682f, 0.46709f),
                    CPColor(0.593622f, 0.567697f, 0.466401f),
                    CPColor(0.597469f, 0.570718f, 0.465821f),
                    CPColor(0.601354f, 0.573743f, 0.465074f),
                    CPColor(0.605211f, 0.576777f, 0.464441f),
                    CPColor(0.609105f, 0.579816f, 0.463638f),
                    CPColor(0.612977f, 0.582861f, 0.46295f),
                    CPColor(0.616852f, 0.585913f, 0.462237f),
                    CPColor(0.620765f, 0.58897f, 0.461351f),
                    CPColor(0.624654f, 0.592034f, 0.460583f),
                    CPColor(0.628576f, 0.595104f, 0.459641f),
                    CPColor(0.632506f, 0.59818f, 0.458668f),
                    CPColor(0.636412f, 0.601264f, 0.457818f),
                    CPColor(0.640352f, 0.604354f, 0.456791f),
                    CPColor(0.64427f, 0.60745f, 0.455886f),
                    CPColor(0.648222f, 0.610553f, 0.454801f),
                    CPColor(0.652178f, 0.613664f, 0.453689f),
                    CPColor(0.656114f, 0.61678f, 0.452702f),
                    CPColor(0.660082f, 0.619904f, 0.451534f),
                    CPColor(0.664055f, 0.623034f, 0.450338f),
                    CPColor(0.668008f, 0.626171f, 0.44927f),
                    CPColor(0.671991f, 0.629316f, 0.448018f),
                    CPColor(0.675981f, 0.632468f, 0.446736f),
                    CPColor(0.679979f, 0.635626f, 0.445424f),
                    CPColor(0.68395f, 0.638793f, 0.444251f),
                    CPColor(0.687957f, 0.641966f, 0.442886f),
                    CPColor(0.691971f, 0.645145f, 0.441491f),
                    CPColor(0.695985f, 0.648334f, 0.440072f),
                    CPColor(0.700008f, 0.651529f, 0.438624f),
                    CPColor(0.704037f, 0.654731f, 0.437147f),
                    CPColor(0.708067f, 0.657942f, 0.435647f),
                    CPColor(0.712105f, 0.66116f, 0.434117f),
                    CPColor(0.716177f, 0.664384f, 0.432386f),
                    CPColor(0.720222f, 0.667618f, 0.430805f),
                    CPColor(0.724274f, 0.670859f, 0.429194f),
                    CPColor(0.728334f, 0.674107f, 0.427554f),
                    CPColor(0.732422f, 0.677364f, 0.425717f),
                    CPColor(0.736488f, 0.680629f, 0.424028f),
                    CPColor(0.740589f, 0.6839f, 0.422131f),
                    CPColor(0.744664f, 0.687181f, 0.420393f),
                    CPColor(0.748772f, 0.69047f, 0.418448f),
                    CPColor(0.752886f, 0.693766f, 0.416472f),
                    CPColor(0.756975f, 0.697071f, 0.414659f),
                    CPColor(0.761096f, 0.700384f, 0.412638f),
                    CPColor(0.765223f, 0.703705f, 0.410587f),
                    CPColor(0.769353f, 0.707035f, 0.408516f),
                    CPColor(0.773486f, 0.710373f, 0.406422f),
                    CPColor(0.777651f, 0.713719f, 0.404112f),
                    CPColor(0.781795f, 0.717074f, 0.401966f),
                    CPColor(0.785965f, 0.720438f, 0.399613f),
                    CPColor(0.790116f, 0.72381f, 0.397423f),
                    CPColor(0.794298f, 0.72719f, 0.395016f),
                    CPColor(0.79848f, 0.73058f, 0.392597f),
                    CPColor(0.802667f, 0.733978f, 0.390153f),
                    CPColor(0.806859f, 0.737385f, 0.387684f),
                    CPColor(0.811054f, 0.740801f, 0.385198f),
                    CPColor(0.815274f, 0.744226f, 0.382504f),
                    CPColor(0.819499f, 0.747659f, 0.379785f),
                    CPColor(0.823729f, 0.751101f, 0.377043f),
                    CPColor(0.827959f, 0.754553f, 0.374292f),
                    CPColor(0.832192f, 0.758014f, 0.371529f),
                    CPColor(0.836429f, 0.761483f, 0.368747f),
                    CPColor(0.840693f, 0.764962f, 0.365746f),
                    CPColor(0.844957f, 0.76845f, 0.362741f),
                    CPColor(0.849223f, 0.771947f, 0.359729f),
                    CPColor(0.853515f, 0.775454f, 0.3565f),
                    CPColor(0.857809f, 0.778969f, 0.353259f),
                    CPColor(0.862105f, 0.782494f, 0.350011f),
                    CPColor(0.866421f, 0.786028f, 0.346571f),
                    CPColor(0.870717f, 0.789572f, 0.343333f),
                    CPColor(0.875057f, 0.793125f, 0.339685f),
                    CPColor(0.879378f, 0.796687f, 0.336241f),
                    CPColor(0.88372f, 0.800258f, 0.332599f),
                    CPColor(0.888081f, 0.803839f, 0.32877f),
                    CPColor(0.89244f, 0.80743f, 0.324968f),
                    CPColor(0.896818f, 0.81103f, 0.320982f),
                    CPColor(0.901195f, 0.814639f, 0.317021f),
                    CPColor(0.905589f, 0.818257f, 0.312889f),
                    CPColor(0.91f, 0.821885f, 0.308594f),
                    CPColor(0.914407f, 0.825522f, 0.304348f),
                    CPColor(0.918828f, 0.829168f, 0.29996f),
                    CPColor(0.923279f, 0.832822f, 0.295244f),
                    CPColor(0.927724f, 0.836486f, 0.290611f),
                    CPColor(0.93218f, 0.840159f, 0.28588f),
                    CPColor(0.93666f, 0.843841f, 0.280876f),
                    CPColor(0.941147f, 0.84753f, 0.275815f),
                    CPColor(0.945654f, 0.851228f, 0.270532f),
                    CPColor(0.950178f, 0.854933f, 0.265085f),
                    CPColor(0.954725f, 0.858646f, 0.259365f),
                    CPColor(0.959284f, 0.862365f, 0.253563f),
                    CPColor(0.963872f, 0.866089f, 0.247445f),
                    CPColor(0.968469f, 0.869819f, 0.24131f),
                    CPColor(0.973114f, 0.87355f, 0.234677f),
                    CPColor(0.97778f, 0.877281f, 0.227954f),
                    CPColor(0.982497f, 0.881008f, 0.220878f),
                    CPColor(0.987293f, 0.884718f, 0.213336f),
                    CPColor(0.992218f, 0.888385f, 0.205468f),
                    CPColor(0.994847f, 0.892954f, 0.203445f),
                    CPColor(0.995249f, 0.898384f, 0.207561f),
                    CPColor(0.995503f, 0.903866f, 0.21237f),
                    CPColor(0.995737f, 0.909344f, 0.217772f)
                )
            )
        )
        twilight = Entry(
            "twilight", Type.DIVERGING, InterpolatedPalette(
                FixedPalette(
                    CPColor(0.8857501584075443f, 0.8500092494306783f, 0.8879736506427196f),
                    CPColor(0.8837852019553906f, 0.8507294054031063f, 0.8872322209694989f),
                    CPColor(0.8817223105928579f, 0.8512759407765347f, 0.8863805692551482f),
                    CPColor(0.8795410528270573f, 0.8516567540749572f, 0.8854143767924102f),
                    CPColor(0.8772488085896548f, 0.8518702833887027f, 0.8843412038131143f),
                    CPColor(0.8748534750857597f, 0.8519152612302319f, 0.8831692696761383f),
                    CPColor(0.8723313408512408f, 0.8518016547808089f, 0.8818970435500162f),
                    CPColor(0.8697047485350982f, 0.8515240300479789f, 0.8805388339000336f),
                    CPColor(0.8669601550533358f, 0.8510896085314068f, 0.8790976697717334f),
                    CPColor(0.86408985081464f, 0.8505039116750779f, 0.8775792578489263f),
                    CPColor(0.8611024543689985f, 0.8497675485700126f, 0.8759924292343957f),
                    CPColor(0.8579825924567037f, 0.8488893481028184f, 0.8743403855344628f),
                    CPColor(0.8547259318925698f, 0.8478748812467282f, 0.8726282980930582f),
                    CPColor(0.8513371457085719f, 0.8467273579611647f, 0.8708608165735044f),
                    CPColor(0.8478071070257792f, 0.8454546229209523f, 0.8690403678369444f),
                    CPColor(0.8441261828674842f, 0.8440648271103739f, 0.8671697332269007f),
                    CPColor(0.8403042080595778f, 0.8425605950855084f, 0.865250882410458f),
                    CPColor(0.8363403180919118f, 0.8409479651895194f, 0.8632852800107016f),
                    CPColor(0.8322270571293441f, 0.8392349062775448f, 0.8612756350042788f),
                    CPColor(0.8279689431601354f, 0.837426007513952f, 0.8592239945130679f),
                    CPColor(0.8235742968025285f, 0.8355248776479544f, 0.8571319132851495f),
                    CPColor(0.8190465467793753f, 0.8335364929949034f, 0.855002062870101f),
                    CPColor(0.8143898212114309f, 0.8314655869419785f, 0.8528375906214702f),
                    CPColor(0.8095999819094809f, 0.8293189667350546f, 0.8506444160105037f),
                    CPColor(0.8046916442981458f, 0.8270983878056066f, 0.8484244929697402f),
                    CPColor(0.79967075421268f, 0.8248078181208093f, 0.8461821002957853f),
                    CPColor(0.7945430508923111f, 0.8224511622630462f, 0.8439218478682798f),
                    CPColor(0.7893144556460892f, 0.8200321318870201f, 0.8416486380471222f),
                    CPColor(0.7839910104276492f, 0.8175542640053343f, 0.8393674746403673f),
                    CPColor(0.7785789200822759f, 0.8150208937874255f, 0.8370834463093898f),
                    CPColor(0.7730841659017094f, 0.8124352473546601f, 0.8348017295057968f),
                    CPColor(0.7675110850441786f, 0.8098007598713145f, 0.8325281663805967f),
                    CPColor(0.7618690793798029f, 0.8071194938764749f, 0.830266486168872f),
                    CPColor(0.7561644358438198f, 0.8043940873347794f, 0.8280213899472f),
                    CPColor(0.750403467654067f, 0.8016269900896532f, 0.8257973785108242f),
                    CPColor(0.7445924777189017f, 0.7988204771958325f, 0.8235986758615652f),
                    CPColor(0.7387377170049494f, 0.7959766573503101f, 0.8214292278043301f),
                    CPColor(0.7328454364552346f, 0.7930974646884407f, 0.8192926338423038f),
                    CPColor(0.726921775128297f, 0.7901846863592763f, 0.8171921746672638f),
                    CPColor(0.7209728066553678f, 0.7872399592345264f, 0.8151307392087926f),
                    CPColor(0.7150040307625213f, 0.7842648709158119f, 0.8131111655994991f),
                    CPColor(0.709020781345393f, 0.7812608871607091f, 0.8111359185511793f),
                    CPColor(0.7030297722540817f, 0.7782290497335813f, 0.8092061884805697f),
                    CPColor(0.6970365443886174f, 0.7751705000806606f, 0.8073233538006345f),
                    CPColor(0.691046410093091f, 0.7720862946067809f, 0.8054884169067907f),
                    CPColor(0.6850644615439593f, 0.7689774029354699f, 0.8037020626717691f),
                    CPColor(0.6790955449988215f, 0.765844721313959f, 0.8019646617300199f),
                    CPColor(0.6731442255942621f, 0.7626890873389048f, 0.8002762854580953f),
                    CPColor(0.6672147980375281f, 0.7595112803730375f, 0.7986367465453776f),
                    CPColor(0.6613112930078745f, 0.7563120270871903f, 0.7970456043491897f),
                    CPColor(0.6554369232645472f, 0.7530920875676843f, 0.7955027112903105f),
                    CPColor(0.6495957300425348f, 0.7498520122194177f, 0.7940067402149911f),
                    CPColor(0.6437910831099849f, 0.7465923800833657f, 0.7925565320130605f),
                    CPColor(0.6380258682854598f, 0.7433137671403319f, 0.7911510045957317f),
                    CPColor(0.6323027138710603f, 0.740016721601314f, 0.7897889276264043f),
                    CPColor(0.6266240202260459f, 0.7367017540369944f, 0.7884690131633456f),
                    CPColor(0.6209919306481755f, 0.733369347989232f, 0.7871899462469658f),
                    CPColor(0.6154084641177048f, 0.7300199523273969f, 0.7859502270675048f),
                    CPColor(0.6098754317609306f, 0.7266539875975829f, 0.7847483573269471f),
                    CPColor(0.6043943420027486f, 0.7232718614323369f, 0.7835829559353559f),
                    CPColor(0.5989665814482068f, 0.7198739489224673f, 0.7824525989934664f),
                    CPColor(0.5935933569683722f, 0.7164606049658685f, 0.781355882376401f),
                    CPColor(0.588275797805555f, 0.7130321464645814f, 0.7802914140563652f),
                    CPColor(0.5830148703693241f, 0.7095888767699747f, 0.7792578182047659f),
                    CPColor(0.5778116438998202f, 0.7061310615715398f, 0.7782534512102552f),
                    CPColor(0.5726668948158774f, 0.7026589535425779f, 0.7772770268091199f),
                    CPColor(0.5675811785386197f, 0.6991727930264627f, 0.776327485342753f),
                    CPColor(0.5625551535721934f, 0.6956727838162965f, 0.7754035914230984f),
                    CPColor(0.5575894041960517f, 0.6921591145825405f, 0.7745041337932782f),
                    CPColor(0.5526845058934713f, 0.6886319451516638f, 0.7736279426902245f),
                    CPColor(0.5478409815301863f, 0.6850914221850988f, 0.7727738647344087f),
                    CPColor(0.5430593242401823f, 0.6815376725306588f, 0.7719407969783508f),
                    CPColor(0.5383401557517628f, 0.677970811290954f, 0.7711273443905772f),
                    CPColor(0.533683891477284f, 0.6743909370521273f, 0.7703325054879735f),
                    CPColor(0.529090861832473f, 0.6707981230280622f, 0.7695555229231313f),
                    CPColor(0.5245615147059358f, 0.6671924299614223f, 0.7687954171423095f),
                    CPColor(0.5200962739223556f, 0.6635739143403039f, 0.768051194033441f),
                    CPColor(0.5156955988596057f, 0.65994260812898f, 0.7673219148959617f),
                    CPColor(0.5113599254160193f, 0.6562985398183186f, 0.7666066378064533f),
                    CPColor(0.5070896957645166f, 0.6526417240314645f, 0.7659044566083585f),
                    CPColor(0.5028853540415561f, 0.6489721673409526f, 0.7652144671817491f),
                    CPColor(0.4987473366135607f, 0.6452898684900934f, 0.7645357873418008f),
                    CPColor(0.4946761847863938f, 0.6415948411950443f, 0.7638671900213091f),
                    CPColor(0.4906722493856122f, 0.6378870485884708f, 0.7632081276316384f),
                    CPColor(0.4867359599430568f, 0.6341664625110051f, 0.7625578008592404f),
                    CPColor(0.4828677867260272f, 0.6304330455306234f, 0.761915371498953f),
                    CPColor(0.47906816236197386f, 0.6266867625186013f, 0.7612800037566242f),
                    CPColor(0.47533752394906287f, 0.6229275728383581f, 0.7606508557181775f),
                    CPColor(0.4716762951887709f, 0.6191554324288464f, 0.7600270922788305f),
                    CPColor(0.46808490970531597f, 0.6153702869579029f, 0.7594078989109274f),
                    CPColor(0.4645637671630393f, 0.6115720882286415f, 0.7587924262302581f),
                    CPColor(0.4611132664702388f, 0.607760777169989f, 0.7581798643680714f),
                    CPColor(0.45773377230160567f, 0.6039363004658646f, 0.7575693690185916f),
                    CPColor(0.45442563977552913f, 0.6000985950385866f, 0.7569601366060649f),
                    CPColor(0.45118918687617743f, 0.5962476205135354f, 0.7563512064324664f),
                    CPColor(0.4480247093358917f, 0.5923833145214658f, 0.7557417647410792f),
                    CPColor(0.4449324685421538f, 0.5885055998308617f, 0.7551311041857901f),
                    CPColor(0.441912717666964f, 0.5846144110017557f, 0.7545183888441067f),
                    CPColor(0.43896563958048396f, 0.5807096924109849f, 0.7539027620828594f),
                    CPColor(0.4360913895835637f, 0.5767913799818608f, 0.7532834105961016f),
                    CPColor(0.43329008867358393f, 0.5728594162560667f, 0.7526594653256667f),
                    CPColor(0.4305617907305757f, 0.5689137457245718f, 0.752030080993127f),
                    CPColor(0.42790652284925834f, 0.5649543060909209f, 0.7513944352191484f),
                    CPColor(0.42532423665011354f, 0.560981049599503f, 0.7507516498900512f),
                    CPColor(0.4228148567577266f, 0.5569939212699658f, 0.7501008698822764f),
                    CPColor(0.42037822361396326f, 0.5529928715810817f, 0.7494412559451894f),
                    CPColor(0.4180141407923363f, 0.5489778542188889f, 0.7487719316700112f),
                    CPColor(0.4157223260454232f, 0.544948827153504f, 0.7480920445900052f),
                    CPColor(0.4135024574331473f, 0.5409057477109848f, 0.7474007329754309f),
                    CPColor(0.4113541469730457f, 0.5368485776500593f, 0.7466971285506578f),
                    CPColor(0.4092768899914751f, 0.5327773017713032f, 0.7459803063570782f),
                    CPColor(0.4072701869421907f, 0.5286918801105741f, 0.7452494263758127f),
                    CPColor(0.4053334378930318f, 0.5245922817498312f, 0.7445036583670813f),
                    CPColor(0.40346600333905397f, 0.5204784765384003f, 0.7437421522356709f),
                    CPColor(0.40166714010896104f, 0.5163504496968876f, 0.7429640345324835f),
                    CPColor(0.39993606933454834f, 0.5122081814321852f, 0.7421684457131799f),
                    CPColor(0.3982719152586337f, 0.5080516653927614f, 0.7413545091809972f),
                    CPColor(0.3966737490566561f, 0.5038808905384797f, 0.7405213858051674f),
                    CPColor(0.3951405880820763f, 0.4996958532637776f, 0.7396682021171571f),
                    CPColor(0.39367135736822567f, 0.4954965577745118f, 0.738794102296364f),
                    CPColor(0.39226494876209317f, 0.4912830033289926f, 0.7378982478447508f),
                    CPColor(0.390920175719949f, 0.4870552025122304f, 0.7369797713388125f),
                    CPColor(0.38963580160340855f, 0.48281316715123496f, 0.7360378254693274f),
                    CPColor(0.3884105330084243f, 0.47855691131792805f, 0.7350715764115726f),
                    CPColor(0.3872430145933025f, 0.4742864593363539f, 0.7340801678785439f),
                    CPColor(0.386131841788921f, 0.4700018340988123f, 0.7330627749243106f),
                    CPColor(0.3850755679365139f, 0.46570306719930193f, 0.732018540336905f),
                    CPColor(0.38407269378943537f, 0.46139018782416635f, 0.7309466543290268f),
                    CPColor(0.3831216808440275f, 0.457063235814072f, 0.7298462679135326f),
                    CPColor(0.38222094988570376f, 0.45272225034283325f, 0.7287165614400378f),
                    CPColor(0.3813688793045416f, 0.4483672766927786f, 0.7275567131714135f),
                    CPColor(0.3805638069656562f, 0.4439983720863372f, 0.7263658704513531f),
                    CPColor(0.3798040374484875f, 0.4396155882122263f, 0.7251432377876109f),
                    CPColor(0.3790878928311076f, 0.43521897612544935f, 0.7238879869132313f),
                    CPColor(0.378413635091359f, 0.43080859411413064f, 0.7225993199306104f),
                    CPColor(0.3777794975351373f, 0.4263845142616835f, 0.7212763999353023f),
                    CPColor(0.3771837184425123f, 0.4219468022345483f, 0.7199184152447577f),
                    CPColor(0.37662448930806297f, 0.41749553747893614f, 0.7185245473617611f),
                    CPColor(0.37610001286385814f, 0.4130307995247706f, 0.7170939691992023f),
                    CPColor(0.375608469194424f, 0.40855267638072096f, 0.7156258509158755f),
                    CPColor(0.37514802505380473f, 0.4040612609993941f, 0.7141193695725726f),
                    CPColor(0.3747168601930223f, 0.3995566498711684f, 0.7125736851650046f),
                    CPColor(0.3743131319931234f, 0.3950389482828331f, 0.7109879652237746f),
                    CPColor(0.3739349933047578f, 0.3905082752937583f, 0.7093613429347845f),
                    CPColor(0.3735806215098284f, 0.3859647438605754f, 0.7076929760731058f),
                    CPColor(0.37324816143326384f, 0.38140848555753937f, 0.7059820097480604f),
                    CPColor(0.3729357864666503f, 0.3768396383521984f, 0.7042275578058994f),
                    CPColor(0.37264166757849604f, 0.3722583500483685f, 0.7024287314570723f),
                    CPColor(0.37236397858465387f, 0.36766477862108266f, 0.7005846349652077f),
                    CPColor(0.3721008970244382f, 0.3630590973698238f, 0.6986943461507372f),
                    CPColor(0.3718506155898596f, 0.3584414828587522f, 0.6967569581025654f),
                    CPColor(0.3716113323440048f, 0.3538121372967869f, 0.6947714991938089f),
                    CPColor(0.37138124223736607f, 0.34917126878479027f, 0.6927370347192883f),
                    CPColor(0.37115856636209105f, 0.3445191141023017f, 0.6906525358646499f),
                    CPColor(0.3709415155133733f, 0.33985591488818123f, 0.6885170337950512f),
                    CPColor(0.3707283327942267f, 0.33518193808489577f, 0.6863294816960677f),
                    CPColor(0.37051738634484427f, 0.3304974124430785f, 0.6840888878885721f),
                    CPColor(0.37030682071842685f, 0.32580269697872455f, 0.6817941168448668f),
                    CPColor(0.37009487130772695f, 0.3210981375964933f, 0.6794440539905685f),
                    CPColor(0.3698798032902536f, 0.31638410101153364f, 0.6770375543809057f),
                    CPColor(0.36965987626565955f, 0.3116609876295197f, 0.6745734474341955f),
                    CPColor(0.3694333459127623f, 0.3069292355186234f, 0.6720505284912062f),
                    CPColor(0.36919847837592484f, 0.3021893217650707f, 0.6694675433161452f),
                    CPColor(0.3689535530659678f, 0.29744175492366276f, 0.6668232208982426f),
                    CPColor(0.3686968223189527f, 0.292687098561501f, 0.6641162529823691f),
                    CPColor(0.36842655638020444f, 0.2879259643777846f, 0.661345269109446f),
                    CPColor(0.3681410147989972f, 0.2831590122118299f, 0.6585088880697231f),
                    CPColor(0.3678384369653108f, 0.2783869718129776f, 0.655605668384537f),
                    CPColor(0.36751707094367697f, 0.2736106331709098f, 0.6526341171161864f),
                    CPColor(0.36717513650699446f, 0.26883085667326956f, 0.6495927229789225f),
                    CPColor(0.3668108554010799f, 0.26404857724525643f, 0.6464799165290824f),
                    CPColor(0.3664224325155063f, 0.25926481158628106f, 0.6432940914076554f),
                    CPColor(0.36600853966739794f, 0.25448043878086224f, 0.6400336180336859f),
                    CPColor(0.3655669837353898f, 0.24969683475296395f, 0.6366967518748858f),
                    CPColor(0.3650957984588681f, 0.24491536803550484f, 0.6332817352005559f),
                    CPColor(0.3645930889012501f, 0.24013747024823828f, 0.629786801550261f),
                    CPColor(0.3640569302208851f, 0.23536470386204195f, 0.6262101345195302f),
                    CPColor(0.36348537610385145f, 0.2305987621839642f, 0.6225498862239288f),
                    CPColor(0.3628764356004103f, 0.2258414929328703f, 0.6188041741082302f),
                    CPColor(0.36222809558295926f, 0.22109488427338303f, 0.6149711234609613f),
                    CPColor(0.36153829010998356f, 0.21636111429594002f, 0.6110488067964093f),
                    CPColor(0.36080493826624654f, 0.21164251793458128f, 0.6070353217206471f),
                    CPColor(0.36002681809096376f, 0.20694122817889948f, 0.6029284543191687f),
                    CPColor(0.35920088560930186f, 0.20226037920758122f, 0.5987265295935138f),
                    CPColor(0.3583248996661781f, 0.197602942459778f, 0.5944276851750107f),
                    CPColor(0.35739663292915563f, 0.1929720819784246f, 0.5900301125106313f),
                    CPColor(0.35641381143126327f, 0.18837119869242164f, 0.5855320765920552f),
                    CPColor(0.3553741530690672f, 0.18380392577704466f, 0.580931914318328f),
                    CPColor(0.3542753496066376f, 0.17927413271618647f, 0.5762280966066872f),
                    CPColor(0.35311574421123737f, 0.17478570377561287f, 0.5714187152355529f),
                    CPColor(0.3518924860887379f, 0.17034320478524959f, 0.5665028491121665f),
                    CPColor(0.3506030444193101f, 0.1659512998472086f, 0.5614796470399323f),
                    CPColor(0.34924513554955644f, 0.16161477763045118f, 0.5563483747416378f),
                    CPColor(0.3478165323877778f, 0.1573386351115298f, 0.5511085345270326f),
                    CPColor(0.3463150717579309f, 0.15312802296627787f, 0.5457599924248665f),
                    CPColor(0.34473901574536375f, 0.1489882058982641f, 0.5403024592040654f),
                    CPColor(0.34308600291572294f, 0.14492465359918028f, 0.534737042820671f),
                    CPColor(0.34135411074506483f, 0.1409427920655632f, 0.5290650094033675f),
                    CPColor(0.33954168752669694f, 0.1370480189671817f, 0.5232879753508524f),
                    CPColor(0.3376473209067111f, 0.13324562282438077f, 0.5174080757397947f),
                    CPColor(0.33566978565015315f, 0.12954074251271822f, 0.5114280721516895f),
                    CPColor(0.33360804901486f, 0.1259381830100592f, 0.505351647966549f),
                    CPColor(0.33146154891145124f, 0.12244245263391232f, 0.4991827458843107f),
                    CPColor(0.3292300520323141f, 0.11905764321981127f, 0.49292595612342666f),
                    CPColor(0.3269137124539796f, 0.1157873496841953f, 0.4865864649569746f),
                    CPColor(0.32451307931207785f, 0.11263459791730848f, 0.48017007211645196f),
                    CPColor(0.3220288227606932f, 0.10960114111258401f, 0.4736849472572688f),
                    CPColor(0.31946262395497965f, 0.1066887988239266f, 0.46713728801395243f),
                    CPColor(0.316816480890235f, 0.10389861387653518f, 0.46053414662739794f),
                    CPColor(0.3140927841475553f, 0.10123077676403242f, 0.45388335612058467f),
                    CPColor(0.31129434479712365f, 0.0986847719340522f, 0.4471931371516162f),
                    CPColor(0.30842444457210105f, 0.09625938534057774f, 0.44047194882050544f),
                    CPColor(0.30548675819945936f, 0.09395276484082374f, 0.4337284999936111f),
                    CPColor(0.3024853636457425f, 0.0917611873973036f, 0.42697404043749887f),
                    CPColor(0.2994248396021477f, 0.08968225371675004f, 0.42021619665853854f),
                    CPColor(0.2963100038890529f, 0.08771325096046395f, 0.41346259134143476f),
                    CPColor(0.2931459309698525f, 0.08585065688962071f, 0.40672178082365834f),
                    CPColor(0.2899379244517661f, 0.08409078829085731f, 0.40000214725256295f),
                    CPColor(0.28669151388283165f, 0.08242987384848069f, 0.39331182532243375f),
                    CPColor(0.28341239797185225f, 0.08086415336549937f, 0.38665868550105914f),
                    CPColor(0.2801063857697547f, 0.07938999480226153f, 0.38005028528138707f),
                    CPColor(0.2767793961581559f, 0.07800394103378822f, 0.37349382846504675f),
                    CPColor(0.2734373934245081f, 0.07670280023749607f, 0.36699616136347685f),
                    CPColor(0.2700863774911405f, 0.07548367558427554f, 0.36056376228111864f),
                    CPColor(0.26673233211995284f, 0.0743440180285462f, 0.3542027606624096f),
                    CPColor(0.26338121807151404f, 0.07328165793989708f, 0.34791888996380105f),
                    CPColor(0.26003895187439957f, 0.0722947810433622f, 0.3417175669546984f),
                    CPColor(0.256711916510839f, 0.07138010624208224f, 0.3356064898460009f),
                    CPColor(0.25340685873736807f, 0.07053358292685183f, 0.3295945757321303f),
                    CPColor(0.2501284530619938f, 0.06975820642910699f, 0.32368100685760637f),
                    CPColor(0.24688226237959f, 0.06905363944920445f, 0.31786993834254956f),
                    CPColor(0.24367372557466271f, 0.06841985515092269f, 0.3121652405088837f),
                    CPColor(0.2405081333229594f, 0.0678571038148556f, 0.3065705449367832f),
                    CPColor(0.23739062429054825f, 0.06736588805055552f, 0.3010892218406587f),
                    CPColor(0.23433055727563878f, 0.0669355996616394f, 0.295740099298676f),
                    CPColor(0.23132955273021344f, 0.06657618693909059f, 0.29051361067988485f),
                    CPColor(0.2283917709422868f, 0.06628997924139618f, 0.28541074411068496f),
                    CPColor(0.22552164337737857f, 0.0660781731193956f, 0.28043398847505197f),
                    CPColor(0.22272706739121817f, 0.06593379067565194f, 0.275597146520537f),
                    CPColor(0.22001251100779617f, 0.0658579189189076f, 0.2709027999432586f),
                    CPColor(0.21737845072382705f, 0.06585966123356204f, 0.2663420934966951f),
                    CPColor(0.21482843531473683f, 0.06594038561377849f, 0.26191675992376573f),
                    CPColor(0.21237411048541005f, 0.06608502466175845f, 0.2576516509356954f),
                    CPColor(0.21001214221188125f, 0.06630857391894718f, 0.2535289048041211f),
                    CPColor(0.2077442377448806f, 0.06661453200418091f, 0.24954644291943817f),
                    CPColor(0.20558051999470117f, 0.06699046239786874f, 0.24572497420147632f),
                    CPColor(0.20352007949514977f, 0.06744417961242422f, 0.24205576625191821f),
                    CPColor(0.2015613376412984f, 0.06798327102620025f, 0.23852974228695395f),
                    CPColor(0.19971571438603364f, 0.06859271055370472f, 0.23517094067076993f),
                    CPColor(0.19794834061899208f, 0.06931406607166066f, 0.23194647381302336f),
                    CPColor(0.1960826032659409f, 0.07032122724242362f, 0.22874673279569585f),
                    CPColor(0.19410351363791453f, 0.07160830485689157f, 0.22558727307410353f),
                    CPColor(0.19199449184606268f, 0.0731828306492733f, 0.22243385243433622f),
                    CPColor(0.18975853639094634f, 0.07501986186214377f, 0.2193005075652994f),
                    CPColor(0.18739228342697645f, 0.07710209689958833f, 0.21618875376309582f),
                    CPColor(0.18488035509396164f, 0.07942573027972388f, 0.21307651648984993f),
                    CPColor(0.18774482037046955f, 0.07725158846803931f, 0.21387448578597812f),
                    CPColor(0.19049578401722037f, 0.07531127841678764f, 0.2146562337112265f),
                    CPColor(0.1931548636579131f, 0.07360681904011795f, 0.21542362939081539f),
                    CPColor(0.19571853588267552f, 0.07215778103960274f, 0.21617499187076789f),
                    CPColor(0.19819343656336558f, 0.07097462525273879f, 0.21690975060032436f),
                    CPColor(0.20058760685133747f, 0.07006457614998421f, 0.21762721310371608f),
                    CPColor(0.20290365333558247f, 0.06943524858045896f, 0.21833167885096033f),
                    CPColor(0.20531725273301316f, 0.06891959226639757f, 0.21911516689288835f),
                    CPColor(0.20785704662965598f, 0.06848439879702528f, 0.22000133917653536f),
                    CPColor(0.21052882914958676f, 0.06812195249816172f, 0.22098759107715404f),
                    CPColor(0.2133313859647627f, 0.06783014842602667f, 0.2220704321302429f),
                    CPColor(0.21625279838647882f, 0.06761633027051639f, 0.22324568672294431f),
                    CPColor(0.21930503925136402f, 0.06746578636294004f, 0.22451023616807558f),
                    CPColor(0.22247308588973624f, 0.06738821405309284f, 0.22585960379408354f),
                    CPColor(0.2257539681670791f, 0.06738213230014747f, 0.22728984778098055f),
                    CPColor(0.2291562027859284f, 0.06743473087115257f, 0.22879681433956656f),
                    CPColor(0.23266299920501882f, 0.06755710438847978f, 0.23037617493752832f),
                    CPColor(0.23627495835774248f, 0.06774359820987802f, 0.23202360805926608f),
                    CPColor(0.23999586188690308f, 0.06798502996477995f, 0.23373434258507808f),
                    CPColor(0.2438114972024792f, 0.06828985152901187f, 0.23550427698321885f),
                    CPColor(0.247720929905011f, 0.06865333790948652f, 0.2373288009471749f),
                    CPColor(0.25172899728289466f, 0.0690646308260355f, 0.23920260612763083f),
                    CPColor(0.2558213554748177f, 0.06953231029187984f, 0.24112190491594204f),
                    CPColor(0.25999463887892144f, 0.07005385560386188f, 0.24308218808684579f),
                    CPColor(0.2642551220706094f, 0.07061659562299544f, 0.24507758869355967f),
                    CPColor(0.2685909594817286f, 0.07122671627792246f, 0.24710443563450618f),
                    CPColor(0.272997015188973f, 0.07188355544616351f, 0.2491584709323293f),
                    CPColor(0.277471508091428f, 0.07258296989925478f, 0.2512349399594277f),
                    CPColor(0.2820174629736694f, 0.07331569321404097f, 0.25332800295084507f),
                    CPColor(0.28662309235899847f, 0.07408846082680887f, 0.2554347867371703f),
                    CPColor(0.29128515387578635f, 0.0748990498474667f, 0.25755101595750435f),
                    CPColor(0.2960004726065818f, 0.07574533600095842f, 0.25967245030364566f),
                    CPColor(0.3007727681291869f, 0.07661782433616476f, 0.2617929409781967f),
                    CPColor(0.30559226007249934f, 0.07752196310753731f, 0.2639100669211966f),
                    CPColor(0.31045520848595526f, 0.07845687167618218f, 0.2660200572779356f),
                    CPColor(0.3153587000920581f, 0.07942099731524319f, 0.2681190407694196f),
                    CPColor(0.3202998655799406f, 0.08041299473755484f, 0.2702032289303951f),
                    CPColor(0.3252788886040126f, 0.08142839007654609f, 0.27226772884656186f),
                    CPColor(0.3302917447118144f, 0.08246763389003825f, 0.27430929404579435f),
                    CPColor(0.3353335322445545f, 0.08353243411900396f, 0.2763253435679004f),
                    CPColor(0.34040164359597463f, 0.08462223619170267f, 0.27831254595259397f),
                    CPColor(0.345493557138718f, 0.08573665496512634f, 0.28026769921081435f),
                    CPColor(0.3506067824603248f, 0.08687555176033529f, 0.28218770540182386f),
                    CPColor(0.35573889947341125f, 0.08803897435024335f, 0.2840695897279818f),
                    CPColor(0.36088752387578377f, 0.0892271943627452f, 0.28591050458531014f),
                    CPColor(0.36605031412464006f, 0.0904406854276979f, 0.2877077458811747f),
                    CPColor(0.3712250843130934f, 0.09167999748026273f, 0.2894586539763317f),
                    CPColor(0.3764103053221462f, 0.09294519809377791f, 0.2911602415731392f),
                    CPColor(0.38160247377467543f, 0.09423873126371218f, 0.2928110750626949f),
                    CPColor(0.3867993907954417f, 0.09556181960083443f, 0.29440901248173756f),
                    CPColor(0.39199887556812907f, 0.09691583650296684f, 0.2959521200550908f),
                    CPColor(0.39719876876325577f, 0.09830232096827862f, 0.2974385647628578f),
                    CPColor(0.40239692379737496f, 0.09972293031495055f, 0.2988667436973397f),
                    CPColor(0.4075912039268871f, 0.10117945586419633f, 0.300235195077286f),
                    CPColor(0.41277985630360303f, 0.1026734006932461f, 0.3015422643746897f),
                    CPColor(0.41796105205173684f, 0.10420644885760968f, 0.3027865203963184f),
                    CPColor(0.42313214269556043f, 0.10578120994917611f, 0.3039675809469457f),
                    CPColor(0.4282910131578975f, 0.1073997763055258f, 0.30508479060294547f),
                    CPColor(0.4334355841041439f, 0.1090642347484701f, 0.3061376792828915f),
                    CPColor(0.4385637818793154f, 0.11077667828375456f, 0.30712600062348083f),
                    CPColor(0.44367358645071275f, 0.11253912421257944f, 0.3080497309546545f),
                    CPColor(0.4487629917317482f, 0.1143535557462255f, 0.30890905921943196f),
                    CPColor(0.4538300508699989f, 0.11622183788331528f, 0.3097044124984492f),
                    CPColor(0.45887288947308297f, 0.11814571137706886f, 0.3104363697903881f),
                    CPColor(0.46389102840284874f, 0.12012561256850712f, 0.31110343446582983f),
                    CPColor(0.46888111384598413f, 0.12216445576414045f, 0.31170911458932665f),
                    CPColor(0.473841437035254f, 0.12426354237989065f, 0.31225470169927194f),
                    CPColor(0.47877034239726296f, 0.12642401401409453f, 0.3127417273582196f),
                    CPColor(0.48366628618847957f, 0.1286467902201389f, 0.31317188565991266f),
                    CPColor(0.48852847371852987f, 0.13093210934893723f, 0.31354553695453014f),
                    CPColor(0.49335504375145617f, 0.13328091630401023f, 0.31386561956734976f),
                    CPColor(0.4981443546207415f, 0.13569380302451714f, 0.314135190862664f),
                    CPColor(0.5028952497497061f, 0.13817086581280427f, 0.3143566215383367f),
                    CPColor(0.5076068118105369f, 0.14071192654913128f, 0.3145320012008257f),
                    CPColor(0.5122783510532176f, 0.14331656120063752f, 0.3146630922831542f),
                    CPColor(0.5169084880054446f, 0.14598463068714407f, 0.3147540759228004f),
                    CPColor(0.5214965286322996f, 0.14871544765633712f, 0.3148076795453443f),
                    CPColor(0.5260418962547748f, 0.15150818660835483f, 0.31482653406646727f),
                    CPColor(0.5305442048985645f, 0.15436183633886777f, 0.3148129978918713f),
                    CPColor(0.5350027976174474f, 0.15727540775107324f, 0.3147708520739653f),
                    CPColor(0.5394173664919906f, 0.16024769309971934f, 0.31470295028655965f),
                    CPColor(0.5437877131360856f, 0.16327738551419116f, 0.31461204226295625f),
                    CPColor(0.5481137003346762f, 0.1663630904279047f, 0.3145010299091471f),
                    CPColor(0.5523952157271191f, 0.16950338809328983f, 0.3143729155461537f),
                    CPColor(0.5566322903496934f, 0.17269677158182117f, 0.31423043195101424f),
                    CPColor(0.5608249903911717f, 0.17594170887918095f, 0.31407639883970623f),
                    CPColor(0.564973435290177f, 0.17923664950367169f, 0.3139136046337036f),
                    CPColor(0.5690778478401143f, 0.18258004462335425f, 0.3137444095679653f),
                    CPColor(0.5731384575410787f, 0.18597036007065024f, 0.3135712686852f),
                    CPColor(0.5771555081299204f, 0.18940601489760422f, 0.3133970433357208f),
                    CPColor(0.5811293276158656f, 0.19288548904692518f, 0.3132239939418394f),
                    CPColor(0.5850602439646688f, 0.19640737049066315f, 0.3130540116373273f),
                    CPColor(0.5889486193554471f, 0.19997020971775276f, 0.31288922211590126f),
                    CPColor(0.5927948053652026f, 0.20357251410079796f, 0.3127323483930494f),
                    CPColor(0.5965991810912237f, 0.207212956082026f, 0.3125852303112123f),
                    CPColor(0.6003621301041158f, 0.21089030138947745f, 0.3124493441041469f),
                    CPColor(0.6040840169673274f, 0.21460331490206347f, 0.31232652641170694f),
                    CPColor(0.6077652399481865f, 0.21835070166659282f, 0.312219032918702f),
                    CPColor(0.6114062072731884f, 0.22213124697023234f, 0.3121288139643524f),
                    CPColor(0.6150072323639137f, 0.22594402043981826f, 0.3120568068576574f),
                    CPColor(0.6185686525887719f, 0.2297879924917992f, 0.3120046383872893f),
                    CPColor(0.6220907982108261f, 0.2336621873300741f, 0.3119738327362739f),
                    CPColor(0.6255741650043496f, 0.23756535071152696f, 0.3119669831491227f),
                    CPColor(0.6290189201698587f, 0.24149689191922535f, 0.3119844719564572f),
                    CPColor(0.6324253485421027f, 0.24545598775548677f, 0.3120276597462445f),
                    CPColor(0.6357937104834237f, 0.24944185818822678f, 0.3120979395330059f),
                    CPColor(0.6391243387840212f, 0.2534536546198314f, 0.3121968961206398f),
                    CPColor(0.642417577481186f, 0.257490519876798f, 0.31232631707560987f),
                    CPColor(0.6456734938264543f, 0.2615520316161528f, 0.31248673753935263f),
                    CPColor(0.6488923016945825f, 0.2656375533620908f, 0.3126794181957019f),
                    CPColor(0.652074172902773f, 0.269746505252367f, 0.3129056060581917f),
                    CPColor(0.6552193260932713f, 0.2738782665241015f, 0.3131666792687211f),
                    CPColor(0.6583280801134499f, 0.2780321095766563f, 0.3134643447952643f),
                    CPColor(0.6614003753260178f, 0.28220778870555907f, 0.3137991292649849f),
                    CPColor(0.6644363246987884f, 0.2864048361425618f, 0.31417223403606975f),
                    CPColor(0.6674360376636913f, 0.29062280081258873f, 0.31458483752056837f),
                    CPColor(0.670399595476762f, 0.29486126309253047f, 0.3150381395687221f),
                    CPColor(0.6733272556481733f, 0.29911962764489264f, 0.3155337232398221f),
                    CPColor(0.6762189792440975f, 0.30339762792450425f, 0.3160724937230589f),
                    CPColor(0.6790747402815734f, 0.30769497879760166f, 0.31665545668946665f),
                    CPColor(0.6818945715094452f, 0.31201133280550686f, 0.3172838048924495f),
                    CPColor(0.6846785094249453f, 0.3163463482122221f, 0.31795870784057567f),
                    CPColor(0.6874265643516962f, 0.32069970535138104f, 0.3186813762227769f),
                    CPColor(0.6901389321505248f, 0.32507091815606004f, 0.319453323328983f),
                    CPColor(0.6928154484676493f, 0.32945984647042675f, 0.3202754315314667f),
                    CPColor(0.6954560834689112f, 0.33386622163232865f, 0.3211488430698579f),
                    CPColor(0.6980608153581771f, 0.3382897632604862f, 0.3220747885521809f),
                    CPColor(0.700629624772421f, 0.34273019305341756f, 0.32305449047765694f),
                    CPColor(0.7031624945881415f, 0.34718723719598f, 0.32408913679491225f),
                    CPColor(0.7056595112261009f, 0.3516605297812094f, 0.32518014084085567f),
                    CPColor(0.7081205956842048f, 0.356149855233803f, 0.32632861885644465f),
                    CPColor(0.7105456546582587f, 0.36065500290840113f, 0.3275357416278876f),
                    CPColor(0.7129346683977347f, 0.36517570519856757f, 0.3288027427038317f),
                    CPColor(0.7152876061484729f, 0.3697117022522345f, 0.3301308728723546f),
                    CPColor(0.7176044490813385f, 0.3742627271068619f, 0.3315213862095893f),
                    CPColor(0.7198852149054985f, 0.37882848839337313f, 0.332975552002454f),
                    CPColor(0.7221299918421461f, 0.3834086450896306f, 0.33449469983585844f),
                    CPColor(0.7243386564778159f, 0.38800301593162145f, 0.3360799596569183f),
                    CPColor(0.7265112290022755f, 0.3926113126792577f, 0.3377325942005665f),
                    CPColor(0.7286477385671655f, 0.39723324476747235f, 0.33945384341064017f),
                    CPColor(0.7307482075484517f, 0.401868526884681f, 0.3412449533046818f),
                    CPColor(0.7328127050626875f, 0.4065168468778026f, 0.3431071517341082f),
                    CPColor(0.7348413359856494f, 0.4111778700451951f, 0.3450416947080907f),
                    CPColor(0.7368342217358587f, 0.4158512585029011f, 0.347049785207584f),
                    CPColor(0.7387914002459927f, 0.4205367299231533f, 0.34913260148542435f),
                    CPColor(0.7407130161950609f, 0.4252339389526239f, 0.35129130890802607f),
                    CPColor(0.7425992159973317f, 0.42994254036133867f, 0.3535270924537459f),
                    CPColor(0.7444501867657067f, 0.4346621718461711f, 0.35584108091122535f),
                    CPColor(0.7462661578916344f, 0.439392450449735f, 0.3582343914230064f),
                    CPColor(0.7480473927555956f, 0.44413297780351974f, 0.36070813602540136f),
                    CPColor(0.7497942054717047f, 0.4488833348154881f, 0.3632633755836028f),
                    CPColor(0.7515068504589166f, 0.45364314496866825f, 0.36590112443835765f),
                    CPColor(0.7531856636904657f, 0.45841199172949604f, 0.3686223664223477f),
                    CPColor(0.7548310506695954f, 0.46318942799460555f, 0.3714280448394211f),
                    CPColor(0.7564434157714071f, 0.4679750143794846f, 0.37431909037543515f),
                    CPColor(0.7580232553845584f, 0.4727682731566229f, 0.3772963553109668f),
                    CPColor(0.7595711110534006f, 0.4775687122205708f, 0.380360657784311f),
                    CPColor(0.7610876378057071f, 0.48237579130289127f, 0.3835127572385229f),
                    CPColor(0.7625733355405261f, 0.48718906673415824f, 0.38675335037837993f),
                    CPColor(0.7640288560928866f, 0.49200802533379656f, 0.39008308392311997f),
                    CPColor(0.7654549259333051f, 0.4968321290972723f, 0.3935025400011538f),
                    CPColor(0.7668522895064389f, 0.5016608471009063f, 0.39701221751773474f),
                    CPColor(0.768221765997353f, 0.5064936237128791f, 0.40061257089416885f),
                    CPColor(0.7695642334401418f, 0.5113298901696085f, 0.4043039806968248f),
                    CPColor(0.7708809196230247f, 0.516168926434691f, 0.40808667584648967f),
                    CPColor(0.7721725722960555f, 0.5210102658711383f, 0.4119608998712287f),
                    CPColor(0.7734402182988989f, 0.5258533209345156f, 0.41592679539764366f),
                    CPColor(0.774684947460632f, 0.5306974938477673f, 0.4199844035696376f),
                    CPColor(0.775907907306857f, 0.5355421788246119f, 0.42413367909988375f),
                    CPColor(0.7771103295521099f, 0.5403867491056124f, 0.4283745037125848f),
                    CPColor(0.7782934580763312f, 0.545230594884266f, 0.432706647838971f),
                    CPColor(0.7794586273150664f, 0.5500730841397727f, 0.4371297985644476f),
                    CPColor(0.7806077474948377f, 0.5549133574489061f, 0.4416433242636464f),
                    CPColor(0.7817418047898184f, 0.5597509805259486f, 0.44624687186865436f),
                    CPColor(0.7828622526444091f, 0.5645853311116688f, 0.45093985823706345f),
                    CPColor(0.7839706083641448f, 0.5694157832671042f, 0.4557215474289206f),
                    CPColor(0.7850684501960684f, 0.5742417003617839f, 0.46059116206904965f),
                    CPColor(0.7861573713233296f, 0.5790624629815756f, 0.465547782819184f),
                    CPColor(0.7872390410818835f, 0.5838774374455721f, 0.47059039582133383f),
                    CPColor(0.7883151404562396f, 0.5886860017356244f, 0.4757179187907608f),
                    CPColor(0.7893873776625194f, 0.5934875421745599f, 0.48092913815357724f),
                    CPColor(0.7904577684772788f, 0.5982813427706246f, 0.48622257801969754f),
                    CPColor(0.7915283284347561f, 0.603066705931472f, 0.49159667021646397f),
                    CPColor(0.7926003430423745f, 0.6078432208703702f, 0.4970502062153201f),
                    CPColor(0.7936755969866496f, 0.6126102933407219f, 0.5025816129126943f),
                    CPColor(0.7947558597265404f, 0.617367344002207f, 0.5081892121310299f),
                    CPColor(0.7958429237958377f, 0.6221137880845115f, 0.5138712409190979f),
                    CPColor(0.7969385471995161f, 0.626849056792967f, 0.5196258425240281f),
                    CPColor(0.7980444781513664f, 0.6315725822508955f, 0.5254510814483478f),
                    CPColor(0.7991624518501963f, 0.6362837937202919f, 0.5313449594256143f),
                    CPColor(0.8002941538975398f, 0.6409821330674986f, 0.5373053518514104f),
                    CPColor(0.8014412429256005f, 0.6456670345921877f, 0.5433300863249918f),
                    CPColor(0.8026053114611295f, 0.6503379374810385f, 0.5494169158460365f),
                    CPColor(0.8037879253107763f, 0.6549942654947263f, 0.5555635086708381f),
                    CPColor(0.804990547908103f, 0.6596354502756416f, 0.5617674511054698f),
                    CPColor(0.8062146052692706f, 0.6642608958528229f, 0.5680262917864979f),
                    CPColor(0.8074614045096935f, 0.6688700095398864f, 0.5743374637345958f),
                    CPColor(0.8087321917008969f, 0.6734621670219452f, 0.5806983480557674f),
                    CPColor(0.8100280946652069f, 0.6780367267397182f, 0.5871062690808275f),
                    CPColor(0.8113501401176333f, 0.6825930154624339f, 0.5935584890905076f),
                    CPColor(0.8126992203988149f, 0.6871303371461888f, 0.600052148204351f),
                    CPColor(0.8140761104699334f, 0.6916479479148213f, 0.6065843782630862f),
                    CPColor(0.8154814662727948f, 0.6961450550830809f, 0.6131522120932265f),
                    CPColor(0.8169157577505589f, 0.7006208301478398f, 0.6197526063725792f),
                    CPColor(0.8183793116449822f, 0.705074381896351f, 0.626382454789333f),
                    CPColor(0.8198723065045529f, 0.7095047497878748f, 0.6330385704006711f),
                    CPColor(0.8213947205565636f, 0.7139109141951604f, 0.6397176669767276f),
                    CPColor(0.8229463511042843f, 0.7182917733129006f, 0.6464164243818421f),
                    CPColor(0.8245268129450285f, 0.7226461431208888f, 0.653131379154226f),
                    CPColor(0.8261354971058026f, 0.7269727551823826f, 0.659859001562165f),
                    CPColor(0.8277716072353446f, 0.7312702332407809f, 0.6665957020468297f),
                    CPColor(0.8294340781648147f, 0.7355371221572935f, 0.6733377200930191f),
                    CPColor(0.8311216352909631f, 0.7397718464763862f, 0.6800812520363146f),
                    CPColor(0.8328327718577798f, 0.7439727181745988f, 0.6868223587464855f),
                    CPColor(0.8345656905566583f, 0.7481379479992134f, 0.6935569764986385f),
                    CPColor(0.8363189884473793f, 0.7522654895287526f, 0.7002799902886496f),
                    CPColor(0.8380912347613196f, 0.7563531486080863f, 0.7069856139021298f),
                    CPColor(0.8398783988412087f, 0.7603990719977968f, 0.7136714781112923f),
                    CPColor(0.8416775076684515f, 0.7644010120098295f, 0.7203329938728462f),
                    CPColor(0.843485292229337f, 0.7683566039987018f, 0.7269653699897204f),
                    CPColor(0.8452981073195511f, 0.7722633860104472f, 0.7335636824054149f),
                    CPColor(0.847111955079651f, 0.7761188023604716f, 0.7401227576280706f),
                    CPColor(0.8489224556311764f, 0.7799202140765015f, 0.7466371929366437f),
                    CPColor(0.8507269702317879f, 0.7836645734238389f, 0.7530974636118285f),
                    CPColor(0.8525190720770844f, 0.7873493613354844f, 0.7594994148789691f),
                    CPColor(0.8542921961147046f, 0.7909719677709199f, 0.765838014779141f),
                    CPColor(0.856040223147254f, 0.7945296360155061f, 0.7721061003767414f),
                    CPColor(0.857756629435049f, 0.7980196314271393f, 0.778295716672475f),
                    CPColor(0.8594346370300241f, 0.8014392309950078f, 0.7843978875138392f),
                    CPColor(0.8610711702756552f, 0.8047851790981223f, 0.7903952966373629f),
                    CPColor(0.8626560105112757f, 0.8080552380426153f, 0.796282666437655f),
                    CPColor(0.8641834372394103f, 0.8112464422465354f, 0.8020461269686395f),
                    CPColor(0.8656493432560532f, 0.8143554406751491f, 0.8076697232416455f),
                    CPColor(0.867053149070485f, 0.8173780404191124f, 0.813134196269114f),
                    CPColor(0.8683995469581863f, 0.8203087551218152f, 0.8184163896312899f),
                    CPColor(0.8696913150261381f, 0.8231415885956916f, 0.8235047668317317f),
                    CPColor(0.8709384671729751f, 0.8258685788943851f, 0.8283849726114961f),
                    CPColor(0.8721533197845432f, 0.8284805282370967f, 0.8330486712880828f),
                    CPColor(0.8733517136091627f, 0.8309671525127262f, 0.8374885100119709f),
                    CPColor(0.8745379332026019f, 0.8333197294864546f, 0.8417192535806901f),
                    CPColor(0.875714587099614f, 0.8355302318472394f, 0.8457553751902708f),
                    CPColor(0.8768784845161469f, 0.8375923807118654f, 0.8496137354915025f),
                    CPColor(0.8780229843664901f, 0.8395016561854007f, 0.8533064535245892f),
                    CPColor(0.8791324424079277f, 0.8412555488447591f, 0.8568557229103964f),
                    CPColor(0.8801929331569581f, 0.8428522482477862f, 0.8602739992715663f),
                    CPColor(0.8811916987134195f, 0.8442906671771735f, 0.8635659516866988f),
                    CPColor(0.8821154248940161f, 0.8455700725455935f, 0.8667376504623333f),
                    CPColor(0.8829516859544853f, 0.8466897027569927f, 0.8697961704819097f),
                    CPColor(0.8836912714589804f, 0.8476489176151927f, 0.8727414710144156f),
                    CPColor(0.8843271305411354f, 0.8484474157205542f, 0.8755678522824297f),
                    CPColor(0.8848513815990857f, 0.849084264228938f, 0.8782823528537247f),
                    CPColor(0.8852589797263047f, 0.8495589281098921f, 0.8808841479402484f),
                    CPColor(0.8855471481195238f, 0.8498717428363158f, 0.8833620612117095f),
                    CPColor(0.8857115512284565f, 0.8500218611585632f, 0.8857253899008712f)
                )
            )
        )
        entries.addAll(
            arrayListOf(
                bty,
                heated,
                rainy,
                linred,
                negpos,
                treemapper,
                composite,
                coolwarm,
                magma,
                inferno,
                plasma,
                viridis,
                cividis,
                twilight,
                publimedia,
                magenta,
                rainbow,
                locs,
                ocs,
                lingrey,
                YlOrRd,
                PRGn,
                PuOr,
                RdGy,
                Spectral,
                Grays,
                PuBuGn,
                RdPu,
                BuPu,
                YlOrBr,
                Greens,
                BuGn,
                Accents,
                GnBu,
                PuRd,
                Purples,
                RdYlGn,
                Paired,
                Blues,
                RdBu,
                Oranges,
                RdYlBu,
                PuBu,
                OrRd,
                Set3,
                Set2,
                Set1,
                Reds,
                PiYG,
                Dark2,
                YlGn,
                BrBG,
                YlGnBu,
                Pastel2,
                Pastel1,
                IsoL,
                CubicL,
                Edge
            )
        )
    }
}