package org.mkui.property

import kotlin.properties.ReadWriteProperty
import kotlin.reflect.KProperty
import kotlin.reflect.KProperty1

typealias BiConsumer<O> = (O, O) -> Unit

class ObservableManager<T> {
    private val properties = mutableMapOf<String, Any?>()
    private val listeners = mutableMapOf<String, MutableList<BiConsumer<Any?>>>()

    private fun updateValue(name: String, value: Any?) {
        val old = properties.put(name, value)
        listeners[name]?.forEach { it(old, value) }
    }

    fun registerListener(name: String, biConsumer: BiConsumer<Any?>) {
        listeners.getOrPut(name) { mutableListOf() }.add(biConsumer)
    }

    inline fun <reified PROP> registerListener(property: KProperty1<T, PROP>, crossinline consumer: BiConsumer<PROP>) =
            registerListener(property.name) { old, new -> consumer(old as PROP, new as PROP) }

    inner class WrappingRWProperty<PROP>(
            prop: KProperty<*>,
            value: Any?,
            private val converter: (Any?) -> PROP
    ) : ReadWriteProperty<T, PROP> {
        //We can register the initial value ;-)
        init {
            properties[prop.name] = value
        }

        override operator fun getValue(thisRef: T, property: KProperty<*>) = properties[property.name].let(converter)
        override operator fun setValue(thisRef: T, property: KProperty<*>, value: PROP) = updateValue(property.name, value)
    }
}

class ObservableValue<CLASS, PROP>(
        private val delegater: ObservableManager<CLASS>,
        val value: Any?,
        private val converter: (Any?) -> PROP
) : DelegateProvider<CLASS, PROP> {
    override operator fun provideDelegate(
            thisRef: CLASS,
            prop: KProperty<*>
    ): ReadWriteProperty<CLASS, PROP> = delegater.WrappingRWProperty(prop, value, converter)
}

interface DelegateProvider<CLASS, PROP> {
    operator fun provideDelegate(
            thisRef: CLASS,
            prop: KProperty<*>
    ): ReadWriteProperty<CLASS, PROP>
}


interface IObservable<CLASS> {
    val manager: ObservableManager<CLASS>
}

open class Observable<CLASS> : IObservable<CLASS> {
    override val manager = ObservableManager<CLASS>()
}


@Suppress("UNCHECKED_CAST")
inline fun <reified CLASS, PROP> IObservable<CLASS>.observable(value: PROP): DelegateProvider<CLASS, PROP> = ObservableValue(manager, value) { a: Any? -> a as PROP }

inline fun <CLASS, reified PROP> IObservable<CLASS>.registerListener(property: KProperty1<CLASS, PROP>, crossinline consumer: BiConsumer<PROP>) = manager.registerListener(property, consumer)


//class Test : IObservable<Test> by Observable() {
//    var test by observable("old")
//}
//
//fun main(args: Array<String>) {
//    val test = Test()
//    test.registerListener(Test::test) { old, new ->
//        print("test: $old -> $new")
//    }
//    test.test = "new" // prints "test: old -> new"
//}
