package org.mkui.canvas

import com.macrofocus.common.crossplatform.CPHelper
import com.macrofocus.common.timer.CPTimer
import com.macrofocus.common.timer.CPTimerListener
import javafx.application.Platform
import javafx.beans.InvalidationListener
import javafx.event.EventHandler
import javafx.scene.canvas.GraphicsContext
import javafx.scene.layout.StackPane
import org.mkui.color.CPColor
import org.mkui.component.CPComponent
import org.mkui.graphics.CPGraphicsContext2D
import org.mkui.graphics.GraphicsContext2D
import java.util.*
import java.util.concurrent.CopyOnWriteArrayList
import kotlin.collections.HashSet

actual open class CPCanvas : CPComponent {
    private val layers: MutableList<JavaFXLayer> = CopyOnWriteArrayList()

    private val listeners: MutableList<CanvasListener> = ArrayList()

    override val nativeComponent = StackPane()

    val canvas = CanvasComponent()

    val listener = InvalidationListener {
        notifySizeChanged(getWidth().toInt(), getHeight().toInt())
        canvas.schedulePrepare()
    }

    init {
        canvas.widthProperty().addListener(listener)
        canvas.heightProperty().addListener(listener)

        nativeComponent.getChildren()!!.add(canvas)
        canvas.widthProperty().bind(nativeComponent.widthProperty())
        canvas.heightProperty().bind(nativeComponent.heightProperty())

        canvas.schedulePrepare()

    }

    actual fun addLayer(d: GraphicsContextDrawing) {
        layers.add(object : JavaFXLayer {
            override fun prepare(width: Int, height: Int) {

            }

            override fun render(g: GraphicsContext?) {
                d.draw(CPGraphicsContext2D(g!!), getWidth(), getHeight())
            }
        })
    }

    fun addNativeLayer(d: JavaFXLayer) {
        layers.add(d)
    }

    inner class CanvasComponent() : ResizableCanvas() {
        private val dirtyLayers: MutableSet<JavaFXLayer?> = HashSet()

        protected val prepareTimer: CPTimer =
            CPHelper.instance.createTimer("CanvasPrepareTimer", 50, true, object : CPTimerListener {
                override fun timerTriggered() {
                    prepare()
                }
            })

        protected val renderTimer: CPTimer =
            CPHelper.instance.createTimer("CanvasRenderTimer", 17, false, object : CPTimerListener {
                override fun timerTriggered() {
                    if (Platform.isFxApplicationThread()) {
                        if (canvas.isVisible) {
                            render()
                        }
                    } else {
                        Platform.runLater {
                            if (canvas.isVisible) {
                                render()
                            }
                        }
                    }
                }
            })

        private fun CanvasComponent() {}

        private fun schedulePrepare(layer: JavaFXLayer) {
            dirtyLayers.add(layer)
            prepareTimer.restart()
        }

        fun schedulePrepare() {
            for (layer in layers) {
                dirtyLayers.add(layer)
            }
            prepareTimer.restart()
        }

        private fun prepare() {
//            prepareTiming.start()
            val width = width
            val height = height
            var dirtyLayers: List<JavaFXLayer?>
            synchronized(this.dirtyLayers) {
                dirtyLayers = ArrayList(this.dirtyLayers)
                this.dirtyLayers.clear()
            }
            for (layer in dirtyLayers) {
                layer!!.prepare(width.toInt(), height.toInt())
            }
//            prepareTiming.end()
            scheduleRender()
        }

        private fun render() {
//            renderTiming.start()
            val graphicsContext2D = graphicsContext2D
            graphicsContext2D.clearRect(0.0, 0.0, canvas.width, canvas.height)
            for (layer in layers) {
                graphicsContext2D.save()
                layer.render(graphicsContext2D)
                graphicsContext2D.restore()
            }
            val width = width
            val height = height
//            renderTiming.end()
//            if (isShowTiming()) {
//                prepareTiming.draw(graphicsContext2D, Dimension(width.toInt(), height.toInt()), 32)
//                renderTiming.draw(graphicsContext2D, Dimension(width.toInt(), height.toInt()), 16)
//            }
        }

        fun scheduleRender() {
            renderTimer.restart()
        }
    }

    actual open fun getWidth(): Double {
        return nativeComponent.width.toDouble()
    }

    actual open fun getHeight(): Double {
        return nativeComponent.height.toDouble()
    }

    actual open fun addCanvasListener(l: CanvasListener) {
        listeners.add(l)
    }

    protected open fun notifySizeChanged(width: Int, height: Int) {
        for (listener in listeners) {
            listener.sizeChange(width, height)
        }
    }

    actual fun redraw() {
        canvas.schedulePrepare()
    }

    actual fun addMouseListener(l: MouseListener) {
        canvas.onMouseClicked = EventHandler { event -> l.mouseClicked(JavaFXMouseEvent(event!!)) }
        canvas.onMousePressed = EventHandler { event -> l.mousePressed(JavaFXMouseEvent(event!!)) }
        canvas.onMouseReleased = EventHandler { event -> l.mouseReleased(JavaFXMouseEvent(event!!)) }
        canvas.onMouseEntered = EventHandler { event -> l.mouseEntered(JavaFXMouseEvent(event!!)) }
        canvas.onMouseExited = EventHandler { event -> l.mouseExited(JavaFXMouseEvent(event!!)) }
    }

    actual fun addMouseMotionListener(l: MouseMotionListener) {
        canvas.onMouseDragged = EventHandler { event -> l.mouseDragged(JavaFXMouseEvent(event!!)) }
        canvas.onMouseMoved = EventHandler { event -> l.mouseMoved(JavaFXMouseEvent(event!!)) }
    }

    actual fun addMouseWheelListener(l: MouseWheelListener) {
        canvas.onScroll = EventHandler { event ->
            l.mouseWheelMoved(object : MouseWheelEvent {
                override val x: Int
                    get() = event.x.toInt()
                override val y: Int
                    get() = event.y.toInt()
                override val wheelRotation: Int
                    get() = (-event.deltaY).toInt()
            })
        }
    }

    actual fun addKeyListener(l: KeyListener) {}

    actual fun addContextMenuListener(l: ContextMenuListener) {}

}