package org.mkui.canvas

import kotlinx.browser.window
import org.mkui.component.CPComponent
import org.w3c.dom.CanvasRenderingContext2D
import org.w3c.dom.HTMLCanvasElement
import react.*
import react.dom.canvas
import kotlin.js.Date
import kotlin.math.absoluteValue

external interface FunctionalReactCanvasProps: RProps {
}

class FunctionalCPCanvas(val graphicsContextDrawings: MutableList<org.mkui.canvas.CanvasRenderingContext2DLayer>, private val listeners: List<CanvasListener>) : CPComponent {
    private lateinit var resizeObserver: org.mkui.canvas.ResizeObserver

    private val mouseListeners : MutableList<MouseListener> = ArrayList()
    private val mouseMotionListeners: MutableList<MouseMotionListener> = ArrayList()
    private val mouseWheelListeners: MutableList<MouseWheelListener> = ArrayList()
    private val keyListeners: MutableList<KeyListener> = ArrayList()
    private val contextMenuListeners: MutableList<ContextMenuListener> = ArrayList()

    var width = 0
    var height = 0
    var scaleFactor = 1.0
    var physicalWidth = 0
    var physicalHeight = 0

    fun redraw() {
        paintTheCanvas()
    }

    private fun paintTheCanvas() {
        val start = Date.now();

        with(ctx) {
//            val width = canvas.clientWidth
//            val height = canvas.clientHeight
//            val scaleFactor = findDesiredScaleFactor()
//            val dim = canvas.getBoundingClientRect()
//            if(dimensionHasChanged(width, height, scaleFactor)) {
//                syncCanvasDimension(width, height, scaleFactor)
//            }

            clearAndResetRenderingContext2D(ctx)

            for (i in graphicsContextDrawings) {
                i.prepare(width, height)
            }

            for (i in graphicsContextDrawings) {
                i.render(ctx)
            }
        }

        val delta = (Date.now() - start);
        val fps = 1/delta

        ctx.fillStyle = "red"
        ctx.fillText("" + delta.absoluteValue + " ms", 5.0, height - 12.0)

//        window.requestAnimationFrame { paintTheCanvas() }

    }

    private fun findDesiredScaleFactor(): Double {
        return if (org.mkui.canvas.HIDPI)
            window.devicePixelRatio
        else
            1.0
    }

    fun dimensionHasChanged(width: Int, height: Int, scaleFactor: Double): Boolean {
        return this.width != width || this.height != height || this.scaleFactor != scaleFactor
    }

    fun syncCanvasDimension(width: Int, height: Int, scaleFactor: Double) {
        this.width = width
        this.height = height
        this.scaleFactor = scaleFactor
        this.physicalWidth = (width * scaleFactor).toInt()
        this.physicalHeight = (height * scaleFactor).toInt()
        val canvas = canvasRef.current
        if(canvas != null) {
            println("Resizing " + width + "x" + height + "@" + scaleFactor + " -> " + physicalWidth + "x" + physicalHeight)
            canvas.width = physicalWidth
            canvas.height = physicalHeight
            window.requestAnimationFrame { paintTheCanvas() }
        }
    }

    fun clearAndResetRenderingContext2D(g: CanvasRenderingContext2D) {
        // This seems like a hack, but works better because it reset the graphic context
//            canvas.setCoordinateSpaceWidth(canvas.getOffsetWidth());
        g.setTransform(scaleFactor, 0.0, 0.0, scaleFactor, 0.0, 0.0)
        g.clearRect(0.0, 0.0, width.toDouble(), height.toDouble())
    }

    private fun publishData(canvas: HTMLCanvasElement) {
//        val width = canvas.clientWidth // Todo: offsetWidth
//        val height = canvas.clientHeight // Todo: offsetHeight
        val width = canvas.offsetWidth // Todo: offsetWidth
        val height = canvas.offsetHeight // Todo: offsetHeight
        println("p " + width + "x" + height)
        val scaleFactor = findDesiredScaleFactor()
        val dim = canvas.getBoundingClientRect()
        if(dimensionHasChanged(width, height, scaleFactor)) {
            syncCanvasDimension(width, height, scaleFactor)
            for (listener in listeners) {
                listener.sizeChange(width, height)
            }
        }
    }

    val nativeComponent = functionalComponent<org.mkui.canvas.FunctionalReactCanvasProps> {
//    val (count, setCount) = useState(0)
        useEffect {
            val targetContainer = canvasRef.current!!
            println("componentDidMount " + targetContainer)
            resizeObserver = org.mkui.canvas.ResizeObserver { _, _ -> publishData(targetContainer) }
            resizeObserver.observe(targetContainer)

            org.mkui.canvas.ElementalCanvasHandler(
                this@FunctionalCPCanvas, targetContainer,
                mouseListeners,
                mouseMotionListeners,
                mouseWheelListeners,
                keyListeners,
                contextMenuListeners
            )

            targetContainer.onmousemove = {
                val event = org.mkui.canvas.JSMouseEvent(it, 0)
                for (listener in mouseMotionListeners) {
                    listener.mouseMoved(event)
                }
            }
        }

        val canvas = canvas("canvas") {
            ref = canvasRef
        }
//        window.requestAnimationFrame { paintTheCanvas() }
    }

    private val ctx get() = canvasRef.current!!.getContext("2d") as CanvasRenderingContext2D

    private val canvasRef = createRef<HTMLCanvasElement>()

    override fun getNativeComponent(): FunctionalComponent<org.mkui.canvas.FunctionalReactCanvasProps> {
        return nativeComponent
    }


    fun addMouseListener(l: MouseListener) {
        mouseListeners.add(l)
    }

    fun addMouseMotionListener(l: MouseMotionListener) {
        mouseMotionListeners.add(l)
    }

    fun addMouseWheelListener(l: MouseWheelListener) {
        mouseWheelListeners.add(l)
    }

    fun addKeyListener(l: KeyListener) {
        keyListeners.add(l)
    }

    fun addContextMenuListener(l: ContextMenuListener) {
        contextMenuListeners.add(l)
    }
}