package org.mkui.component.menu

import com.macrofocus.common.command.Command
import com.macrofocus.common.command.ToggleUICommand
import com.macrofocus.common.command.UICommand
import com.macrofocus.common.logging.Logging
import com.macrofocus.common.properties.MutableProperty
import com.macrofocus.common.properties.Property
import com.macrofocus.common.properties.PropertyEvent
import com.macrofocus.common.properties.PropertyListener
import org.mkui.component.CPComponent
import java.awt.Desktop
import java.awt.event.ActionListener
import java.io.IOException
import java.net.URISyntaxException
import java.net.URL
import javax.swing.*

actual class CPMenu : CPComponent {
    private var menu: JMenu

    private val titleListener: PropertyListener<String?> = object : PropertyListener<String?> {
        override fun propertyChanged(event: PropertyEvent<String?>) {
            nativeComponent.text = event.newValue
        }
    }

    actual constructor(title: String) {
        menu = JMenu(title)
    }

    actual constructor(title: Property<String?>) {
        menu = JMenu(title.value)
        title.addPropertyListener(titleListener)
    }

    actual fun addMenu(title: String): CPMenu {
        val subMenu: CPMenu = CPMenu(title)
        menu.add(subMenu.nativeComponent)
        return subMenu
    }

    actual fun addMenu(title: Property<String?>): CPMenu {
        val subMenu: CPMenu = CPMenu(title)
        menu.add(subMenu.nativeComponent)
        return subMenu
    }

    actual fun addAction(title: String?, keystroke: String?, command: Command?) {
        val menuItem = JMenuItem(title)
        menuItem.accelerator = KeyStroke.getKeyStroke(keystroke)
        menuItem.addActionListener(ActionListener { command!!.execute() })
        menuItem.isEnabled = command != null
        menu.add(menuItem)
    }

    actual fun addAction(command: UICommand?) {
        val menuItem = createUICommandMenuItem(command)
        menu.add(menuItem)
    }

    fun createUICommandMenuItem(command: UICommand?): JMenuItem {
        val menuItem = JMenuItem(command!!.titleProperty()!!.value)
        command.titleProperty()!!.addPropertyListener(TitlePropertyListener(menuItem))
        if (command.keystrokeProperty() != null) {
            menuItem.accelerator = KeyStroke.getKeyStroke(command.keystrokeProperty()!!.value)
            command.keystrokeProperty()!!.addPropertyListener(object : PropertyListener<String?> {
                override fun propertyChanged(event: PropertyEvent<String?>) {
                    menuItem.accelerator = KeyStroke.getKeyStroke(event.newValue)
                }
            })
        }
        if (command.iconProperty() != null) {
            val value = command.iconProperty()!!.value!!
            if (value is Icon) {
                menuItem.icon = value
            }
            command.iconProperty()!!.addPropertyListener(object : PropertyListener<Any?> {
                override fun propertyChanged(event: PropertyEvent<Any?>) {
                    if (event.newValue is Icon) {
                        menuItem.icon = event.newValue as Icon
                    }
                }
            })
        }
        menuItem.isEnabled = command.enabledProperty()!!.value!! && command != null
        command.enabledProperty()!!.addPropertyListener(object : PropertyListener<Boolean?> {
            override fun propertyChanged(event: PropertyEvent<Boolean?>) {
                menuItem.isEnabled = event.newValue!!
            }
        })
        menuItem.addActionListener(ActionListener { e -> command.execute(e.source) })
        return menuItem
    }

    actual fun addEllipsisAction(title: String, keystroke: String?, command: Command?) {
        val menuItem = JMenuItem("$title...")
        menuItem.accelerator = KeyStroke.getKeyStroke(keystroke)
        menuItem.addActionListener(ActionListener { command!!.execute() })
        menuItem.isEnabled = command != null
        menu.add(menuItem)
    }

    actual fun addEllipsisAction(command: UICommand?) {
        val menuItem = JMenuItem(command!!.titleProperty()!!.value.toString() + "...")
        command.titleProperty()!!.addPropertyListener(object : PropertyListener<String?> {
            override fun propertyChanged(event: PropertyEvent<String?>) {
                menuItem.text = event.newValue + "..."
            }
        })
        if (command.keystrokeProperty() != null) {
            menuItem.accelerator = KeyStroke.getKeyStroke(command.keystrokeProperty()!!.value)
            command.keystrokeProperty()!!.addPropertyListener(object : PropertyListener<String?> {
                override fun propertyChanged(event: PropertyEvent<String?>) {
                    menuItem.accelerator = KeyStroke.getKeyStroke(event.newValue)
                }
            })
        }
        if (command.iconProperty() != null) {
            val value = command.iconProperty()!!.value!!
            if (value is Icon) {
                menuItem.icon = value
            }
            command.iconProperty()!!.addPropertyListener(object : PropertyListener<Any?> {
                override fun propertyChanged(event: PropertyEvent<Any?>) {
                    if (event.newValue is Icon) {
                        menuItem.icon = event.newValue as Icon
                    }
                }
            })
        }
        menuItem.isEnabled = command.enabledProperty()!!.value!! && command != null
        command.enabledProperty()!!.addPropertyListener(object : PropertyListener<Boolean?> {
            override fun propertyChanged(event: PropertyEvent<Boolean?>) {
                menuItem.isEnabled = event.newValue!!
            }
        })
        menuItem.addActionListener(ActionListener { e -> command.execute(e.source) })
        menu.add(menuItem)
    }

    actual fun addCheckBox(command: ToggleUICommand?) {
        val menuItem = createToggleUICommandMenuItem(command)
        menu.add(menuItem)
    }

    fun createToggleUICommandMenuItem(
        command: ToggleUICommand?,
    ): JCheckBoxMenuItem {
        val menuItem = JCheckBoxMenuItem(command!!.titleProperty()!!.value)
        command.titleProperty()!!.addPropertyListener(TitlePropertyListener(menuItem))
        if (command.keystrokeProperty() != null) {
            menuItem.accelerator = KeyStroke.getKeyStroke(command.keystrokeProperty()!!.value)
            command.keystrokeProperty()!!.addPropertyListener(object : PropertyListener<String?> {
                override fun propertyChanged(event: PropertyEvent<String?>) {
                    menuItem.accelerator = KeyStroke.getKeyStroke(event.newValue)
                }
            })
        }
        //        menuItem.addActionListener(new ActionListener() {
//            @Override
//            public void actionPerformed(ActionEvent e) {
//                command.execute();
//            }
//        });
        menuItem.isEnabled = command.enabledProperty()!!.value!! && command != null
        command.enabledProperty()!!.addPropertyListener(object : PropertyListener<Boolean?> {
            override fun propertyChanged(event: PropertyEvent<Boolean?>) {
                menuItem.isEnabled = event.newValue!!
            }
        })
        menuItem.addActionListener(ActionListener { e -> command.execute(e.source) })
        menuItem.state = command.toggleProperty().value
        command.toggleProperty().addPropertyListener(object : PropertyListener<Boolean> {
            override fun propertyChanged(event: PropertyEvent<Boolean>) {
                menuItem.state = event.newValue
            }
        })
        return menuItem
    }

    actual fun addCheckBox(title: String, keystroke: String?, property: MutableProperty<Boolean?>?) {
        val menuItem = JCheckBoxMenuItem(title)
        menuItem.accelerator = KeyStroke.getKeyStroke(keystroke)
        BooleanPropertyToggleButtonModelCoordinator(property, menuItem.model)
        menu.add(menuItem)
    }

    actual fun <T> addRadio(title: String, property: MutableProperty<T?>?, vararg values: T?) {
        val subMenu: CPMenu = addMenu(title)
        for (value in values) {
            val menuItem = JRadioButtonMenuItem(value.toString())
            if (property != null) {
                EqualPropertyToggleButtonModelCoordinator<T?>(property, value, menuItem.model)
            }
            subMenu.nativeComponent.add(menuItem)
        }
        menu.add(subMenu.nativeComponent)
    }

    actual fun addRadio(command: ToggleUICommand, group: ToggleGroup) {
        val menuItem = SwingRadioButtonMenuItem(command)
        menu.add(menuItem.nativeComponent)
        group.add(command)
    }

    actual fun addLink(property: Property<String?>, url: String?) {
        val menuItem = JMenuItem(property.value)
        menuItem.addActionListener(ActionListener {
            try {
                Desktop.getDesktop().browse(URL("url").toURI())
            } catch (e: IOException) {
                Logging.instance.process(e)
            } catch (e: URISyntaxException) {
                Logging.instance.process(e)
            }
        })
        menuItem.isEnabled = url != null
        menu.add(menuItem)
    }

    actual open fun addSeparator() {
        menu.add(JSeparator())
    }

    open fun removeAll() {
        menu.removeAll()
    }

    override val nativeComponent: JMenu
        get() = menu

    open fun add(menuItem: JMenuItem?) {
        menu.add(menuItem)
    }

    private class BooleanPropertyToggleButtonModelCoordinator(
        property: MutableProperty<Boolean?>?,
        buttonModel: ButtonModel
    ) {
        init {
            if (property != null) {
                property.addPropertyListener(object : PropertyListener<Boolean?> {
                    override fun propertyChanged(event: PropertyEvent<Boolean?>) {
                        buttonModel.isSelected = event.newValue!!
                    }
                })
                buttonModel.isSelected = property.value!!
                buttonModel.addChangeListener { property.value = buttonModel.isSelected }
            }
        }
    }

    private class EqualPropertyToggleButtonModelCoordinator<V>(
        property: MutableProperty<V>,
        value: V,
        buttonModel: ButtonModel
    ) {
        init {
            property.addPropertyListener(object : PropertyListener<V> {
                override fun propertyChanged(event: PropertyEvent<V>) {
                    buttonModel.isSelected = event.newValue === value
                }
            })
            buttonModel.isSelected = property.value === value
            buttonModel.addChangeListener {
                if (buttonModel.isSelected) {
                    property.value = value
                }
            }
        }
    }

    private class TitlePropertyListener(private val menuItem: JMenuItem) :
        PropertyListener<String?> {
        override fun propertyChanged(event: PropertyEvent<String?>) {
            menuItem.text = event.newValue
        }
    }

    actual interface ViewCommand<V> : Command {
        actual fun setup(view: V?)
    }
}