package org.mkui.graphics

import org.mkui.color.CPColor
import org.mkui.font.CPFont
import org.mkui.font.CPFontMetrics
import org.mkui.geom.*
import org.mkui.geom.Shape
import java.awt.*
import java.awt.geom.GeneralPath

class SwingIGraphics(context: Graphics2D?) : AbstractIGraphics() {
    val context: Graphics2D
    private var baseline: IGraphics.TextBaseline = IGraphics.TextBaseline.Alphabetic
    private var path: GeneralPath? = null

    override fun getLineWidth(): Double {
        return (context.stroke as BasicStroke).lineWidth.toDouble()
    }

    override fun setLineWidth(lineWidth: Double) {
        // CAP_BUTT is the default with HTML5 Canvas
        context.stroke = BasicStroke(lineWidth.toFloat(), BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER)
    }

    override fun setLineDash(dashPattern: FloatArray?) {
        // CAP_BUTT is the default with HTML5 Canvas
        val stroke: Stroke = BasicStroke(1f, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 10f, dashPattern, 0f)
        context.stroke = stroke
    }

    override fun setGlobalAlpha(alpha: Double) {
        context.composite = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, alpha.toFloat())
    }

    override fun getColor(): org.mkui.color.CPColor {
        return org.mkui.color.CPColor(context.color)
    }

    override fun setColor(color: org.mkui.color.CPColor) {
        context.color = color.nativeColor
    }

    override fun draw(geometry: Shape?) {
        if (geometry is Rectangle2D) {
            val r: Rectangle2D = geometry
            context.draw(java.awt.geom.Rectangle2D.Double(r.x, r.y, r.width, r.height))
        } else if (geometry is Arc2D) {
            val arc: Arc2D = geometry
            context.draw(
                java.awt.geom.Arc2D.Double(
                    arc.x,
                    arc.y,
                    arc.width,
                    arc.height,
                    arc.angleStart,
                    arc.angleExtent,
                    arc.arcType
                )
            )
        } else {
            if (geometry != null) {
                val shape = GeometryShape(geometry)
                context.draw(shape)
            }
        }
    }

    override fun fill(geometry: Shape?) {
        if (geometry is Rectangle2D) {
            val r: Rectangle2D = geometry
            context.fill(java.awt.geom.Rectangle2D.Double(r.x, r.y, r.width, r.height))
        } else if (geometry is Ellipse2D) {
            val r: Ellipse2D = geometry
            context.fill(java.awt.geom.Ellipse2D.Double(r.x, r.y, r.width, r.height))
        } else {
            val shape = GeometryShape(geometry!!)
            context.fill(shape)
        }
    }

    override fun drawLine(x1: Int, y1: Int, x2: Int, y2: Int) {
        context.drawLine(x1, y1, x2, y2)
    }

    override fun drawLine(x1: Double, y1: Double, x2: Double, y2: Double) {
        context.draw(java.awt.geom.Line2D.Double(x1, y1, x2, y2))
    }

    override fun fillRectangle(x: Int, y: Int, width: Int, height: Int) {
        context.fillRect(x, y, width, height)
    }

    override fun drawRectange(x: Int, y: Int, width: Int, height: Int) {
        context.drawRect(x, y, width, height)
    }

    override fun drawPoint(point: Point2D?) {
        context.fill(java.awt.geom.Rectangle2D.Double(point!!.x, point.y, 1.0, 1.0))
    }

    override fun setTextBaseline(baseline: IGraphics.TextBaseline?) {
        this.baseline = baseline!!
    }

    override fun setFont(font: CPFont?) {
        context.font = font!!.nativeFont
    }

    override fun getStringBounds(text: String?): Rectangle2D {
        val stringBounds = context.fontMetrics.getStringBounds(text, context)
        return Rectangle2D.Double(stringBounds.x, stringBounds.y, stringBounds.width, stringBounds.height)
    }

    override fun getStringWidth(text: String?): Float {
        return if (text != null) {
            context.fontMetrics.stringWidth(text).toFloat()
        } else {
            0f
        }
    }

    override fun getStringHeight(text: String?): Float {
        return if (text != null) {
            val metrics = context.font.getLineMetrics(text, context.fontRenderContext)
            metrics.height
        } else {
            0f
        }
    }

    override fun getAscent(): Double {
        return context.fontMetrics.ascent.toDouble()
    }

    override fun getDescent(): Double {
        return context.fontMetrics.descent.toDouble()
    }

    override fun getFontMetrics(): CPFontMetrics? {
        return SwingFontMetrics(context)
    }

    override fun drawString(text: String?, x: Float, y: Float) {
        var y = y
        when (baseline) {
            IGraphics.TextBaseline.Alphabetic -> {
            }
            IGraphics.TextBaseline.Top -> y -= context.fontMetrics.ascent.toFloat()
            IGraphics.TextBaseline.Bottom -> y += context.fontMetrics.descent.toFloat()
            IGraphics.TextBaseline.Middle -> {
                val fm = context.fontMetrics
                val lm = fm.getLineMetrics(text, context)
                y = y - (lm.ascent + lm.descent) / 2 + lm.ascent
            }
            IGraphics.TextBaseline.Hanging -> throw UnsupportedOperationException()
            IGraphics.TextBaseline.Ideographic -> throw UnsupportedOperationException()
            else -> throw IllegalArgumentException()
        }
        context.drawString(text, x, y)
    }

    override fun translate(x: Int, y: Int) {
        context.translate(x, y)
    }

    override fun rotate(theta: Double) {
        context.rotate(theta)
    }

    override fun beginPath() {
        path = GeneralPath()
    }

    override fun moveTo(x: Double, y: Double) {
        path!!.moveTo(x, y)
    }

    override fun lineTo(x: Double, y: Double) {
        path!!.lineTo(x, y)
    }

    override fun closePath() {
        path!!.closePath()
    }

    override fun stroke() {
        context.draw(path)
    }

    override fun fill() {
        context.fill(path)
    }

    init {
        this.context = context ?: HeadlessGraphics2D()
        this.context.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON)
        this.context.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON)
    }
}
