package org.mkui.labeling

import com.macrofocus.common.properties.PropertiesListener
import com.macrofocus.common.properties.PropertyEvent
import org.mkui.font.CPFont
import java.awt.*
import java.awt.font.LineBreakMeasurer
import java.awt.font.TextAttribute
import java.awt.geom.AffineTransform
import java.text.AttributedString
import java.text.BreakIterator
import javax.swing.*
import javax.swing.plaf.basic.BasicHTML
import javax.swing.plaf.basic.BasicLabelUI
import javax.swing.text.View

/**
 * Enhanced version of JLabel supporting text effects and word-wrap.
 */
open class EnhancedJLabel(data: EnhancedLabel) : JLabel() {
    private var data: EnhancedLabel? = null
    private var propertiesListener: PropertiesListener<String?>? = null

    constructor(text: String?) : this() {
        setText(text)
    }

    constructor() : this(EnhancedLabel()) {
        setUI(EnhancedLabelUI())
    }

    fun getMinimumSize(g2: Graphics2D): Dimension {
        return (ui as EnhancedLabelUI).getMinimumSize(g2, this)
        //        return super.getPreferredSize();
    }

    fun getPreferredSize(g2: Graphics2D): Dimension {
        return (ui as EnhancedLabelUI).getPreferredSize(g2, this)
    }

    override fun getPreferredSize(): Dimension {
        return super.getPreferredSize()
    }

    override fun getMinimumSize(): Dimension {
        return super.getMinimumSize()
    }

    fun setData(data: EnhancedLabel) {
        unregisterListener(this.data)
        this.data = data

//        super.setSize(new Dimension(data.getWidth(), data.getHeight()));
        if (data.isHTML) {
            super.setText(addTag(data.getText()))
        } else {
            super.setText(data.getText())
        }
        super.setVerticalAlignment(data.getVerticalAlignment())
        super.setHorizontalAlignment(data.getHorizontalAlignment())
        if (data.getFont() != null) {
            super.setFont(data.getFont()!!.nativeFont)
        }
        if (data.getForeground() != null) {
            super.setForeground(data.getForeground()!!.nativeColor)
        }
        if (data.getBackground() != null) {
            super.setBackground(data.getBackground()!!.nativeColor)
        }
        super.setBorder(
            BorderFactory.createEmptyBorder(
                data.getInsetTop(),
                data.getInsetLeft(),
                data.getInsetBottom(),
                data.getInsetRight()
            )
        )
        registerListener(data)
    }

    fun addTag(value: String?): String? {
        return if (value != null) {
            if (!value.startsWith("<html>")) {
                "<html>$value</html>"
            } else {
                value
            }
        } else {
            null
        }
    }

    protected open fun unregisterListener(data: EnhancedLabel?) {
        if (data != null) {
            data.getProperties()!!.removePropertiesListener(propertiesListener!!)
        }
    }

    protected open fun registerListener(data: EnhancedLabel) {
        if (propertiesListener == null) {
            propertiesListener = object : PropertiesListener<String?> {
                override fun propertyChanged(name: String?, event: PropertyEvent<Any?>) {
                    repaint()
                }
            }
        }
        data.getProperties()!!.addWeakPropertiesListener(propertiesListener!!)
    }

    fun setEffect(effect: EnhancedLabel.Effect?) {
        data!!.setEffect(effect!!)
    }

    var effectOpacity: Float
        get() = data!!.getEffectOpacity()
        set(effectOpacity) {
            data!!.getEffectOpacity()
        }
    var rendering: EnhancedLabel.Rendering?
        get() = data!!.getRendering()
        set(rendering) {
            data!!.setRendering(rendering!!)
        }
    var minimumCharactersToDisplay: Int?
        get() = data!!.getMinimumCharactersToDisplay()
        set(minimumCharactersToDisplay) {
            data!!.getMinimumCharactersToDisplay()
        }
    var isJustified: Boolean
        get() = data!!.isJustified()
        set(justified) {
            data!!.setJustified(justified)
        }

    /**
     * Sets the angle in degrees at which the text will be painted
     *
     * @param angle the angle in degrees
     */
    var angle: Double
        get() = data!!.getAngle()
        set(angle) {
            data!!.setAngle(angle)
        }

    override fun getFont(): Font? {
        return if (data != null && data!!.getFont() != null) {
            data!!.getFont()!!.nativeFont
        } else {
            super.getFont()
        }
    }

    override fun setFont(font: Font?) {
        val oldFont = getFont()
        if (data != null) {
            data!!.setFont(CPFont(font!!))
        } else {
            super.setFont(font)
        }
        firePropertyChange("font", oldFont, font)
    }

    protected inner class EnhancedLabelUI : BasicLabelUI() {
        private val useSwingPaintText = false
        private val paintIconR = Rectangle()
        private val paintTextR = Rectangle()
        override fun paint(g: Graphics, c: JComponent) {
            val g2 = g as Graphics2D
            val label = c as JLabel
            when (rendering) {
                EnhancedLabel.Rendering.WordWrap -> {
                    val y: Int
                    y = if (label.verticalAlignment != TOP) {
                        val d = getPreferredSize(g, c)
                        when (label.verticalAlignment) {
                            CENTER -> Math.max(0, (c.getHeight() - d.height) / 2)
                            BOTTOM -> Math.max(0, c.getHeight() - d.height)
                            else -> 0
                        }
                    } else {
                        0
                    }
                    g2.font = font
                    //                    g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
//                    g2.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
                    paintWordWrapText(
                        label, g2, g2.fontMetrics,
                        text, desiredSize.width, y, label.foreground, label.background
                    )
                }
                else -> {
                    when (label.horizontalAlignment) {
                        LEFT, LEADING -> {
                        }
                        CENTER -> {
                        }
                        RIGHT, TRAILING -> {
                        }
                        else -> {
                        }
                    }
                    when (label.verticalAlignment) {
                        TOP -> {
                        }
                        CENTER -> {
                        }
                        BOTTOM -> {
                        }
                        else -> {
                        }
                    }
                    val text = label.text
                    val icon = if (label.isEnabled) label.icon else label.disabledIcon
                    if (icon == null && text == null) {
                        return
                    }
                    val fm = label.getFontMetrics(g2.font)

                    // ToDo: Length of the text depends on the angle
                    val width: Int
                    val height: Int
                    width = c.getWidth()
                    height = c.getHeight()
                    val clippedText = layout(label, fm, width, height)
                    if (angle != 0.0) {
                        g2.rotate(Math.PI * angle / 180.0, paintTextR.centerX, paintTextR.centerY)
                    }
                    icon?.paintIcon(c, g2, paintIconR.x, paintIconR.y)
                    if (clippedText != null) {
                        val v: View? = c.getClientProperty(BasicHTML.propertyKey) as View?
                        if (v != null) {
                            v.paint(g2, paintTextR)
                        } else {
                            val textX = paintTextR.x
                            val textY = paintTextR.y + fm.ascent
                            if (label.isEnabled) {
                                paintEnabledText(label, g2, clippedText, textX, textY)
                            } else {
                                paintDisabledText(label, g2, clippedText, textX, textY)
                            }
                        }
                    }
                }
            }
            //            g2.dispose();
        }

        protected fun layout(
            label: JLabel, fm: FontMetrics,
            width: Int, height: Int
        ): String? {
            val insets = label.getInsets(null)
            val text = label.text
            val icon = if (label.isEnabled) label.icon else label.disabledIcon
            val paintViewR = Rectangle()
            paintViewR.x = insets.left
            paintViewR.y = insets.top
            paintViewR.width = width - (insets.left + insets.right)
            paintViewR.height = height - (insets.top + insets.bottom)
            paintIconR.height = 0
            paintIconR.width = paintIconR.height
            paintIconR.y = paintIconR.width
            paintIconR.x = paintIconR.y
            paintTextR.height = 0
            paintTextR.width = paintTextR.height
            paintTextR.y = paintTextR.width
            paintTextR.x = paintTextR.y
            return layoutCL(
                label, fm, text, icon, paintViewR, paintIconR,
                paintTextR
            )
        }

        fun getMinimumSize(g: Graphics, c: JComponent): Dimension {
            return when (rendering) {
                EnhancedLabel.Rendering.WordWrap -> {
                    val g2 = g as Graphics2D
                    val label = c as EnhancedJLabel
                    g2.font = font
                    //                    g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
                    //                    g2.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_OFF);
                    val fontMetrics = g2.getFontMetrics(font)
                    val d = paintWordWrapText(
                        label, null, fontMetrics,
                        text, label.desiredSize.width, 0,
                        null, null
                    )
                    Dimension(d.width, fontMetrics.height)
                }
                else -> getPreferredSize(c)
            }
        }

        override fun getMinimumSize(c: JComponent): Dimension {
            return when (rendering) {
                EnhancedLabel.Rendering.WordWrap -> {
                    val label = c as EnhancedJLabel
                    val d = paintWordWrapText(
                        label, null, label.getFontMetrics(label.font),
                        text, label.desiredSize.width, 0,
                        null, null
                    )
                    Dimension(d.width, getFontMetrics(font).height)
                }
                else -> getPreferredSize(c)
            }
        }

        fun getPreferredSize(g: Graphics, c: JComponent): Dimension {
            val insets = insets
            val size: Dimension
            when (rendering) {
                EnhancedLabel.Rendering.WordWrap -> {
                    val g2 = g as Graphics2D
                    val label = c as EnhancedJLabel
                    g2.font = font
                    //                    g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
//                    g2.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_OFF);
                    val d = paintWordWrapText(
                        label, null, g2.getFontMetrics(font),
                        text, label.desiredSize.width, 0,
                        null, null
                    )
                    size = Dimension(d.width, d.height)
                }
                else -> size = if (angle != 0.0) {
                    val d = getRotatedDimension(super.getPreferredSize(c))
                    Dimension(d.width + insets.left + insets.right, d.height + insets.top + insets.bottom)
                } else {
                    super.getPreferredSize(c)
                }
            }
            return size
        }

        override fun getPreferredSize(c: JComponent): Dimension {
            val insets = insets
            val size: Dimension
            size = when (rendering) {
                EnhancedLabel.Rendering.WordWrap -> {
                    val label = c as EnhancedJLabel
                    val d = paintWordWrapText(
                        label, null, label.getFontMetrics(label.font),
                        text, label.desiredSize.width, 0,
                        null, null
                    )
                    Dimension(d.width, d.height)
                }
                else -> if (angle != 0.0) {
                    val d = getRotatedDimension(super.getPreferredSize(c))
                    Dimension(d.width + insets.left + insets.right, d.height + insets.top + insets.bottom)
                } else {
                    super.getPreferredSize(c)
                }
            }
            return size
        }

        private fun getRotatedDimension(dimension: Dimension): Dimension {
            val at = createAffineTransform(dimension)
            val rect = at.createTransformedShape(Rectangle(dimension)).bounds
            return Dimension(rect.width, rect.height)
        }

        private fun createAffineTransform(dimension: Dimension): AffineTransform {
            val at = AffineTransform()
            at.rotate(Math.PI * angle / 180.0, dimension.getWidth() / 2.0, dimension.getHeight() / 2.0)
            return at
        }

        fun getUnrotatedPreferredSize(c: JComponent?): Dimension {
            return super.getPreferredSize(c)
        }

        override fun paintEnabledText(label: JLabel, g: Graphics, text: String, x: Int, y: Int) {
            paintText(label, g, text, x, y, label.foreground, label.background)
        }

        override fun paintDisabledText(label: JLabel, g: Graphics, text: String, x: Int, y: Int) {
            paintText(label, g, text, x, y, label.foreground, label.background)
        }

        override fun layoutCL(
            label: JLabel,
            fontMetrics: FontMetrics,
            text: String,
            icon: Icon?,
            viewR: Rectangle,
            iconR: Rectangle,
            textR: Rectangle
        ): String? {
            when (rendering) {
                EnhancedLabel.Rendering.Truncate -> {
                    val truncatedString = super.layoutCL(label, fontMetrics, text, icon, viewR, iconR, textR)
                    return if (text == truncatedString || truncatedString != null && truncatedString.length > 3 && (minimumCharactersToDisplay == null || truncatedString.length - 3 >= minimumCharactersToDisplay!!)) {
                        truncatedString
                    } else {
                        null
                    }
                }
                EnhancedLabel.Rendering.Clip -> {
                    val clippedString = super.layoutCL(label, fontMetrics, text, icon, viewR, iconR, textR)
                    if (text != clippedString && clippedString != null && minimumCharactersToDisplay != null && clippedString.length < minimumCharactersToDisplay!!) {
                        return null
                    }
                }
            }
            return text
        }

        private fun paintText(
            label: JLabel,
            g: Graphics,
            text: String?,
            x: Int,
            y: Int,
            foreground: Color,
            background: Color?
        ) {
            g.color = foreground
            if (background != null) {
                g.translate(x, y)
                when (data!!.getEffect()) {
                    EnhancedLabel.Effect.Shadow -> paintTextShadow(
                        g as Graphics2D, text,
                        background
                    )
                    EnhancedLabel.Effect.Glow -> paintTextGlow(g as Graphics2D, text, background)
                    EnhancedLabel.Effect.Outline -> paintTextGlow(g as Graphics2D, text, background)
                    EnhancedLabel.Effect.Emphasize -> paintTextEmphasize(g as Graphics2D, text, background)
                }
                g.translate(-x, -y)
            }
            if (useSwingPaintText) {
                super.paintEnabledText(label, g, text, x, y)
            } else {
                if (text != null) {
                    g.drawString(text, x, y)
                }
            }
        }

        private fun paintWordWrapText(
            label: JLabel,
            g: Graphics2D?,
            fontMetrics: FontMetrics,
            text: String?,
            desiredWidth: Int,
            yStart: Int,
            foreground: Color?,
            background: Color?
        ): Dimension {
            val insets = insets
            val width = desiredWidth + (insets.left + insets.right)
            //            final int noBorderWidth = width - (insets.left + insets.right);
            var w = insets.left + insets.right.toFloat()
            val x = insets.left.toFloat()
            var y = yStart + insets.top.toFloat()
            if (desiredWidth > 0 && text != null && text.length > 0) {
                if (g != null) {
//                    g.setFont(getFont());
//                    g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
//                    g.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_OFF);
                }
                val `as` = AttributedString(text)
                `as`.addAttribute(TextAttribute.FONT, font)
                `as`.addAttribute(TextAttribute.FOREGROUND, foreground)
                //                as.addAttribute(TextAttribute.BACKGROUND, background);
                val aci = `as`.iterator
                val fontRenderContext = fontMetrics.fontRenderContext
                //                if(g != null) {
//                    fontRenderContext = g.getFontRenderContext();
//                } else {
//                    if (this.frc == null) {
//                        this.frc = new FontRenderContext(null, true, false);
//                    }
//                    fontRenderContext = frc;
//                }
                val breakIterator = BreakIterator.getLineInstance()
                //                final BreakIterator breakIterator = BreakIterator.getWordInstance();
//                final BreakIterator breakIterator = BreakIterator.getSentenceInstance();
//                final BreakIterator breakIterator = BreakIterator.getCharacterInstance();
                val lbm = LineBreakMeasurer(aci, breakIterator, fontRenderContext)
                var max = 0f
                try {
                    while (lbm.position < aci.endIndex) {
                        var layout = lbm.nextLayout(desiredWidth.toFloat())
                        if (g != null && isJustified && layout.visibleAdvance > 0.80 * desiredWidth) layout =
                            layout.getJustifiedLayout(
                                desiredWidth.toFloat()
                            )
                        val ascent = layout.ascent
                        val visibleAdvance = layout.visibleAdvance
                        if (g != null) {
                            when (label.horizontalAlignment) {
                                LEFT, LEADING -> layout.draw(g, x, y + ascent)
                                CENTER -> {
                                    val leftMargin = insets.left.toFloat()
                                    val rightMargin = getWidth() - insets.right.toFloat()
                                    layout.draw(g, (leftMargin + rightMargin - visibleAdvance) / 2, y + ascent)
                                }
                                RIGHT, TRAILING -> {
                                    val rightMargin = getWidth() - insets.right.toFloat()
                                    layout.draw(g, rightMargin - visibleAdvance, y + ascent)
                                }
                                else -> layout.draw(g, x, y + ascent)
                            }
                        }
                        val leading = layout.leading
                        val descent = layout.descent
                        y += descent + leading + ascent
                        max = Math.max(max, visibleAdvance)
                    }
                } catch (e: ArrayIndexOutOfBoundsException) {
                    // See TREEMAP-517
                    e.printStackTrace()
                }
                w += max
            }
            return Dimension(
                Math.ceil(w.toDouble()).toInt(), Math.ceil(y.toDouble())
                    .toInt() + insets.bottom
            )
        }

        /**
         * Draw a string with a blur or shadow effect.
         *
         * @param g     graphics the graphics
         * @param text  the text to draw
         * @param color the effect color
         * @param x     x position
         * @param y     y position
         */
        private fun paintEffect(
            g: Graphics2D, text: String?, color: Color?,
            x: Double, y: Double
        ) {
            if (text != null) {

                // Save state
                val oldTextAAHint = g.getRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING)
                val oldComposite = g.composite
                val oldColor = g.color
                g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON)
                g.color = color
                g.translate(x, y)
                val additionalOffset = g.fontMetrics.height / 32
                when (data!!.getEffect()) {
                    EnhancedLabel.Effect.Emphasize -> {
                        val size1 = 1 + additionalOffset
                        var j = 0
                        while (j < size1) {
                            val distance = j * j.toDouble()
                            var alpha = effectOpacity
                            if (distance > 0.0) {
                                alpha = (0.5f / distance * effectOpacity).toFloat()
                            }
                            if (alpha > 1.0f) {
                                alpha = 1.0f
                            }
                            g.composite = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, alpha)
                            g.drawString(text, 0, j)
                            j++
                        }
                    }
                    EnhancedLabel.Effect.Shadow -> {
                        val size = 1 + additionalOffset
                        var i = 0
                        while (i < size) {
                            var j = 0
                            while (j < size) {
                                val distance = i * i + j * j.toDouble()
                                var alpha = effectOpacity
                                if (distance > 0.0) {
                                    alpha = (0.5f / distance * effectOpacity).toFloat()
                                }
                                if (alpha > 1.0f) {
                                    alpha = 1.0f
                                }
                                g.composite = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, alpha)
                                g.drawString(text, i, j)
                                j++
                            }
                            i++
                        }
                    }
                    EnhancedLabel.Effect.Glow -> {
                        val size3 = 1 + additionalOffset
                        var i = -size3
                        while (i <= size3) {
                            var j = -size3
                            while (j <= size3) {
                                val distance = i * i + j * j.toDouble()
                                var alpha = effectOpacity
                                if (distance > 0.0) {
                                    alpha = (1.0f / distance * effectOpacity).toFloat()
                                    if (alpha > 1.0f) {
                                        alpha = 1.0f
                                    }
                                    g.composite = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, alpha)
                                    g.drawString(text, i, j)
                                }
                                j++
                            }
                            i++
                        }
                    }
                    EnhancedLabel.Effect.Outline -> {
                        val oldAAHint = g.getRenderingHint(RenderingHints.KEY_ANTIALIASING)
                        g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON)
                        val frc = g.fontRenderContext
                        val gv = font!!.createGlyphVector(frc, text)

                        // draw the shadow
                        g.stroke = BasicStroke((2 + additionalOffset).toFloat())
                        g.composite = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, effectOpacity)
                        g.draw(gv.outline)
                        g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, oldAAHint)
                    }
                }

                // Restore graphics
                g.translate(-x, -y)
                g.composite = oldComposite
                g.color = oldColor
                if (oldTextAAHint != null) {
                    g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, oldTextAAHint)
                }
            }
        }

        /**
         * Draw a string with a drop shadow. The light angle is assumed to be 0
         * degrees, (i.e., window is illuminated from top) and the shadow size is 2,
         * with a 1 pixel vertical displacement. The shadow is intended to be subtle
         * to be usable in as many text components as possible. The shadow is
         * generated with multiple calls to draw string. This method paints the text
         * on coordinates 0, 1. If text should be painted elsewhere, a transform
         * should be applied to the graphics before passing it.
         *
         *
         * All modifications to the graphics object is restored by this method
         * before returning.
         *
         * @param g graphics component to paint on
         * @param s the string to paint
         * @param c the color of the shadow. Any alpha channel will be discarded
         */
        fun paintTextShadow(g: Graphics2D, s: String?, c: Color?) {
            paintEffect(g, s, removeAlpha(c), 1.0, 1.0)
        }

        fun paintTextEmphasize(g: Graphics2D, s: String?, c: Color?) {
            paintEffect(g, s, removeAlpha(c), 0.0, 1.0)
        }

        /**
         * Draw a string with a glow effect. Glow differs from a drop shadow in that
         * it isn't offset in any direction (i.e., not affected by
         * "lighting conditions").
         *
         *
         * All modifications to the graphics object is restored by this method
         * before returning.
         *
         * @param g    graphics component to paint on
         * @param s    the string to draw
         * @param glow the solid glow color. Do not use the alpha channel as this
         * will be discarded
         */
        fun paintTextGlow(g: Graphics2D, s: String?, glow: Color?) {
            paintEffect(g, s, removeAlpha(glow), 0.0, 0.0)
            //            paintEffect(g, s, removeAlpha(glow), _effectThickness, -_effectThickness, -_effectThickness, false);
        }

        /**
         * Remove the alpha channel from the passed color.
         *
         * @param c a color
         * @return the same color, but without an alpha channel
         */
        fun removeAlpha(c: Color?): Color? {
            var c = c
            if (c != null && c.alpha != 100) {
                c = Color(c.rgb)
            }
            return c
        }
    }

    override fun getMaximumSize(): Dimension {
        return super.getMaximumSize()
    }

    var desiredSize: Dimension
        get() = Dimension(data!!.getDesiredWidth(), data!!.getDesiredWidth())
        set(desiredSize) {
            data!!.setDesiredWidth(desiredSize.width)
            data!!.setDesiredHeight(desiredSize.height)
        }

    companion object {
        @JvmStatic
        fun main(args: Array<String>) {
            val frame = JFrame("EnhancedJLabel")
            frame.size = Dimension(600, 600)
            frame.contentPane.layout = BorderLayout()
            val label = EnhancedJLabel()
            label.text = "This is a test"
            label.font = label.font!!.deriveFont(72f)
            label.setEffect(EnhancedLabel.Effect.Shadow)
            label.background = Color.yellow
            label.rendering = EnhancedLabel.Rendering.Truncate
            label.verticalAlignment = EnhancedLabel.CENTER
            label.horizontalAlignment = EnhancedLabel.CENTER
            label.iconTextGap = 0
            frame.contentPane.add(label, BorderLayout.CENTER)
            frame.isVisible = true
            val timer = Timer(
                10
            ) { label.angle = (label.angle + 1) % 360 }
            timer.start()
        }
    }

    init {
        setUI(EnhancedLabelUI())
        setData(data)
    }
}
