package org.molap.exporter.arrow

import com.macrofocus.common.units.Quantity
import org.apache.arrow.memory.RootAllocator
import org.apache.arrow.vector.*
import org.apache.arrow.vector.dictionary.DictionaryProvider
import org.apache.arrow.vector.dictionary.DictionaryProvider.MapDictionaryProvider
import org.apache.arrow.vector.ipc.ArrowStreamWriter
import org.apache.arrow.vector.types.DateUnit
import org.apache.arrow.vector.types.FloatingPointPrecision
import org.apache.arrow.vector.types.TimeUnit
import org.apache.arrow.vector.types.pojo.ArrowType
import org.apache.arrow.vector.types.pojo.Field
import org.apache.arrow.vector.types.pojo.FieldType
import org.apache.arrow.vector.types.pojo.Schema
import org.molap.dataframe.DataFrame
import java.io.*
import java.math.BigDecimal
import java.net.URL
import java.sql.Time
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.LocalTime
import java.time.OffsetDateTime
import java.util.*
import kotlin.reflect.KClass

class ArrowDataFrameExporter<R,C,V>(val output: OutputStream) {
    constructor(file: File?) : this(FileOutputStream(file))

    /**
     * The root allocator. Typically only one created for a JVM.
     * Arrow provides a tree-based model for memory allocation.
     * The RootAllocator is created first, then all allocators are
     * created as children of that allocator. The RootAllocator is
     * responsible for being the master bookeeper for memory
     * allocations. All other allocators are created as children
     * of this tree. Each allocator can first determine whether
     * it has enough local memory to satisfy a particular request.
     * If not, the allocator can ask its parent for an additional
     * memory allocation.
     */
    private var allocator: RootAllocator? = null

    /**
     * The number of records in a record batch.
     * An Apache Arrow record batch is conceptually similar
     * to the Parquet row group. Parquet recommends a
     * disk/block/row group/file size of 512 to 1024 MB on HDFS.
     * 1 million rows x 100 columns of double will be about
     * 800 MB and also cover many use cases in machine learning.
     */
    private val batch = 1000000

    /*
     * When a field is dictionary encoded, the values are represented
     * by an array of Int32 representing the index of the value in the
     * dictionary. The Dictionary is received as one or more
     * DictionaryBatches with the id referenced by a dictionary attribute
     * defined in the metadata (Message.fbs) in the Field table.
     * The dictionary has the same layout as the type of the field would
     * dictate. Each entry in the dictionary can be accessed by its index
     * in the DictionaryBatches. When a Schema references a Dictionary id,
     * it must send at least one DictionaryBatch for this id.
     */
    val provider: DictionaryProvider = MapDictionaryProvider()

    var vectorSchemaRoot: VectorSchemaRoot? = null
    var writer : ArrowStreamWriter? = null

    @Throws(IOException::class)
    fun init(df: DataFrame<R, C, V>) {
        if (allocator == null) {
            allocate(Long.MAX_VALUE)
        }
        val schema = toArrowSchema(df)

        vectorSchemaRoot = VectorSchemaRoot.create(schema, allocator)

        writer = ArrowStreamWriter(vectorSchemaRoot, provider, output)
        writer!!.start()

//        vectorSchemaRoot!!.use { root ->
//            val arrowStreamWriter = ArrowStreamWriter(root, provider, output)
//            arrowStreamWriter.use { writer ->
//                writer.start()
//            }
//        }
    }

    @Throws(IOException::class)
    fun write(df: DataFrame<R, C, V>) {
        if(vectorSchemaRoot == null) {
            init(df)
        }

        val vectorSchemaRoot: VectorSchemaRoot = this.vectorSchemaRoot!!
        var writer : ArrowStreamWriter = this.writer!!

//        vectorSchemaRoot!!.use { vectorSchemaRoot ->
//            val writer = ArrowStreamWriter(vectorSchemaRoot, provider, output)
//            writer.use { writer ->
//                writer.start()
                val size: Int = df.rowCount
                var from = 0
                while (from < size) {
                    val count = Math.min(batch, size - from)
                    // set the batch row count
                    vectorSchemaRoot.rowCount = count
                    val fields =
                        vectorSchemaRoot.schema.fields
                    for (i in fields.indices) {
                        val field = fields[i]
                        val vector = vectorSchemaRoot.getVector(field.name)
                        val column: C = df.getColumnKey(i)
                        val type: KClass<*> = df.getColumnClass(column)
                        //                    if (Integer.class.equals(type)) {
//                        writeIntField(df, column, vector, from, count);
//                    } else if (Long.class.equals(type)) {
//                        writeLongField(df, column, vector, from, count);
//                    } else if (Double.class.equals(type)) {
//                        writeDoubleField(df, column, vector, from, count);
//                    } else if (Float.class.equals(type)) {
//                        writeFloatField(df, column, vector, from, count);
//                    } else if (Boolean.class.equals(type)) {
//                        writeBooleanField(df, column, vector, from, count);
//                    } else if (Byte.class.equals(type)) {
//                        writeByteField(df, column, vector, from, count);
//                    } else if (Short.class.equals(type)) {
//                        writeShortField(df, column, vector, from, count);
//                    } else if (Character.class.equals(type)) {
//                        writeCharField(df, column, vector, from, count);
//                    } else if (String.class.equals(type)) {
//                        writeStringField(df, column, vector, from, count);
//                    } else if (Date.class.equals(type)) {
//                        writeDateField(df, column, vector, from, count);
//                    } else if (Time.class.equals(type)) {
//                        writeTimeField(df, column, vector, from, count);
//                    } else if (DateTime.class.equals(type)) {
//                        writeDateTimeField(df, column, vector, from, count);
//                    } else if (Object.class.equals(type)) {
                        val clazz: KClass<*> = df.getColumnClass(column)
                        if (clazz == Int::class) {
                            writeIntObjectField(df, column, vector, from, count)
                        } else if (clazz == Long::class) {
                            writeLongObjectField(df, column, vector, from, count)
                        } else if (clazz == Double::class) {
                            writeDoubleObjectField(df, column, vector, from, count)
                        } else if (clazz == Float::class) {
                            writeFloatObjectField(df, column, vector, from, count)
                        } else if (clazz == Boolean::class) {
                            writeBooleanObjectField(df, column, vector, from, count)
                        } else if (clazz == Byte::class) {
                            writeByteObjectField(df, column, vector, from, count)
                        } else if (clazz == Short::class) {
                            writeShortObjectField(df, column, vector, from, count)
                        } else if (clazz == Char::class) {
                            writeCharObjectField(df, column, vector, from, count)
                        } else if (clazz == BigDecimal::class) {
                            writeDecimalField(df, column, vector, from, count)
                        } else if (clazz == String::class) {
                            writeStringField(df, column, vector, from, count)
                        } else if (clazz == LocalDate::class) {
                            writeDateField(df, column, vector, from, count)
                        } else if (clazz == LocalTime::class) {
                            writeTimeField(df, column, vector, from, count)
                        } else if (clazz == LocalDateTime::class) {
                            writeDateTimeField(df, column, vector, from, count)
                        } else if (clazz == URL::class) {
                            writeURLField(df, column, vector, from, count)
                        } else {
                            throw UnsupportedOperationException("Unsupported type: $type")
                        }
                        //                        case Array: {
//                            DataType etype = ((ArrayType) type).getComponentType();
//                            switch (etype.id()) {
//                                case Byte:
//                                    writeByteArrayField(df, vector, from, count);
//                                    break;
//                                default:
//                                    throw new UnsupportedOperationException("Unsupported type: " + type);
//                            }
//                            break;
//                        }
//                    } else {
//                        throw new UnsupportedOperationException("Unsupported type: " + type);
//                    }
                    }
                    writer.writeBatch()
                    from += batch
                }
//            }
//        }
    }

    fun close() {
        output.close()
    }

    /**
     * Creates the root allocator.
     * The RootAllocator is responsible for being the master
     * bookeeper for memory allocations.
     *
     * @param limit memory allocation limit in bytes.
     */
    fun allocate(limit: Long) {
        require(limit > 0) { "Invalid RootAllocator limit: $limit" }
        allocator = RootAllocator(limit)
    }

    /** Writes an int column.  */
    private fun <R, C, V> writeIntField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as IntVector
        var i = 0
        var j = from
        while (i < count) {
            vector[i] = (df.getValueAt(df.getRowKey(j), column) as Int)
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a nullable int column.  */
    private fun <R, C, V> writeIntObjectField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as IntVector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as Int?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x)
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a boolean column.  */
    private fun <R, C, V> writeBooleanField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as BitVector
        var i = 0
        var j = from
        while (i < count) {
            vector[i] = if (df.getValueAt(df.getRowKey(j), column) as Boolean) 1 else 0
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a nullable boolean column.  */
    private fun <R, C, V> writeBooleanObjectField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as BitVector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as Boolean?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, if (x) 1 else 0)
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a byte column.  */
    private fun <R, C, V> writeCharField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as UInt2Vector
        var i = 0
        var j = from
        while (i < count) {
            vector[i] = (df.getValueAt(df.getRowKey(j), column) as Char)
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a nullable char column.  */
    private fun <R, C, V> writeCharObjectField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as UInt2Vector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as Char?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x)
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a byte column.  */
    private fun <R, C, V> writeByteField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as TinyIntVector
        var i = 0
        var j = from
        while (i < count) {
            vector[i] = (df.getValueAt(df.getRowKey(j), column) as Byte)
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a nullable byte column.  */
    private fun <R, C, V> writeByteObjectField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as TinyIntVector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as Byte?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x)
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a short column.  */
    private fun <R, C, V> writeShortField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as SmallIntVector
        var i = 0
        var j = from
        while (i < count) {
            vector[i] = (df.getValueAt(df.getRowKey(j), column) as Short)
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a nullable short column.  */
    private fun <R, C, V> writeShortObjectField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as SmallIntVector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as Short?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x)
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a long column.  */
    private fun <R, C, V> writeLongField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as BigIntVector
        var i = 0
        var j = from
        while (i < count) {
            vector[i] = (df.getValueAt(df.getRowKey(j), column) as Long)
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a nullable long column.  */
    private fun <R, C, V> writeLongObjectField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as BigIntVector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as Long?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x)
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a float column.  */
    private fun <R, C, V> writeFloatField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as Float4Vector
        var i = 0
        var j = from
        while (i < count) {
            vector[i] = (df.getValueAt(df.getRowKey(j), column) as Float)
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a nullable float column.  */
    private fun <R, C, V> writeFloatObjectField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as Float4Vector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as Float?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x)
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a double column.  */
    private fun <R, C, V> writeDoubleField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as Float8Vector
        var i = 0
        var j = from
        while (i < count) {
            vector[i] = (df.getValueAt(df.getRowKey(j), column) as Double)
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a nullable double column.  */
    private fun <R, C, V> writeDoubleObjectField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as Float8Vector
        var i = 0
        var j = from
        while (i < count) {
            val value = df.getValueAt(df.getRowKey(j), column)
            val x = if(value is Quantity<*>) {
                (value as Quantity<*>).amount
            } else {
                if(value is Double?) {
                    value as Double?
                } else {
                    println("Value $value for column $column is of type ${value!!::class} when Double was expected")
                    null
                }
            }
//            val x = value as Double
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x)
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a string column.  */
    @Throws(UnsupportedEncodingException::class)
    private fun <R, C, V> writeStringField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as VarCharVector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as String?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x.toByteArray(charset("UTF-8")))
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a decimal column.  */
    @Throws(UnsupportedEncodingException::class)
    private fun <R, C, V> writeDecimalField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as DecimalVector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as BigDecimal?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x)
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a date column.  */
    private fun <R, C, V> writeDateField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as DateDayVector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as LocalDate?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x.toEpochDay().toInt())
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a time column.  */
    private fun <R, C, V> writeTimeField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as TimeNanoVector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as LocalTime?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x.toNanoOfDay())
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a datetime column.  */
    private fun <R, C, V> writeDateTimeField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as TimeStampMilliTZVector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as LocalDateTime?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x.toInstant(OffsetDateTime.now().offset).toEpochMilli())
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a byte array column.  */
    private fun <R, C, V> writeByteArrayField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as VarBinaryVector
        var i = 0
        var j = from
        while (i < count) {
            val bytes = df.getValueAt(df.getRowKey(j), column) as ByteArray?
            if (bytes == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, bytes)
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Writes a string column.  */
    @Throws(UnsupportedEncodingException::class)
    private fun <R, C, V> writeURLField(
        df: DataFrame<R, C, V>,
        column: C,
        fieldVector: FieldVector,
        from: Int,
        count: Int
    ) {
        fieldVector.setInitialCapacity(count)
        fieldVector.allocateNew()
        val vector = fieldVector as VarCharVector
        var i = 0
        var j = from
        while (i < count) {
            val x = df.getValueAt(df.getRowKey(j), column) as URL?
            if (x == null) {
                vector.setNull(i)
            } else {
                vector.setIndexDefined(i)
                vector.setSafe(i, x.toExternalForm().toByteArray(charset("UTF-8")))
            }
            i++
            j++
        }
        fieldVector.setValueCount(count)
    }

    /** Converts a smile schema to arrow schema.  */
    private fun toArrowSchema(schema: DataFrame<R, C, V>): Schema {
        val fields: MutableList<Field> = ArrayList()
        for (field in schema.columns()) {
            fields.add(toArrowField(schema, field))
        }
        return Schema(fields, null)
    }

    /** Converts a smile struct field to arrow field.  */
    private fun <R, C, V> toArrowField(schema: DataFrame<R, C, V>, field: C): Field {
        val columnClass: KClass<*> = schema.getColumnClass(field)
        if (Int::class == columnClass) {
            return Field(schema.getColumnName(field), FieldType(false, ArrowType.Int(32, true), null), null)
        } else if (Long::class == columnClass) {
            return Field(schema.getColumnName(field), FieldType(false, ArrowType.Int(64, true), null), null)
        } else if (Double::class == columnClass) {
            return Field(
                schema.getColumnName(field),
                FieldType(false, ArrowType.FloatingPoint(FloatingPointPrecision.DOUBLE), null),
                null
            )
        } else if (Float::class == columnClass) {
            return Field(
                schema.getColumnName(field),
                FieldType(false, ArrowType.FloatingPoint(FloatingPointPrecision.SINGLE), null),
                null
            )
        } else if (Boolean::class == columnClass) {
            return Field(schema.getColumnName(field), FieldType(false, ArrowType.Bool(), null), null)
        } else if (Byte::class == columnClass) {
            return Field(schema.getColumnName(field), FieldType(false, ArrowType.Int(8, true), null), null)
        } else if (Short::class == columnClass) {
            return Field(schema.getColumnName(field), FieldType(false, ArrowType.Int(16, true), null), null)
        } else if (Char::class == columnClass) {
            return Field(schema.getColumnName(field), FieldType(false, ArrowType.Int(16, false), null), null)
        } else if (BigDecimal::class == columnClass) {
            return Field(schema.getColumnName(field), FieldType.nullable(ArrowType.Decimal(28, 10, 128)), null)
        } else if (String::class == columnClass) {
            return Field(schema.getColumnName(field), FieldType.nullable(ArrowType.Utf8()), null)
        } else if (Date::class.java == columnClass) {
            return Field(schema.getColumnName(field), FieldType.nullable(ArrowType.Date(DateUnit.DAY)), null)
        } else if (Time::class == columnClass) {
            return Field(
                schema.getColumnName(field),
                FieldType.nullable(ArrowType.Time(TimeUnit.MILLISECOND, 32)),
                null
            )
            //        } else if (DateTime.class.equals(columnClass)) {
//            return new Field(schema.getColumnName(field), FieldType.nullable(new ArrowType.Timestamp(TimeUnit.MILLISECOND, ZoneOffset.UTC.getId())), null);
        } else if (URL::class == columnClass) {
            return Field(schema.getColumnName(field), FieldType.nullable(ArrowType.Utf8()), null)
        } else if (Any::class == columnClass) {
            val clazz: KClass<*> = schema.getColumnClass(field)
            if (clazz == Int::class) {
                return Field(schema.getColumnName(field), FieldType.nullable(ArrowType.Int(32, true)), null)
            } else if (clazz == Long::class) {
                return Field(schema.getColumnName(field), FieldType.nullable(ArrowType.Int(64, true)), null)
            } else if (clazz == Double::class) {
                return Field(
                    schema.getColumnName(field),
                    FieldType.nullable(ArrowType.FloatingPoint(FloatingPointPrecision.DOUBLE)),
                    null
                )
            } else if (clazz == Float::class) {
                return Field(
                    schema.getColumnName(field),
                    FieldType.nullable(ArrowType.FloatingPoint(FloatingPointPrecision.SINGLE)),
                    null
                )
            } else if (clazz == Boolean::class) {
                return Field(schema.getColumnName(field), FieldType.nullable(ArrowType.Bool()), null)
            } else if (clazz == Byte::class) {
                return Field(schema.getColumnName(field), FieldType.nullable(ArrowType.Int(8, true)), null)
            } else if (clazz == Short::class) {
                return Field(schema.getColumnName(field), FieldType.nullable(ArrowType.Int(16, true)), null)
            } else if (clazz == Char::class) {
                return Field(schema.getColumnName(field), FieldType.nullable(ArrowType.Int(16, false)), null)
            }
            //            case Array.class: {
//                DataType etype = ((ArrayType) schema.getColumnClass(field)).getComponentType();
//                switch (etype.id()) {
//                    case Integer:
//                        return new Field(schema.getColumnName(field),
//                                new FieldType(false, new ArrowType.List(), null),
//                                // children type
//                                Arrays.asList(new Field(null, new FieldType(false, new ArrowType.Int(32, true), null), null))
//                        );
//                    case Long:
//                        return new Field(schema.getColumnName(field),
//                                new FieldType(false, new ArrowType.List(), null),
//                                // children type
//                                Arrays.asList(new Field(null, new FieldType(false, new ArrowType.Int(64, true), null), null))
//                        );
//                    case Double:
//                        return new Field(schema.getColumnName(field),
//                                new FieldType(false, new ArrowType.List(), null),
//                                // children type
//                                Arrays.asList(new Field(null, new FieldType(false, new ArrowType.FloatingPoint(DOUBLE), null), null))
//                        );
//                    case Float:
//                        return new Field(schema.getColumnName(field),
//                                new FieldType(false, new ArrowType.List(), null),
//                                // children type
//                                Arrays.asList(new Field(null, new FieldType(false, new ArrowType.FloatingPoint(SINGLE), null), null))
//                        );
//                    case Boolean:
//                        return new Field(schema.getColumnName(field),
//                                new FieldType(false, new ArrowType.List(), null),
//                                // children type
//                                Arrays.asList(new Field(null, new FieldType(false, new ArrowType.Bool(), null), null))
//                        );
//                    case Byte:
//                        return new Field(schema.getColumnName(field), FieldType.nullable(new ArrowType.Binary()), null);
//                    case Short:
//                        return new Field(schema.getColumnName(field),
//                                new FieldType(false, new ArrowType.List(), null),
//                                // children type
//                                Arrays.asList(new Field(null, new FieldType(false, new ArrowType.Int(16, true), null), null))
//                        );
//                    case Char:
//                        return new Field(schema.getColumnName(field), FieldType.nullable(new ArrowType.Utf8()), null);
//                }
//                break;
//            }
//            case Struct: {
//                StructType children = (StructType) field.type;
//                return new Field(schema.getColumnName(field),
//                        new FieldType(false, new ArrowType.Struct(), null),
//                        // children type
//                        Arrays.stream(children.fields()).map(this::toArrowField).collect(Collectors.toList())
//                );
//            }
        }
        throw UnsupportedOperationException("Unsupported smile to arrow type conversion: $columnClass")
    }
}
