package org.molap.reader

import kotlinx.io.Source
import kotlinx.io.readLine

class UltimateTokenizer(
    reader: Source,
    private val quoteMatcher: CharMatcher,
    private val endLine: String,
    vararg delimeterMatchers: CharMatcher
) :
    Tokenizer {
    private val currentColumn: StringBuilder = StringBuilder()

    private var delimeterMatcher: CharMatcher? = null
    private val delimeterMatchers: Array<CharMatcher>
    private val surroundingSpacesNeedQuotes: Boolean
    private val commentMatcher: StringMatcher?

    private val columns: MutableList<String?>
    private val ready: Boolean

    private enum class TokenizerState {
        NORMAL, QUOTE_MODE
    }

    init {
        this.delimeterMatchers = delimeterMatchers as Array<CharMatcher>
        if (delimeterMatchers.size == 1) {
            delimeterMatcher = delimeterMatchers[0]
        }
        this.surroundingSpacesNeedQuotes = false
        this.commentMatcher = null

        columns = ArrayList<String?>()
        ready = readColumns(reader, columns)
    }

//    @Throws(IOException::class)
    private fun readColumns(reader: Source, columns: MutableList<String?>?): Boolean {
        if (columns == null) {
            throw NullPointerException("columns should not be null")
        }

        columns.clear()
        currentColumn.setLength(0)

        var line: String?
        do {
            line = reader.readLine()
            if (line == null) {
                return false
            } else if (line == endLine) {
                return false
            }
        } while (line!!.length == 0 || (commentMatcher != null && commentMatcher.isMatch(line)))

        line += NEWLINE

        var state = TokenizerState.NORMAL
        var quoteScopeStartingLine = -1
        var potentialSpaces = 0
        var charIndex = 0

        while (true) {
            val c = line!![charIndex]

            if (TokenizerState.NORMAL == state) {
                // When multiple delimiters are used (e.g. comma and semicolon, use the first one that has been detected
                // as the default.
                if (delimeterMatcher == null) {
                    for (matcher in delimeterMatchers) {
                        if (matcher.isMatch(c)) {
                            delimeterMatcher = matcher
                            break
                        }
                    }
                }

                if (delimeterMatcher != null && delimeterMatcher!!.isMatch(c)) {
                    if (!surroundingSpacesNeedQuotes) {
                        appendSpaces(currentColumn, potentialSpaces)
                    }
                    columns.add(if (currentColumn.length > 0) currentColumn.toString() else null)
                    potentialSpaces = 0
                    currentColumn.setLength(0)
                } else if (c == SPACE) {
                    potentialSpaces++
                } else if (c == NEWLINE) {
                    if (!surroundingSpacesNeedQuotes) {
                        appendSpaces(currentColumn, potentialSpaces)
                    }
                    columns.add(if (currentColumn.length > 0) currentColumn.toString() else null)
                    return true
                } else if (quoteMatcher.isMatch(c) && currentColumn.length == 0) {
                    state = TokenizerState.QUOTE_MODE
//                    quoteScopeStartingLine = reader.getLineNumber()

                    if (!surroundingSpacesNeedQuotes || currentColumn.length > 0) {
                        appendSpaces(currentColumn, potentialSpaces)
                    }
                    potentialSpaces = 0
                } else {
                    if (!surroundingSpacesNeedQuotes || currentColumn.length > 0) {
                        appendSpaces(currentColumn, potentialSpaces)
                    }

                    potentialSpaces = 0
                    currentColumn.append(c)
                }
            } else {
                if (c == NEWLINE) {
                    currentColumn.append(NEWLINE)

                    charIndex = -1
                    line = reader.readLine()
                    checkNotNull(line) {
                        println("unexpected end of file while reading quoted column")
//                        String.format(
//                            "unexpected end of file while reading quoted column beginning on line %d and ending on line %d",
//                            quoteScopeStartingLine, reader.getLineNumber()
//                        )
                    }
                    line += NEWLINE
                } else if (quoteMatcher.isMatch(c)) {
                    if (quoteMatcher.isMatch(line[charIndex + 1])) {
                        currentColumn.append(c)
                        charIndex++
                    } else {
                        state = TokenizerState.NORMAL
                        quoteScopeStartingLine = -1
                    }
                } else {
                    currentColumn.append(c)
                }
            }

            charIndex++ // read next char of the line
        }
    }

    override fun iterator(): Iterator<String?> {
        return columns.iterator()
    }

    override fun size(): Int {
        return if (ready) columns.size else -1
    }

    /**
     * Gets the last delimiter(s) that has/have been used
     *
     * @return the last delimiter(s)
     */
    fun getDelimeterMatchers(): Array<CharMatcher> {
        return if (delimeterMatcher != null) arrayOf(delimeterMatcher!!) else delimeterMatchers
    }

    companion object {
        private const val NEWLINE = '\n'
        private const val SPACE = ' '

        private fun appendSpaces(sb: StringBuilder, spaces: Int) {
            for (i in 0 until spaces) {
                sb.append(SPACE)
            }
        }
    }
}