package org.molap.tokenizer

import kotlin.jvm.JvmOverloads

class CharTokenizerModel @JvmOverloads constructor(
    private val str: String,
    private var delimiter: Char = '\t',
    private val retDelims: Boolean = false
) :
    TokenizerModel, Iterator<String?> {
    private var currentPosition = 0
    private var newPosition: Int
    private val maxPosition: Int
    private var delimsChanged = false

    /**
     * maxDelimChar stores the value of the delimiter character with the highest value. It is used to optimize the
     * detection of delimiter characters.
     */
    private var maxDelimChar = 0.toChar()

    /**
     * Set maxDelimChar to the highest char in the delimiter set.
     */
    private fun setMaxDelimChar() {
        maxDelimChar = delimiter
    }

    /**
     * Constructs a string tokenizer for the specified string. All characters in the `delim` argument are the
     * delimiters for separating tokens.
     *
     * If the `returnDelims` flag is `true`, then the delimiter characters are also returned as
     * tokens. Each delimiter is returned as a string of length one. If the flag is `false`, the delimiter
     * characters are skipped and only serve as separators between tokens.
     *
     * @param str          a string to be parsed.
     * @param delimiter        the delimiters.
     * @param retDelims flag indicating whether to return the delimiters as tokens.
     */
    /**
     * Constructs a string tokenizer for the specified string. The tokenizer uses the default delimiter set, which is
     * `"&nbsp;&#92;t&#92;n&#92;r&#92;f"`: the space character, the tab character, the newline character, the
     * carriage-return character, and the form-feed character. Delimiter characters themselves will not be treated as
     * tokens.
     *
     * @param str a string to be parsed.
     */
    /**
     * Constructs a string tokenizer for the specified string. The characters in the `delim` argument are the
     * delimiters for separating tokens. Delimiter characters themselves will not be treated as tokens.
     *
     * @param str   a string to be parsed.
     * @param delim the delimiters.
     */
    init {
        newPosition = -1
        maxPosition = str.length
        setMaxDelimChar()
    }

    /**
     * Skips delimiters starting from the specified position. If retDelims is false, returns the index of the first
     * non-delimiter character at or after startPos. If retDelims is true, startPos is returned.
     */
    private fun skipDelimiters(startPos: Int): Int {
        var position = startPos

        if (position < maxPosition) {
            if (str[position] == delimiter) {
                position++
            }
        }
        return position
    }

    /**
     * Skips ahead from startPos and returns the index of the next delimiter character encountered, or maxPosition if no
     * such delimiter is found.
     */
    private fun scanToken(startPos: Int): Int {
        var position = startPos
        while (position < maxPosition) {
            val c = str[position]
            if ((c <= maxDelimChar) && (delimiter == c)) break
            position++
        }
        if (retDelims && (startPos == position)) {
            val c = str[position]
            if ((c <= maxDelimChar) && (delimiter == c)) position++
        }
        return position
    }

    /**
     * Tests if there are more tokens available from this tokenizer's string. If this method returns <tt>true</tt>, then
     * a subsequent call to <tt>nextToken</tt> with no argument will successfully return a token.
     *
     * @return `true` if and only if there is at least one token in the string after the current position;
     * `false` otherwise.
     */
    override fun hasMoreTokens(): Boolean {
        /*
         * Temporary store this position and use it in the following
         * nextToken() method only if the delimiters have'nt been changed in
         * that nextToken() invocation.
         */
        newPosition = skipDelimiters(currentPosition)
        return (newPosition < maxPosition)
    }

    /**
     * Returns the next token from this string tokenizer.
     *
     * @return the next token from this string tokenizer.
     */
    override fun nextToken(): String {
        /*
         * If next position already computed in hasMoreElements() and
         * delimiters have changed between the computation and this invocation,
         * then use the computed value.
         */

        currentPosition = if ((newPosition >= 0 && !delimsChanged)) newPosition else skipDelimiters(currentPosition)

        /* Reset these anyway */
        delimsChanged = false
        newPosition = -1

        if (currentPosition > maxPosition) throw NoSuchElementException()
        val start = currentPosition
        currentPosition = scanToken(currentPosition)
        return str.substring(start, currentPosition)
    }

    /**
     * Returns the next token in this string tokenizer's string. First, the set of characters considered to be
     * delimiters by this <tt>StringTokenizer</tt> object is changed to be the characters in the string <tt>delim</tt>.
     * Then the next token in the string after the current position is returned. The current position is advanced beyond
     * the recognized token.  The new delimiter set remains the default after this call.
     *
     * @param delim the new delimiters.
     * @return the next token, after switching to the new delimiter set.
     */
    fun nextToken(delim: Char): String {
        delimiter = delim

        /* delimiter string specified, so set the appropriate flag. */
        delimsChanged = true

        setMaxDelimChar()
        return nextToken()
    }

    override fun hasNext(): Boolean {
        return hasMoreTokens()
    }

    override fun next(): String? {
        return nextToken()
    }

    /**
     * Calculates the number of times that this tokenizer's `nextToken` method can be called before it
     * generates an exception. The current position is not advanced.
     *
     * @return the number of tokens remaining in the string using the current delimiter set.
     * @see java.util.StringTokenizer.nextToken
     */
    override fun countTokens(): Int {
        var count = 0
        var currpos = currentPosition
        while (currpos < maxPosition) {
            currpos = skipDelimiters(currpos)
            if (currpos > maxPosition) break
            currpos = scanToken(currpos)
            count++
        }
        return count
    }
}
