package org.molap.reader

import com.macrofocus.common.logging.Logging
import kotlinx.io.IOException
import kotlinx.io.Source
import kotlinx.io.readLine
import org.molap.tokenizer.CharTokenizerModel
import org.molap.tokenizer.TokenizerModel
import org.molap.typemap.DefaultTypemapModel
import org.molap.typemap.TypemapModel
import kotlin.jvm.JvmOverloads
import kotlin.math.min
import kotlin.reflect.KClass
import kotlin.reflect.KFunction

/**
 * Responsible for loading tabular data.
 */
class GenericDataReader @JvmOverloads constructor(
    reader: Source?,
    private val delim: Char,
    hasColumnHeader: Boolean = true,
    hasClassHeader: Boolean = true,
    className: String? = "String",
    startLineNumber: Int = 0,
    private val endLineNumber: Int = Int.MAX_VALUE,
    columnHeaderLines: Int = 1,
    typemapModel: TypemapModel = DefaultTypemapModel()
) :
    DataReader {
    override fun getColumnName(column: Int): Any? {
        return labels[column]
    }

    override fun getClass(column: Int): KClass<*>? {
        return classes[column]
    }

    override val rowCount: Int
        get() = Companion.rowCount

    override fun nextRow() {
        try {
            if (first == null) {
                val line: String? = lnr.readLine()
                if (line != null) {
                    current = CharTokenizerModel(line, delim)
                } else {
                    current = null
                }
            } else {
                current = first
                first = null
            }

            row++
            column = -1
        } catch (e: IOException) {
            Logging.instance.process(e)
        }
    }

    override fun nextColumn(): Any? {
        column++
        token = current!!.nextToken()
        var value: Any? = null
        if (!(token == null || token == "" || token == "n.a." || token == "N/A" || token == "x" || token == "NaN" || token == "---" || token == "Infinity" || token == "-Infinity" || token == "NULL" || token == "#VALUE!")) {
            if (token!!.startsWith("\"") && token!!.endsWith("\"")) {
                token = token!!.substring(1, token!!.length - 1)
            }
            try {
                if (classes[column] == String::class) {
                    value = token
                } else if (classes[column] == Byte::class) {
                    value = token!!.toByte()
                } else if (classes[column] == Double::class) {
                    value = token!!.toDouble()
                } else if (classes[column] == Float::class) {
                    value = token!!.toFloat()
                } else if (classes[column] == Int::class) {
                    value = token!!.toInt()
                } else if (classes[column] == Long::class) {
                    value = token!!.toLong()
                } else if (classes[column] == Short::class) {
                    value = token!!.toShort()
                } else {
                    val initargs = arrayOf<Any>(token!!)
                    val constructor = constructors[column]!!
//                    value = constructor.call(initargs)
                }

                if (value == null) {
                    Logging.instance
                        .process(RuntimeException("Instantiation failed for " + token + " as " + constructors[column]))
                }
            } catch (e: Exception) {
                Logging.instance.process(
                    Exception(
                        "Instantiation failed for column " + getColumnName(column) + " of " + token + " as " + constructors[column],
                        e
                    )
                )
            }
        } else {
            value = null
        }

        return value
    }

    override fun hasMoreRow(): Boolean {
        try {
            return !lnr.exhausted() && lineNumber <= endLineNumber
        } catch (e: IOException) {
            Logging.instance.process(e)
            return false
        }
    }

    override fun hasMoreColumn(): Boolean {
        try {
            return current != null && current!!.hasMoreTokens()
        } catch (e: Exception) {
            Logging.instance.process(Exception("Row:$row, column:$column", e))
            return false
        }
    }

    override fun close() {
        try {
            lnr.close()
        } catch (e: IOException) {
            Logging.instance.process(e)
        }
    }

    override val lineNumber: Int
        get() = -1

//    fun setClasses(classes: Array<KClass<*>?>) {
//        this.classes = classes
//        for (i in 0 until columnCount) {
//            try {
//                val arg: Array<KClass<*>> = arrayOf<KClass<*>>(String::class)
//                constructors[i] = classes[i].constructors.(*arg)
//            } catch (e: NoSuchMethodException) {
//                Logging.instance.process(e)
//            }
//        }
//    }

    override val isHasClassHeader: Boolean
        get() = true

    private var labels: Array<Any?>
    private var classes: Array<KClass<*>?>
    private lateinit var constructors: Array<KFunction<Any>?>
    override var columnCount: Int = 0
    override var row: Int
        private set
    override var column: Int
        private set
    private val lnr: Source = reader!!
    private var current: TokenizerModel? = null
    private var first: TokenizerModel? = null
    private var token: String? = null

    /**
     * Creates a new `GenericDataReader` instance.
     *
     * @param reader          a `java.io.Reader` value
     * @param delim           a `String` value
     * @param hasColumnHeader a `boolean` value
     * @param hasClassHeader  a `boolean` value
     * @param className       a `String` value
     */
    init {
        try {
            while (lineNumber < startLineNumber) {
                lnr.readLine()
            }

            val columnHeader: Array<TokenizerModel?> = arrayOfNulls<TokenizerModel>(columnHeaderLines)
            var classHeader: TokenizerModel? = null
            if (hasColumnHeader) {
                for (l in 0 until columnHeaderLines) {
                    columnHeader[l] = CharTokenizerModel(lnr.readLine()!!, delim)
                }
            }
            if (hasClassHeader) {
                classHeader = CharTokenizerModel(lnr.readLine()!!, delim)
            }

            first = null
            if (hasColumnHeader && hasClassHeader) {
                for (l in 0 until columnHeaderLines) {
                    this.columnCount = min(columnHeader[l]!!.countTokens(), classHeader!!.countTokens())
                }
            } else if (hasColumnHeader) {
                this.columnCount = Int.MAX_VALUE
                for (l in 0 until columnHeaderLines) {
                    this.columnCount = min(columnHeader[l]!!.countTokens(), this.columnCount)
                }
            } else if (hasClassHeader) {
                this.columnCount = classHeader!!.countTokens()
            } else {
                // Try guessing the number of columns
                first = CharTokenizerModel(lnr.readLine()!!, delim)
                this.columnCount = first!!.countTokens()
            }

            labels = arrayOfNulls(columnCount)
            classes = arrayOfNulls<KClass<*>>(columnCount)
            constructors = arrayOfNulls<KFunction<Any>>(columnCount)
            for (i in 0 until columnCount) {
                if (hasColumnHeader) {
                    if (columnHeaderLines == 1) {
                        var token: String = columnHeader[0]!!.nextToken()!!
                        if (token.startsWith("\"") && token.endsWith("\"")) {
                            token = token.substring(1, token.length - 1)
                        }
                        labels[i] = token
                    } else {
                        val s = arrayOfNulls<String>(columnHeaderLines)
                        for (l in 0 until columnHeaderLines) {
                            var token: String = columnHeader[l]!!.nextToken()!!
                            if (token.startsWith("\"") && token.endsWith("\"")) {
                                token = token.substring(1, token.length - 1)
                            }
                            s[l] = token
                        }
                        labels[i] = MultilineHeaderModel(s)
                    }
                } else {
                    labels[i] = "Column $i"
                }
                var c = if (hasClassHeader) {
                    classHeader!!.nextToken()
                } else {
                    className
                }

                try {
                    val cl: KClass<*> = typemapModel.getClass(c)!!
                    classes[i] = cl
                } catch (e: Exception) {
                    Logging.instance.process(e)
                }

                try {
                    fun newString(s: String): String = s

                    constructors[i] = ::newString
                    } catch (e: Exception) {
                    Logging.instance.process(e)
                }
            }
        } catch (e: IOException) {
            Logging.instance.process(e)
            this.columnCount = 0
            labels = arrayOfNulls<Any>(0)
            classes = arrayOfNulls<KClass<*>>(0)
        }

        row = -1
        column = -1
    }

    companion object {
        private const val rowCount = -1
    }
}
