package org.molap.reader

import com.macrofocus.common.logging.Logging
import kotlinx.io.IOException
import kotlinx.io.Source
import kotlinx.io.readLine
import org.molap.typemap.Builder
import org.molap.typemap.DefaultTypemapModel
import org.molap.typemap.InvalidValueException
import org.molap.typemap.TypemapModel
import kotlin.math.min
import kotlin.reflect.KClass

/**
 * Responsible for loading tabular data.
 */
class PerfectDataReader(
    reader: Source?,
    tokenizerFactory: TokenizerFactory,
    hasColumnHeader: Boolean,
    hasClassHeader: Boolean,
    className: String?,
    startLineNumber: Int,
    endLineNumber: Int,
    columnHeaderLines: Int,
    typemapModel: TypemapModel,
    vararg builders: Builder<*>
) :
    DataReader {
    private var labels: Array<Any?>
    private var classes: Array<KClass<*>?>
    private var builders: Array<Builder<*>?>? = null
    override val rowCount: Int = -1
    override var columnCount: Int = 0
    override var row: Int
        private set
    override var column: Int
        private set
    private val lnr: Source
    private var current: Tokenizer? = null
    private var currentIterator: Iterator<String?>? = null
    private var first: Tokenizer? = null
    private var second: Tokenizer? = null
    private var next: Tokenizer? = null
    private val tokenizerFactory: TokenizerFactory
    private var token: String? = null
    private val startLineNumber: Int
    private val endLineNumber: Int

    override var isHasClassHeader: Boolean = false

//    private var fileLoader: FileLoader? = null

    private var exceptionThrown = false

    private val nullStrings: HashSet<String> = HashSet<String>(
        mutableListOf<String>(
            "",
            "n.a.",
            "N/A",
            "x",
            "NaN",
            "---",
            "Infinity",
            "-Infinity",
            "NULL",
            "#VALUE!",
            "#N/A"
        )
    )

    constructor(reader: Source?, tokenizerFactory: TokenizerFactory) : this(
        reader,
        tokenizerFactory,
        true,
        true,
        "String",
        0,
        Int.MAX_VALUE
    )

    constructor(
        reader: Source?,
        tokenizerFactory: TokenizerFactory,
        hasColumnHeader: Boolean,
        hasClassHeader: Boolean
    ) : this(reader, tokenizerFactory, hasColumnHeader, hasClassHeader, "String", 0, Int.MAX_VALUE)

    constructor(
        reader: Source?,
        tokenizerFactory: TokenizerFactory,
        hasColumnHeader: Boolean,
        hasClassHeader: Boolean,
        className: String?
    ) : this(reader, tokenizerFactory, hasColumnHeader, hasClassHeader, className, 0, Int.MAX_VALUE)

    constructor(
        reader: Source?,
        tokenizerFactory: TokenizerFactory,
        hasColumnHeader: Boolean,
        hasClassHeader: Boolean,
        className: String?,
        startLineNumber: Int,
        endLineNumber: Int
    ) : this(
        reader,
        tokenizerFactory,
        hasColumnHeader,
        hasClassHeader,
        className,
        startLineNumber,
        endLineNumber,
        1,
        DefaultTypemapModel()
    )

    /**
     * Creates a new `GenericDataReader` instance.
     * @param reader           a `java.io.Reader` value
     * @param tokenizerFactory a `String` value
     * @param hasColumnHeader  a `boolean` value
     * @param hasClassHeader   a `boolean` value
     * @param className        a `String` value
     */
    init {
        var hasClassHeader = hasClassHeader
        this.tokenizerFactory = tokenizerFactory
        this.startLineNumber = startLineNumber
        this.endLineNumber = endLineNumber

        // If we have multiple datasets following each other, then we shouldn't buffer
        lnr = reader!!

        try {
            while (lineNumber < startLineNumber) {
                lnr.readLine()
            }

            val columnHeader = arrayOfNulls<Tokenizer>(columnHeaderLines)
            var classHeader: Tokenizer? = null
            var firstLine: Tokenizer? = null
            if (hasColumnHeader) {
                for (l in 0 until columnHeaderLines) {
                    if (firstLine == null) {
                        firstLine = tokenizerFactory.createTokenizer(lnr)
                        columnHeader[l] = firstLine
                    } else {
                        columnHeader[l] = tokenizerFactory.createTokenizer(lnr)
                    }
                }
            }

            if (!hasClassHeader) {
                // Let's look if the second line only contains classes we know about
                val secondLine: Tokenizer = tokenizerFactory.createTokenizer(lnr)

                var known = true
                for (token in secondLine) {
                    if (token != null) {
                        if (typemapModel.findBuilder(token) == null) {
                            known = false
                            break
                        }
                    }
                }

                if ((!hasColumnHeader || columnHeader[0]!!.size() === secondLine.size()) && known) {
                    hasClassHeader = true
                    classHeader = secondLine
                } else {
                    second = secondLine
                }
            } else {
                classHeader = tokenizerFactory.createTokenizer(lnr)
            }
            val classHeaderIterator = classHeader?.iterator()
            this.isHasClassHeader = hasClassHeader

            first = null
            if (hasColumnHeader && hasClassHeader) {
                for (l in 0 until columnHeaderLines) {
                    this.columnCount = min(columnHeader[l]!!.size(), classHeader!!.size())
                }
            } else if (hasColumnHeader) {
                this.columnCount = Int.MAX_VALUE
                for (l in 0 until columnHeaderLines) {
                    this.columnCount = min(columnHeader[l]!!.size(), this.columnCount)
                }
            } else if (hasClassHeader) {
                this.columnCount = classHeader!!.size()
            } else {
                // Try guessing the number of columns
                first = tokenizerFactory.createTokenizer(lnr)
                this.columnCount = first!!.size()
            }

            if (firstLine != null) {
                var lastValidColumn = -1
                val firstLineTokenizer: Tokenizer = firstLine

                var column = -1
                for (token in firstLineTokenizer) {
                    column++
                    if (token != null) {
                        lastValidColumn = column
                    }
                }
                columnCount = lastValidColumn + 1
            }

            labels = arrayOfNulls(columnCount)
            classes = arrayOfNulls<KClass<*>>(columnCount)
            this.builders = arrayOfNulls<Builder<*>?>(columnCount)
            if (builders != null) {
                for (i in builders.indices) {
                    val builder: Builder<*> = builders[i]
                    this.builders!![i] = builder
                }
            }
            val columnHeaderIterator = Array<Iterator<String?>>(columnHeader.size) { c ->
                columnHeader[c]!!.iterator()
            }
            for (i in 0 until columnCount) {
                if (hasColumnHeader) {
                    if (columnHeaderLines == 1) {
                        val token = columnHeaderIterator[0].next()
                        labels[i] = token
                    } else {
                        val s = arrayOfNulls<String>(columnHeaderLines)
                        for (l in 0 until columnHeaderLines) {
                            val token = columnHeaderIterator[l].next()
                            s[l] = token
                        }
                        labels[i] = MultilineHeaderModel(s)
                    }
                } else {
                    labels[i] = "Column $i"
                }
                var c = if (hasClassHeader) {
                    classHeaderIterator!!.next()
                } else {
                    className
                }

                try {
                    val builder: Builder<*>? = if (c == null) {
                        typemapModel.getBuilder("String")
                    } else {
                        typemapModel.getBuilder(c)
                    }
                    if (this.builders!![i] == null) {
                        this.builders!![i] = builder
                    }
                    classes[i] = this.builders!![i]!!.type
                } catch (e: Exception) {
                    Logging.instance.process(e)
                }
            }
        } catch (e: IOException) {
            Logging.instance.process(e)
            this.columnCount = 0
            labels = emptyArray()
            classes = emptyArray()
        }

        row = -1
        column = -1
    }

    override fun getColumnName(column: Int): Any? {
        return labels[column]
    }

    override fun getClass(column: Int): KClass<*>? {
        return classes[column]
    }

    override fun nextRow() {
        if (first != null) {
            current = first
            currentIterator = current!!.iterator()
            first = null
        } else if (second != null) {
            current = second
            currentIterator = current!!.iterator()
            second = null
        } else {
            current = next
            currentIterator = current!!.iterator()
        }

        row++

        column = -1
    }

    override fun nextColumn(): Any? {
        column++
        token = currentIterator!!.next()
        var value: Any? = null
        if (!(token == null || (token is String && nullStrings.contains(token)))) {
            try {
//                if (builders!![column] is DefaultTypemapModel.ImageBuilderFactory.ImageBuilder) {
//                    (builders[column] as DefaultTypemapModel.ImageBuilderFactory.ImageBuilder).setFileLoader(fileLoader)
//                }
                value = builders!![column]!!.create(token)
            } catch (e: Exception) {
                if (!exceptionThrown) {
                    exceptionThrown = true
                    var message = "Incorrect value \"$token\""
                    if (builders != null && builders!![column] != null) {
                        message += " for type " + builders!![column]?.name
                    }
//                    message += " at line " + lnr.getLineNumber() + " for column " + getColumnName(column)
                    val t: InvalidValueException = InvalidValueException(message, e)
                    if (Logging.instance.process(t)) {
                        throw RuntimeException(t)
                    } else {
                        return null
                    }

                    //                    Logging.getInstance().process(new Exception("Instantiation failed for column " + getColumnName(column) + " of " + token + " as " + constructors[column] + " at line " + lnr.getLineNumber(), e));
                }
            }
        } else {
            value = null
        }

        return value
    }

    override fun hasMoreRow(): Boolean {
        try {
            if (lineNumber <= endLineNumber) {
                val c1 = second != null
                val c2 = current != null
                val c3 = current != null && current!!.size() >= 0

                //                return second != null || (current != null && current.size() >= 0 && lnr.ready()) || (tokenizerFactory.getEndLine() == null && lnr.ready());
                if (second != null) {
                    return true
                } else {
                    if (current != null && current!!.size() > 0) {
                        next = tokenizerFactory.createTokenizer(lnr)
                        return next!!.size() >= 0
                    } else {
                        if (tokenizerFactory.getEndLine() == null) {
                            next = tokenizerFactory.createTokenizer(lnr)
                            return next!!.size() >= 0
                        } else {
                            return false
                        }
                    }
                }
            } else {
                return false
            }
        } catch (e: IOException) {
            Logging.instance.process(e)
            return false
        }
    }

    override fun hasMoreColumn(): Boolean {
        try {
            return current != null && currentIterator!!.hasNext()
        } catch (e: Exception) {
            Logging.instance.process(Exception("Row:$row, column:$column", e))
            return false
        }
    }

    override fun close() {
        try {
            lnr.close()
        } catch (e: IOException) {
            Logging.instance.process(e)
        }
    }

    override val lineNumber: Int
        get() = -1

//    fun setFileLoader(fileLoader: FileLoader?) {
//        this.fileLoader = fileLoader
//    }
}
