package org.molap.geojson

import com.macrofocus.common.units.Quantity
import org.locationtech.jts.geom.*
import org.molap.dataframe.DataFrame
import org.molap.datetime.DateTimeTz
import org.molap.exporter.AbstractDataFrameWriter
import org.molap.exporter.JSONDataFrameWriter
import java.io.IOException
import java.io.Writer
import java.text.NumberFormat
import java.util.*
import kotlin.reflect.KClass


class GeoJSONDataFrameWriter(private val writer: Writer, newLine: String) : AbstractDataFrameWriter() {
    private val columnNames: MutableMap<Int, String>
    private val columnClass: MutableMap<Int, Class<*>>
    private val newLine: String
    private var row = 0
    private var column = 0
    private var geoColumn = -1
    private var valueWritten = false
    @Throws(IOException::class)
    override fun writeColumnName(name: String?, hasMore: Boolean) {
        columnNames[column] = name!!
        column++
    }

    override fun includeType(): Boolean {
        return true
    }

    @Throws(IOException::class)
    override fun writeType(name: KClass<*>, hasMore: Boolean) {
        columnClass[column] = name.java
        if (geoColumn < 0 && Geometry::class.java.isAssignableFrom(name.java)) {
            geoColumn = column
        }
        column++
    }

    @OptIn(ExperimentalStdlibApi::class)
    @Throws(IOException::class)
    override fun <Row,Column,Value> writeCell(value: Any?, dataFrame: DataFrame<Row,Column,Value>, rowKey: Row, columnKey: Column) {
        if (column != geoColumn || (column == 0 && geoColumn == 0)) {
            if (column == 0) {
                if (row > 2) {
                    writer.write(newLine)
                    writer.write("\t\t\t}")
                    writer.write(newLine)
                    writer.write("\t\t},")
                }
                writer.write(newLine)
                writer.write("\t\t{")
                writer.write(newLine)
                writer.write("\t\t\t\"type\": \"Feature\",")
                if (geoColumn >= 0) {
                    val geometry = dataFrame.getValueAt(rowKey, dataFrame.getColumnKey(geoColumn)) as Geometry?
                    if (geometry != null) {
                        writer.write(newLine)
                        writer.write("\t\t\t\"geometry\": ")
                        write(writer, geometry)
                        writer.write(",")
                    }
                }
                writer.write(newLine)
                writer.write("\t\t\t\"properties\": {")
                writer.write(newLine)
            } else {
                if (value != null && valueWritten) {
                    writer.write(", ")
                    writer.write(newLine)
                }
            }
            if (value != null && column != geoColumn) {
                writer.write("\t\t\t\t\"" + columnNames[column] + "\": ")
                var str: String?
                if (value is Date) {
                    str = dateToString(dataFrame, columnKey, value as Date?)
                } else {
                    str = value.toString()
                }
                if (value is Number) {
                    str = numberToString(value as Number?)
                } else if(value is Quantity<*>) {
                    str = numberToString(value.amount)
                } else if(value is DateTimeTz) {
                    str = "\"" + value.toISOString() + "\""
                } else if (value is Boolean) {
                    str = value.toString()
                } else if(value is ByteArray) {
                    str = value.toHexString(format = HexFormat { number.prefix = "0x" })
                } else {
                    if (value is Array<*>) {
                        val array = value
                        str = "["
                        for (i in array.indices) {
                            val s = array[i]
                            str += "\"" + JSONDataFrameWriter.quote(s as String).toString() + "\""
                            if (i < array.size - 1) {
                                str += ", "
                            }
                        }
                        str += "]"
                    } else {
                        str = "\"" + JSONDataFrameWriter.quote(str).toString() + "\""
                    }
                }
                writer.write(str)
                valueWritten = true
            }
        }
    }

    @Throws(IOException::class)
    fun write(output: Writer, geom: Geometry) {
        val nf = NumberFormat.getInstance(Locale.ROOT)
        nf.isGroupingUsed = false
        nf.maximumFractionDigits = 6
        nf.minimumFractionDigits = 0
        if (geom is Point) {
            output.append("{\"type\":\"Point\",\"coordinates\":")
            write(output, nf, geom.coordinate)
            output.append("}")
            return
        } else if (geom is Polygon) {
            output.append("{\"type\":\"Polygon\",\"coordinates\":")
            write(output, nf, geom)
            output.append("}")
            return
        } else if (geom is LineString) {
            output.append("{\"type\":\"LineString\",\"coordinates\":")
            write(output, nf, geom.coordinateSequence)
            output.append("}")
            return
        } else if (geom is MultiPoint) {
            output.append("{\"type\":\"MultiPoint\",\"coordinates\":")
            write(output, nf, geom.coordinates)
            output.append("}")
            return
        } else if (geom is MultiLineString) {
            val v = geom
            output.append("{\"type\":\"MultiLineString\",\"coordinates\":[")
            for (i in 0 until v.numGeometries) {
                if (i > 0) {
                    output.append(',')
                }
                write(output, nf, v.getGeometryN(i).coordinates!!)
            }
            output.append("]}")
        } else if (geom is MultiPolygon) {
            val v = geom
            output.append("{\"type\":\"MultiPolygon\",\"coordinates\":[")
            for (i in 0 until v.numGeometries) {
                if (i > 0) {
                    output.append(',')
                }
                write(output, nf, v.getGeometryN(i) as Polygon)
            }
            output.append("]}")
        } else if (geom is GeometryCollection) {
            val v = geom
            output.append("{\"type\":\"GeometryCollection\",\"geometries\":[")
            for (i in 0 until v.numGeometries) {
                if (i > 0) {
                    output.append(',')
                }
                write(output, v.getGeometryN(i))
            }
            output.append("]}")
        } else {
            throw UnsupportedOperationException("unknown: $geom")
        }
    }

    @Throws(IOException::class)
    protected fun write(output: Writer, nf: NumberFormat, coord: Coordinate?) {
        output.write('['.code)
        output.write(nf.format(coord!!.x))
        output.write(','.code)
        output.write(nf.format(coord.y))
        output.write(']'.code)
    }

    @Throws(IOException::class)
    protected fun write(output: Writer, nf: NumberFormat, coordseq: CoordinateSequence?) {
        output.write('['.code)
        val dim = coordseq!!.dimension
        for (i in 0 until coordseq.size()) {
            if (i > 0) {
                output.write(','.code)
            }
            output.write('['.code)
            output.write(nf.format(coordseq.getOrdinate(i, 0)))
            output.write(','.code)
            output.write(nf.format(coordseq.getOrdinate(i, 1)))
            if (dim > 2) {
                val v = coordseq.getOrdinate(i, 2)
                if (!java.lang.Double.isNaN(v)) {
                    output.write(','.code)
                    output.write(nf.format(v))
                }
            }
            output.write(']'.code)
        }
        output.write(']'.code)
    }

    @Throws(IOException::class)
    protected fun write(output: Writer, nf: NumberFormat, coord: Array<Coordinate>) {
        output.write('['.code)
        for (i in coord.indices) {
            if (i > 0) {
                output.append(',')
            }
            write(output, nf, coord[i])
        }
        output.write(']'.code)
    }

    @Throws(IOException::class)
    protected fun write(output: Writer, nf: NumberFormat, p: Polygon) {
        output.write('['.code)
        val exteriorRing = p.exteriorRing
        if(exteriorRing != null) {
            write(output, nf, exteriorRing.coordinateSequence)
            for (i in 0 until p.getNumInteriorRing()) {
                output.append(',')
                write(output, nf, p.getInteriorRingN(i)!!.coordinateSequence)
            }
        }
        output.write(']'.code)
    }

    @Throws(IOException::class)
    override fun nextColumn(hasMore: Boolean) {
        column++
    }

    @Throws(IOException::class)
    override fun nextRow() {
        column = 0
        row++
        valueWritten = false
    }

    @Throws(IOException::class)
    override fun close() {
        if (row > 2) {
            writer.write("}")
            writer.write(newLine)
            writer.write("\t\t}")
        }
        writer.write(newLine)
        writer.write("\t]")
        writer.write(newLine)
        writer.write("}")
        writer.close()
    }

    init {
        columnNames = HashMap()
        columnClass = HashMap()
        this.newLine = newLine
        writer.write("{")
        writer.write(newLine)
        writer.write("\t\"type\": \"FeatureCollection\",")
        writer.write(newLine)
        writer.write("\t\"features\": [")
    }
}
