/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.db.jdbc

import org.locationtech.jts.geom.Geometry
import java.math.BigDecimal
import java.sql.*
import java.sql.Date
import java.util.*
import javax.swing.table.TableModel

abstract class JDBCDatabaseDriver protected constructor() : DatabaseDriver {
    @Throws(SQLException::class)
    override fun getConnection(url: String?, username: String?, password: String?): Connection? {
        var connection: Connection? = null
        connection = if (username != null) {
            try {
                DriverManager.getConnection(url, username, password)
            } catch (e: SQLException) {
                try {
                    DriverManager.getConnection(url)
                } catch (e1: SQLException) {
                    throw e
                }
            }
        } else {
            DriverManager.getConnection(url)
        }
        return connection
    }

    @Throws(SQLException::class)
    open fun getConnection(url: String?, info: Properties?): Connection? {
        var connection: Connection? = null
        connection = try {
            DriverManager.getConnection(url, info)
        } catch (e: SQLException) {
            try {
                DriverManager.getConnection(url)
            } catch (e1: SQLException) {
                throw e
            }
        }
        return connection
    }

    @Throws(SQLException::class)
    override fun getColumnType(metaData: ResultSetMetaData, column: Int): Class<*>? {
        val type = metaData.getColumnType(column)
        return JDBCTypes.getType(type)
    }

    open fun getColumnType(type: Class<*>, precision: Int): String? {
        return if (type == Boolean::class.java) {
            "boolean"
        } else if (type == Int::class.java) {
            "int"
        } else if (type == Long::class.java) {
            "bigint"
        } else if (type == Float::class.java) {
            "real"
        } else if (type == Double::class.java) {
            "double"
        } else if (type == BigDecimal::class.java) {
            "decimal"
        } else if (type == String::class.java) {
            if (precision > 0) {
                if (precision > 255) {
                    "text"
                } else {
                    "varchar($precision)"
                }
            } else {
                // TEXT
                "varchar"
            }
        } else if (type == Date::class.java) {
            "date"
        } else if (type == ByteArray::class.java) {
            "binary"
        } else if (Geometry::class.java.isAssignableFrom(type)) {
            "binary"
        } else {
            throw UnsupportedOperationException("Unknown type $type")
            //            return "blob";
        }
    }

    @Throws(SQLException::class)
    override fun getColumnValue(rs: ResultSet, columnType: Class<*>?, column: Int): Any? {
        var `object`: Any?
        if (columnType == String::class.java) {
            `object` = rs.getString(column)
        } else if (columnType == Int::class.java) {
            `object` = rs.getInt(column)
        } else if (columnType == Byte::class.java) {
            `object` = rs.getByte(column)
        } else if (columnType == Short::class.java) {
            `object` = rs.getShort(column)
        } else if (columnType == Long::class.java) {
            `object` = rs.getLong(column)
        } else if (columnType == Float::class.java) {
            `object` = rs.getFloat(column)
        } else if (columnType == Double::class.java) {
            `object` = rs.getDouble(column)
        } else if (columnType == BigDecimal::class.java) {
            `object` = rs.getBigDecimal(column)
        } else if (columnType == Boolean::class.java) {
            `object` = rs.getBoolean(column)
        } else if (columnType == Date::class.java) {
            `object` = rs.getDate(column)
        } else if (columnType == Timestamp::class.java) {
            `object` = rs.getTimestamp(column)
        } else {
            `object` = rs.getObject(column)
            val cl: Class<*>? = `object`?.javaClass
            System.err.println("JDBCDatabaseDriver: Unknown class $columnType for value $cl")
        }
        if (rs.wasNull()) {
            `object` = null
        }
        return `object`
    }

    override fun getURL(host: String?, schema: String?, username: String?, password: String?): String {
        var url = "jdbc:$prefix:"
        if (host != null) {
            url += "//$host/"
        }
        if (schema != null) {
            url += schema
        }
        return url
    }

    override fun getQuery(query: String): String {
        return "$query;"
    }

    @Throws(SQLException::class)
    override fun getDatabases(host: String?, username: String?, password: String?): Iterable<String> {
        val conn = getConnection(getURL(host, null, username, password), username, password)
        val databases: MutableList<String> = ArrayList()
        val schemas = conn!!.metaData.schemas
        while (schemas.next()) {
            databases.add(schemas.getString(1))
        }
        return databases
    }

    @Throws(SQLException::class)
    override fun getTables(url: String?, database: String?, username: String?, password: String?): Iterable<String?>? {
        val conn = getConnection(url, username, password)
        val types = arrayOf("TABLE", "VIEW")
        return getTables(conn, types)
    }

    @Throws(SQLException::class)
    private fun getTables(conn: Connection?, types: Array<String>): List<String?> {
        val tables: MutableList<String?> = ArrayList()
        val meta = conn!!.metaData
        val result = meta.getTables(null, null, null, types)
        while (result.next()) {
            val schema = result.getString("TABLE_SCHEM")
            val table = result.getString("TABLE_NAME")
            tables.add(if (schema != null) "$schema.$table" else table)
        }
        return tables
    }

    @Throws(SQLException::class)
    override fun createTable(table: String?, tableModel: TableModel, md: ResultSetMetaData, primary: ResultSet?): String {
        val result = StringBuffer()
        result.append("CREATE TABLE ")
        result.append(table)
        result.append(" ( ")
        for (column in 0 until tableModel.columnCount) {
            val i = column + 1
            if (i != 1) result.append(',')
            result.append('`'.toString() + md.getColumnName(i) + '`')
            result.append(' ')

//                String type = md.getColumnTypeName(i), md.getPrecision(i));
            val type = getColumnType(tableModel.getColumnClass(column), md.getPrecision(i))
            result.append(type)

//                if (md.getPrecision(i) < 65535) {
//                    result.append('(');
//                    result.append(md.getPrecision(i));
//                    if (md.getScale(i) > 0) {
//                        result.append(',');
//                        result.append(md.getScale(i));
//                    }
//                    result.append(") ");
//                } else
//                    result.append(' ');
//
//                if (this.isNumeric(md.getColumnType(i))) {
//                    if (!md.isSigned(i))
//                        result.append("UNSIGNED ");
//                }
            if (md.isNullable(i) == ResultSetMetaData.columnNoNulls) result.append(" NOT NULL") else result.append(" NULL")
            if (md.isAutoIncrement(i)) result.append(" auto_increment")
        }
        if (primary != null) {
            var first = true
            while (primary.next()) {
                if (first) {
                    first = false
                    result.append(',')
                    result.append("PRIMARY KEY(")
                } else result.append(",")
                result.append('`'.toString() + primary.getString("COLUMN_NAME") + '`')
            }
            if (!first) result.append(')')
        }
        result.append(")")
        return result.toString()
    }

    @Throws(SQLException::class)
    override fun createTable(table: String?, tableModel: TableModel, vararg primary: String): String {
        val result = StringBuffer()
        result.append("CREATE TABLE `")
        result.append(table)
        result.append("` (")
        for (column in 0 until tableModel.columnCount) {
            val i = column + 1
            if (i != 1) result.append(',')
            result.append("`")
            result.append(tableModel.getColumnName(column))
            result.append("` ")

//                String type = md.getColumnTypeName(i), md.getPrecision(i));
            val type = getColumnType(tableModel, column)
            result.append(type)

//            if (md.isNullable(i) == ResultSetMetaData.columnNoNulls)
//                result.append(" NOT NULL");
//            else
//                result.append(" NULL");
//            if (md.isAutoIncrement(i))
//                result.append(" auto_increment");
        }
        if (primary != null) {
            var first = true
            for (s in primary) {
                if (first) {
                    first = false
                    result.append(',')
                    result.append("PRIMARY KEY(")
                } else result.append(",")
                result.append("`$s`")
            }
            if (!first) result.append(')')
        }
        result.append(")")
        return result.toString()
    }

    private fun getColumnType(tableModel: TableModel, column: Int): String? {
        return if (tableModel.getColumnClass(column) == String::class.java) {
            var min = Int.MAX_VALUE
            var max = 0
            var missing = 0
            for (row in 0 until tableModel.rowCount) {
                val value = tableModel.getValueAt(row, column) as String
                if (value != null) {
                    val length = value.length
                    if (length > max) {
                        max = length
                    }
                    if (length < min) {
                        min = length
                    }
                } else {
                    missing++
                }
            }
            if (missing == 0 && min > 0 && min == max) {
                "char($max)"
            } else if (max < 256) {
                "varchar(" + 255 + ")"
            } else {
                "text"
            }
        } else {
            getColumnType(tableModel.getColumnClass(column), 0)
        }
    }

    @Throws(SQLException::class)
    override fun dropTable(connection: Connection, table: String) {
        val types = arrayOf("TABLE")
        if (getTables(connection, types).contains(table)) {
            connection.prepareStatement("DROP TABLE $table").execute()
        }
    }

    override fun generateParseDate(date: String): String {
        return "PARSEDATETIME('$date', 'yyyy-MM-dd')"
    }

    override fun generateIsNull(value: String): String {
        return "$value IS NULL"
    }

    override val isNetworkEnabled: Boolean
        get() = true
    protected abstract val className: String
    protected abstract val prefix: String

    companion object {
        @JvmStatic
        protected fun exist(className: String?): Boolean {
            val cl: Class<*>?
            cl = try {
//            cl = Class.forName(className, false, ClassLoader.getSystemClassLoader());
                Class.forName(className, false, Thread.currentThread().contextClassLoader)
            } catch (e: ClassNotFoundException) {
                return false
            }
            return cl != null
        }
    }

    init {
        try {
            Class.forName(className).newInstance()
        } catch (e: InstantiationException) {
            e.printStackTrace()
        } catch (e: IllegalAccessException) {
            e.printStackTrace()
        } catch (e: ClassNotFoundException) {
            e.printStackTrace()
        }
    }
}