/*
 * Copyright (c) 2009 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.db.jdbc

import java.io.*
import java.sql.Connection
import java.sql.DriverManager
import java.sql.SQLException
import java.util.*

class ODBCDatabaseDriver : JDBCDatabaseDriver() {
    private val nullWriter = PrintWriter(object : OutputStream() {
        @Throws(IOException::class)
        override fun write(b: Int) {
        }
    })
    override val className: String
        get() = Companion.className
    override val prefix: String
        get() = "odbc"

    override fun toString(): String {
        return "ODBC"
    }

    override fun getURL(host: String?, schema: String?, username: String?, password: String?): String {
        var url = "jdbc:$prefix:"
        if (schema != null) {
            url += schema
        }
        return url
    }

    @Throws(SQLException::class)
    override fun getConnection(url: String?, username: String?, password: String?): Connection? {
        val writer = DriverManager.getLogWriter()
        DriverManager.setLogWriter(nullWriter)
        val connection = super.getConnection(url, username, password)
        DriverManager.setLogWriter(writer)
        return connection
    }

    @Throws(SQLException::class)
    override fun getConnection(url: String?, info: Properties?): Connection? {
        val writer = DriverManager.getLogWriter()
        DriverManager.setLogWriter(nullWriter)
        val connection = super.getConnection(url, info)
        DriverManager.setLogWriter(writer)
        return connection
    }

    @Throws(SQLException::class)
    override fun getDatabases(host: String?, username: String?, password: String?): Iterable<String> {
        val list = ArrayList<String>()
        list.addAll(getDatabases("HKEY_CURRENT_USER\\SOFTWARE\\ODBC\\ODBC.INI"))
        list.addAll(getDatabases("HKEY_LOCAL_MACHINE\\SOFTWARE\\ODBC\\ODBC.INI"))
        return list
    }

    private fun getDatabases(registryKey: String): List<String> {
        val command = arrayOf("reg", "query", registryKey)
        val dsnList: MutableList<String> = ArrayList()
        try {
            val process = Runtime.getRuntime().exec(command)
            val stream = BufferedReader(InputStreamReader(process.inputStream))
            var dsn = ""
            while (stream.readLine().also { dsn = it } != null) {
                if (dsn.indexOf(registryKey + "\\") != -1) {
                    val dsnName = dsn.substring(dsn.lastIndexOf("\\") + 1)
                    if ("ODBC Data Sources" != dsnName) {
                        dsnList.add(dsnName)
                    }
                }
            }
        } catch (e: IOException) {
            e.printStackTrace()
        }
        return dsnList
    }

    @Throws(SQLException::class)
    override fun getTables(url: String?, database: String?, username: String?, password: String?): Iterable<String?>? {
        val writer = DriverManager.getLogWriter()
        DriverManager.setLogWriter(nullWriter)
        val tables = super.getTables(url, database, username, password)
        DriverManager.setLogWriter(writer)
        return tables
    }

    override val isNetworkEnabled: Boolean
        get() = false

    companion object {
        private const val className = "sun.jdbc.odbc.JdbcOdbcDriver"
        fun exist(): Boolean {
            return JDBCDatabaseDriver.exist(className)
        }
    }
}