/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.db.jdbc

import org.locationtech.jts.geom.*
import org.postgis.PGboxbase
import org.postgis.PGgeometry
import java.sql.ResultSet
import java.sql.ResultSetMetaData
import java.sql.SQLException
import java.sql.Types

class PostGISDatabaseDriver : PostgreSQLDatabaseDriver() {
    override val className: String
        get() = Companion.className
    override val prefix: String
        get() = "postgresql_postGIS"

    override fun toString(): String {
        return "PostGIS"
    }

    @Throws(SQLException::class)
    override fun getColumnType(metaData: ResultSetMetaData, column: Int): Class<*>? {
        return if (metaData.getColumnType(column) == Types.OTHER) {
            Geometry::class.java
        } else {
            super.getColumnType(metaData, column)
        }
    }

    @Throws(SQLException::class)
    override fun getColumnValue(rs: ResultSet, columnType: Class<*>?, column: Int): Any? {
        return if (columnType == Geometry::class.java) {
            val o = rs.getObject(column)
            convert2JTS(o)
        } else {
            super.getColumnValue(rs, columnType, column)
        }
    }

    /**
     * Converts the native geometry object to a JTS `Geometry`.
     *
     * @param object native database geometry object (depends on the JDBC spatial
     * extension of the database)
     * @return JTS geometry corresponding to geomObj.
     */
    fun convert2JTS(`object`: Any?): Geometry? {
        var `object` = `object` ?: return null

        // in some cases, Postgis returns not PGgeometry objects
        // but org.postgis.Geometry instances.
        // This has been observed when retrieving GeometryCollections
        // as the result of an SQL-operation such as Union.
        if (`object` is org.postgis.Geometry) {
            `object` = PGgeometry(`object`)
        }
        return if (`object` is PGgeometry) {
            val geom = `object`
            var out: Geometry? = null
            out = when (geom.getGeoType()) {
                org.postgis.Geometry.POINT -> convertPoint(geom.getGeometry() as org.postgis.Point)
                org.postgis.Geometry.LINESTRING -> convertLineString(
                    geom
                        .getGeometry() as org.postgis.LineString
                )
                org.postgis.Geometry.POLYGON -> convertPolygon(geom.getGeometry() as org.postgis.Polygon?)
                org.postgis.Geometry.MULTILINESTRING -> convertMultiLineString(
                    geom
                        .getGeometry() as org.postgis.MultiLineString
                )
                org.postgis.Geometry.MULTIPOINT -> convertMultiPoint(geom.getGeometry() as org.postgis.MultiPoint)
                org.postgis.Geometry.MULTIPOLYGON -> convertMultiPolygon(
                    geom
                        .getGeometry() as org.postgis.MultiPolygon
                )
                org.postgis.Geometry.GEOMETRYCOLLECTION -> convertGeometryCollection(
                    geom
                        .getGeometry() as org.postgis.GeometryCollection
                )
                else -> throw RuntimeException("Unknown type of PGgeometry")
            }
            out
        } else if (`object` is PGboxbase) {
            convertBox(`object`)
        } else {
            throw IllegalArgumentException(
                "Can't convert object of type "
                        + `object`.javaClass.canonicalName
            )
        }
    }

    private fun convertBox(box: PGboxbase): Geometry {
        val ll: org.postgis.Point = box.getLLB()
        val ur: org.postgis.Point = box.getURT()
        val ringCoords = arrayOfNulls<Coordinate>(5)
        ringCoords[0] = Coordinate(ll.x, ll.y)
        ringCoords[1] = Coordinate(ur.x, ll.y)
        ringCoords[2] = Coordinate(ur.x, ur.y)
        ringCoords[3] = Coordinate(ll.x, ur.y)
        ringCoords[4] = Coordinate(ll.x, ll.y)
        val shell: LinearRing = geometryFactory
            .createLinearRing(ringCoords.requireNoNulls())
        return geometryFactory.createPolygon(shell, null)
    }

    private fun convertGeometryCollection(collection: org.postgis.GeometryCollection): Geometry {
        val geometries: Array<org.postgis.Geometry> = collection.getGeometries()
        val jtsGeometries = arrayOfNulls<Geometry>(geometries.size)
        for (i in geometries.indices) {
            jtsGeometries[i] = convert2JTS(geometries[i])
        }
        return geometryFactory.createGeometryCollection(jtsGeometries.requireNoNulls())
    }

    private fun convertMultiPolygon(pgMultiPolygon: org.postgis.MultiPolygon): Geometry {
        val polygons = arrayOfNulls<Polygon>(pgMultiPolygon.numPolygons())
        for (i in polygons.indices) {
            val pgPolygon: org.postgis.Polygon = pgMultiPolygon.getPolygon(i)
            polygons[i] = convertPolygon(pgPolygon) as Polygon?
        }
        val out = geometryFactory
            .createMultiPolygon(polygons.requireNoNulls())
        out.SRID = pgMultiPolygon.srid
        return out
    }

    private fun convertMultiPoint(pgMultiPoint: org.postgis.MultiPoint): Geometry {
        val points = arrayOfNulls<Point>(pgMultiPoint.numPoints())
        for (i in points.indices) {
            points[i] = convertPoint(pgMultiPoint.getPoint(i))
        }
        val out: MultiPoint = geometryFactory
            .createMultiPoint(points.requireNoNulls())
        out.SRID = pgMultiPoint.srid
        return out
    }

    private fun convertMultiLineString(mlstr: org.postgis.MultiLineString): Geometry? {
        val lstrs = arrayOfNulls<LineString>(mlstr.numLines())
        for (i in 0 until mlstr.numLines()) {
            lstrs[i] = geometryFactory.createLineString(
                toJTSCoordinates(
                    mlstr
                        .getLine(i).getPoints()
                )
            )
        }
        val out = geometryFactory.createMultiLineString(lstrs.requireNoNulls())
        out.SRID = mlstr.srid
        return out
    }

    protected fun convertPolygon(polygon: org.postgis.Polygon?): Geometry? {
        if(polygon != null) {
            val ring = polygon.getRing(0)
            if(ring != null) {
                val shell: LinearRing = geometryFactory.createLinearRing(toJTSCoordinates(ring.getPoints()))
                var out: Polygon?
                if (polygon.numRings() > 1) {
                    val rings = arrayOfNulls<LinearRing>(
                        polygon
                            .numRings() - 1
                    )
                    for (r in 1 until polygon.numRings()) {
                        rings[r - 1] = geometryFactory
                            .createLinearRing(
                                toJTSCoordinates(
                                    polygon.getRing(r)
                                        .getPoints()
                                )
                            )
                    }
                    out = geometryFactory.createPolygon(shell, rings.requireNoNulls())
                } else {
                    out = geometryFactory.createPolygon(shell, null)
                }
                out.SRID = polygon.srid
                return out
            } else {
                return null
            }
        } else {
            return null
        }
    }

    protected fun convertPoint(pnt: org.postgis.Point): Point {
        val g = geometryFactory.createPoint(Coordinate(pnt.x, pnt.y))
        g.SRID = pnt.getSrid()
        return g
    }

    protected fun convertLineString(lstr: org.postgis.LineString): LineString {
        val out: LineString = geometryFactory
            .createLineString(toJTSCoordinates(lstr.getPoints()))
        out.SRID = lstr.getSrid()
        return out
    }

    private fun toJTSCoordinates(points: Array<org.postgis.Point>): Array<Coordinate> {
        val coordinates = arrayOfNulls<Coordinate>(points.size)
        for (i in points.indices) {
            coordinates[i] = Coordinate(points[i].x, points[i].y)
        }
        return coordinates.requireNoNulls()
    }

    companion object {
        private const val className = "org.postgis.DriverWrapper"
        val geometryFactory = GeometryFactory()
        fun exist(): Boolean {
            return JDBCDatabaseDriver.Companion.exist(className)
        }
    }
}