/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.db.jdbc

import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.io.WKBReader
import javax.swing.table.TableModel
import java.lang.Exception
import java.sql.Connection
import java.sql.ResultSet
import java.sql.ResultSetMetaData
import java.sql.SQLException

class H2DatabaseDriver : JDBCDatabaseDriver() {
    @Throws(SQLException::class)
    override fun getColumnType(metaData: ResultSetMetaData, column: Int): Class<*>? {
        return if ("VARBINARY" == metaData.getColumnTypeName(column)) {
            Geometry::class.java
        } else {
            super.getColumnType(metaData, column)
        }
    }

    @Throws(SQLException::class)
    override fun getColumnValue(rs: ResultSet, columnType: Class<*>?, column: Int): Any? {
        return if (columnType == Geometry::class.java) {
            try {
                val bytes = rs.getBytes(column)
                if (bytes != null) {
                    WKBReader().read(bytes)
                } else {
                    null
                }
            } catch (e: Exception) {
                e.printStackTrace()
                null
            }
        } else {
            super.getColumnValue(rs, columnType, column)
        }
    }

    override fun getURL(host: String?, schema: String?, username: String?, password: String?): String {
        var url = "jdbc:$prefix:"
        if (schema != null) {
            url += schema
        }
        return url
    }

    @Throws(SQLException::class)
    override fun createTable(table: String?, tableModel: TableModel, md: ResultSetMetaData, primary: ResultSet?): String {
        val result = StringBuffer()
        val sql = StringBuffer()
        result.append("CREATE TABLE ")
        result.append(table)
        result.append(" ( ")
        for (column in 0 until tableModel.columnCount) {
            val i = column + 1
            if (i != 1) result.append(',')
            val columnName = md.getColumnName(i)
            if (columnName.equals("location", ignoreCase = true) ||
                columnName.equals("case", ignoreCase = true) ||
                columnName.equals("group", ignoreCase = true) ||
                columnName.equals("check", ignoreCase = true) ||
                columnName.equals("comment", ignoreCase = true)
            ) {
                result.append("\"" + columnName.replace('.', '_') + "\"")
            } else {
                result.append(columnName.replace('.', '_'))
            }
            result.append(' ')
            val type = getColumnType(tableModel.getColumnClass(column), md.getPrecision(i))
            result.append(type)
        }
        result.append(")")
        return result.toString()
    }

    @Throws(SQLException::class)
    override fun dropTable(connection: Connection, table: String) {
        super.dropTable(connection, table.toUpperCase())
    }

    override val className: String
        get() = Companion.className
    override val prefix: String
        get() = "h2"

    override fun toString(): String {
        return "H2"
    }

    companion object {
        private const val className = "org.h2.Driver"
        fun exist(): Boolean {
            return JDBCDatabaseDriver.Companion.exist(className)
        }
    }
}