/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.db.jdbc

import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.io.WKBReader
import java.lang.Exception
import java.sql.Connection
import java.sql.ResultSet
import java.sql.ResultSetMetaData
import java.sql.SQLException

class HSQLDBDatabaseDriver : JDBCDatabaseDriver() {
    @Throws(SQLException::class)
    override fun getColumnType(metaData: ResultSetMetaData, column: Int): Class<*>? {
        return if ("VARBINARY" == metaData.getColumnTypeName(column)) {
            Geometry::class.java
        } else {
            super.getColumnType(metaData, column)
        }
    }

    @Throws(SQLException::class)
    override fun getColumnValue(rs: ResultSet, columnType: Class<*>?, column: Int): Any? {
        return if (columnType == Geometry::class.java) {
            try {
                val bytes = rs.getBytes(column)
                if (bytes != null) {
                    WKBReader().read(bytes)
                } else {
                    null
                }
            } catch (e: Exception) {
                e.printStackTrace()
                null
            }
        } else {
            super.getColumnValue(rs, columnType, column)
        }
    }

    override fun getURL(host: String?, schema: String?, username: String?, password: String?): String {
        var url = "jdbc:$prefix:"
        if (schema != null) {
            url += schema
        }
        return url
    }

    @Throws(SQLException::class)
    override fun dropTable(connection: Connection, table: String) {
        super.dropTable(connection, table.toUpperCase())
    }

    override val className: String
        get() = Companion.className
    override val prefix: String
        get() = "hsqldb"

    override fun toString(): String {
        return "HSQLDB"
    }

    companion object {
        private const val className = "org.hsqldb.jdbc.JDBCDriver"
        fun exist(): Boolean {
            return JDBCDatabaseDriver.Companion.exist(className)
        }
    }
}