/*
 * Copyright (c) 2011 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.db.jdbc

import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.io.ParseException
import java.sql.*
import java.util.*

class MariaDBDatabaseDriver  //    private static final GeometryFactory geometryFactory = new GeometryFactory();
    : JDBCDatabaseDriver() {
    override val className: String
        get() = Companion.className

    @Throws(SQLException::class)
    override fun getColumnType(metaData: ResultSetMetaData, column: Int): Class<*>? {
        return if ("UNKNOWN" == metaData.getColumnTypeName(column) || "GEOMETRY" == metaData.getColumnTypeName(column)) {
            Geometry::class.java
        } else {
            super.getColumnType(metaData, column)
        }
    }

    override fun getColumnType(type: Class<*>, precision: Int): String? {
        return if (Geometry::class.java.isAssignableFrom(type)) {
            "geometry"
        } else {
            super.getColumnType(type, precision)
        }
    }

    @Throws(SQLException::class)
    override fun getColumnValue(rs: ResultSet, columnType: Class<*>?, column: Int): Any? {
        return if (columnType == Geometry::class.java) {
            val o = rs.getObject(column)
            convert2JTS(o)
        } else {
            super.getColumnValue(rs, columnType, column)
        }
    }

    @Throws(SQLException::class)
    override fun getConnection(url: String?, username: String?, password: String?): Connection? {
        var connection: Connection? = null
        val info = Properties()
        if (username != null) {
            info["user"] = username
        }
        if (password != null) {
            info["password"] = password
        }
        info.setProperty("cacheResultSetMetaData", "true")
        info.setProperty("autoReconnect", "true")
        connection = if (username != null) {
            try {
                DriverManager.getConnection(url, info)
            } catch (e: SQLException) {
                try {
                    DriverManager.getConnection(url)
                } catch (e1: SQLException) {
                    throw e
                }
            }
        } else {
            DriverManager.getConnection(url)
        }
        return connection
    }

    @Throws(SQLException::class)
    override fun getDatabases(host: String?, username: String?, password: String?): Iterable<String> {
        val info = Properties()
        if (username != null) {
            info["user"] = username
        }
        if (password != null) {
            info["password"] = password
        }
        info.setProperty("useInformationSchema", "true")
        val conn = getConnection(getURL(host, null, username, password), info)
        val databases: MutableList<String> = ArrayList()
        if (conn != null) {
            val schemas = conn.metaData.catalogs
            while (schemas.next()) {
                databases.add(schemas.getString(1))
            }
        }
        return databases
    }

    override val prefix: String
        get() = "mariadb"

    override fun toString(): String {
        return "MariaDB"
    }

    /**
     * Converts the native geometry object to a JTS `Geometry`.
     *
     * @param object native database geometry object (depends on the JDBC spatial
     * extension of the database)
     * @return JTS geometry corresponding to geomObj.
     */
    fun convert2JTS(`object`: Any?): Geometry? {
        if (`object` == null) return null
        if (`object` is ByteArray) {
            try {
                return MyWKBReader().read(`object`)
            } catch (e: ParseException) {
                e.printStackTrace()
            }
        }
        return null
    }

    companion object {
        private const val className = "org.mariadb.jdbc.Driver"
        fun exist(): Boolean {
            return JDBCDatabaseDriver.Companion.exist(className)
        }
    }
}