/*
 * Copyright (c) 2010 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.db.jdbc

import org.locationtech.jts.geom.*
import org.locationtech.jts.io.*
import java.io.IOException
import java.lang.RuntimeException

class MyWKBReader @JvmOverloads constructor(private val factory: GeometryFactory = GeometryFactory()) {
    private val precisionModel: PrecisionModel

    // default dimension - will be set on read
    private var inputDimension = 2
    private var hasSRID = false
    private var SRID = 0
    private val dis = ByteOrderDataInStream()
    private var ordValues: DoubleArray? = null

    /**
     * Reads a single [Geometry] in WKB format from a byte array.
     *
     * @param bytes the byte array to read from
     * @return the geometry read
     * @throws ParseException if the WKB is ill-formed
     */
    @Throws(ParseException::class)
    fun read(bytes: ByteArray?): Geometry? {
        // possibly reuse the ByteArrayInStream?
        // don't throw IOExceptions, since we are not doing any I/O
        return try {
            read(ByteArrayInStream(bytes!!))
        } catch (ex: IOException) {
            throw RuntimeException("Unexpected IOException caught: " + ex.message)
        }
    }

    /**
     * Reads a [Geometry] in binary WKB format from an [).][is]
     */
    @Throws(IOException::class, ParseException::class)
    fun read(`is`: InStream?): Geometry? {
        dis.setInStream(`is`)
        dis.readInt()
        val g = readGeometry()
        setSRID(g)
        return g
    }

    @Throws(IOException::class, ParseException::class)
    private fun readGeometry(): Geometry? {

        // determine byte order
        val byteOrder = dis.readByte()
        // default is big endian
        if (byteOrder == WKBConstants.wkbNDR) dis.setOrder(ByteOrderValues.LITTLE_ENDIAN)

//      dis.readByte();
        val typeInt = dis.readInt()
        val geometryType = typeInt and 0xff
        // determine if Z values are present
        val hasZ = typeInt and -0x80000000 != 0
        inputDimension = if (hasZ) 3 else 2
        // determine if SRIDs are present
        hasSRID = typeInt and 0x20000000 != 0
        if (hasSRID) {
            SRID = dis.readInt()
        }

        // only allocate ordValues buffer if necessary
        if (ordValues == null || ordValues!!.size < inputDimension) ordValues = DoubleArray(inputDimension)
        when (geometryType) {
            WKBConstants.wkbPoint -> return readPoint()
            WKBConstants.wkbLineString -> return readLineString()
            WKBConstants.wkbPolygon -> return readPolygon()
            WKBConstants.wkbMultiPoint -> return readMultiPoint()
            WKBConstants.wkbMultiLineString -> return readMultiLineString()
            WKBConstants.wkbMultiPolygon -> return readMultiPolygon()
            WKBConstants.wkbGeometryCollection -> return readGeometryCollection()
        }
        val message = "Unknown WKB type $geometryType"
        throw ParseException(message)
        //return null;
    }

    /**
     * Sets the SRID, if it was specified in the WKB
     *
     * @param g the geometry to update
     * @return the geometry with an updated SRID value, if required
     */
    private fun setSRID(g: Geometry?): Geometry? {
        if (SRID != 0) g!!.SRID = SRID
        return g
    }

    @Throws(IOException::class)
    private fun readPoint(): Point {
        val pts = readCoordinateSequence(1)
        return factory.createPoint(pts)
    }

    @Throws(IOException::class)
    private fun readLineString(): LineString {
        val size = dis.readInt()
        val pts = readCoordinateSequence(size)
        return factory.createLineString(pts)
    }

    @Throws(IOException::class)
    private fun readLinearRing(): LinearRing {
        val size = dis.readInt()
        val pts = readCoordinateSequence(size)
        return factory.createLinearRing(pts)
    }

    @Throws(IOException::class)
    private fun readPolygon(): Polygon {
        val numRings = dis.readInt()
        var holes: Array<LinearRing?>? = null
        if (numRings > 1) holes = arrayOfNulls(numRings - 1)
        val shell = readLinearRing()
        for (i in 0 until numRings - 1) {
            holes!![i] = readLinearRing()
        }
        return factory.createPolygon(shell, holes!!.requireNoNulls())
    }

    @Throws(IOException::class, ParseException::class)
    private fun readMultiPoint(): MultiPoint {
        val numGeom = dis.readInt()
        val geoms = arrayOfNulls<Point>(numGeom)
        for (i in 0 until numGeom) {
            val g = readGeometry() as? Point ?: throw ParseException(INVALID_GEOM_TYPE_MSG + "MultiPoint")
            geoms[i] = g
        }
        return factory.createMultiPoint(geoms.requireNoNulls())
    }

    @Throws(IOException::class, ParseException::class)
    private fun readMultiLineString(): MultiLineString? {
        val numGeom = dis.readInt()
        val geoms = arrayOfNulls<LineString>(numGeom)
        for (i in 0 until numGeom) {
            val g = readGeometry() as? LineString ?: throw ParseException(INVALID_GEOM_TYPE_MSG + "MultiLineString")
            geoms[i] = g
        }
        return factory.createMultiLineString(geoms.requireNoNulls())
    }

    @Throws(IOException::class, ParseException::class)
    private fun readMultiPolygon(): MultiPolygon {
        val numGeom = dis.readInt()
        val geoms = arrayOfNulls<Polygon>(numGeom)
        for (i in 0 until numGeom) {
            val g = readGeometry() as? Polygon ?: throw ParseException(INVALID_GEOM_TYPE_MSG + "MultiPolygon")
            geoms[i] = g
        }
        return factory.createMultiPolygon(geoms.requireNoNulls())
    }

    @Throws(IOException::class, ParseException::class)
    private fun readGeometryCollection(): GeometryCollection {
        val numGeom = dis.readInt()
        val geoms = arrayOfNulls<Geometry>(numGeom)
        for (i in 0 until numGeom) {
            geoms[i] = readGeometry()
        }
        return factory.createGeometryCollection(geoms.requireNoNulls())
    }

    @Throws(IOException::class)
    private fun readCoordinateSequence(size: Int): CoordinateSequence {
        val seq = factory.coordinateSequenceFactory.create(size, inputDimension)
        var targetDim = seq.getDimension()
        if (targetDim > inputDimension) targetDim = inputDimension
        for (i in 0 until size) {
            readCoordinate()
            for (j in 0 until targetDim) {
                seq.setOrdinate(i, j, ordValues!![j])
            }
        }
        return seq
    }

    /**
     * Reads a coordinate value with the specified dimensionality.
     * Makes the X and Y ordinates precise according to the precision model
     * in use.
     */
    @Throws(IOException::class)
    private fun readCoordinate() {
        for (i in 0 until inputDimension) {
            if (i <= 1) {
                ordValues!![i] = precisionModel.makePrecise(dis.readDouble())
            } else {
                ordValues!![i] = dis.readDouble()
            }
        }
    }

    companion object {
        /**
         * Converts a hexadecimal string to a byte array.
         * The hexadecimal digit symbols are case-insensitive.
         *
         * @param hex a string containing hex digits
         * @return an array of bytes with the value of the hex string
         */
        fun hexToBytes(hex: String): ByteArray {
            val byteLen = hex.length / 2
            val bytes = ByteArray(byteLen)
            for (i in 0 until hex.length / 2) {
                val i2 = 2 * i
                require(i2 + 1 <= hex.length) { "Hex string has odd length" }
                val nib1 = hexToInt(hex[i2])
                val nib0 = hexToInt(hex[i2 + 1])
                val b = ((nib1 shl 4) + nib0.toByte()).toByte()
                bytes[i] = b
            }
            return bytes
        }

        private fun hexToInt(hex: Char): Int {
            val nib = Character.digit(hex, 16)
            require(nib >= 0) { "Invalid hex digit: '$hex'" }
            return nib
        }

        private const val INVALID_GEOM_TYPE_MSG = "Invalid geometry type encountered in "
    }

    init {
        precisionModel = factory.precisionModel
    }
}