/*
 * Copyright (c) 2012 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.db.jdbc

import java.sql.SQLException
import java.util.ArrayList

class SQLServerDatabaseDriver : JDBCDatabaseDriver() {
    override val className: String
        get() = Companion.className
    override val prefix: String
        get() = "sqlserver"

    @Throws(SQLException::class)
    override fun getDatabases(host: String?, username: String?, password: String?): Iterable<String> {
        val conn = getConnection(getURL(host, null, username, password), username, password)
        val databases: MutableList<String> = ArrayList()
        val schemas = conn!!.metaData.catalogs
        while (schemas.next()) {
            databases.add(schemas.getString(1))
        }
        return databases
    }

    @Throws(SQLException::class)
    override fun getTables(url: String?, database: String?, username: String?, password: String?): Iterable<String?>? {
        val conn = getConnection(url, username, password)
        val tables: MutableList<String?> = ArrayList()
        val meta = conn!!.metaData
        val result = meta.getTables(null, null, "%", arrayOf("TABLE"))
        while (result.next()) {
            var tableName = result.getString("TABLE_NAME")
            val schemaName = result.getString("TABLE_SCHEM")
            if (schemaName != null) {
                tableName = "$schemaName.$tableName"
            }
            tables.add(tableName)
        }
        return tables
    }

    override fun toString(): String {
        return "Microsoft SQL Server"
    }

    override fun getURL(host: String?, schema: String?, username: String?, password: String?): String {
        var url = "jdbc:$prefix:"
        if (host != null) {
            url += "//$host"
        }
        url += ";"
        if (schema != null) {
            url += "databaseName=$schema;"
        }
        if (username == null || username == "") {
            url += "integratedSecurity=true;"
        }
        return url
    }

    companion object {
        private const val className = "com.microsoft.sqlserver.jdbc.SQLServerDriver"
        fun exist(): Boolean {
            return JDBCDatabaseDriver.Companion.exist(className)
        }
    }
}