/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.aggregates.cuboid

import com.macrofocus.common.collection.SoftReference
import com.macrofocus.common.collection.TIntArrayList
import com.macrofocus.common.collection.TIntList
import com.macrofocus.common.collection.emptyConcurrentHashSet
import org.molap.aggregates.cube.Cube
import org.molap.aggregates.cube.Dice
import org.molap.aggregates.cube.Dimensions
import org.molap.aggregates.cube.Group
import org.molap.aggregates.query.Structure
import org.molap.series.Series

/**
 * A child cuboid, in principle ultimately rooted on an `ApexCuboid`.
 *
 * @see org.molap.aggregates.cuboid.ApexCuboid
 */
class DrilledCuboid(cube: Cube, parent: Cuboid, dimensions: Dimensions, dice: Dice, structure: Structure) : AbstractCuboid(cube) {
    private val parent: Cuboid
    override val dimensions: Dimensions
    private val dice: Dice
    private val structure: Structure
    override var values: Set<Any>? = null
    get() {
        if (isDirty) {
            build()
        }
        return field
    }
    private var parentChildren: Map<Group, MutableSet<Group>>? = null
    override var groups: Set<Group>? = null
    get() {
        if (isDirty) {
            build()
        }

        return field
    }
    private var rows: SoftReference<Map<Group, TIntList>>? = null
    protected override fun build() {
        super.build()
        val values: MutableSet<Any> = LinkedHashSet()
        val parentChildren: MutableMap<Group, MutableSet<Group>> = LinkedHashMap<Group, MutableSet<Group>>()
        val groups: MutableSet<Group> = HashSet<Group>()
        val last: Any? = dimensions.last
        val series: Series<Any?,out Any?>? = cube.dataFrame.getColumn(last)
        for (parentGroup in parent.groups!!) {
            val children: MutableSet<Group>
            if (!parentChildren.containsKey(parentGroup)) {
                children = emptyConcurrentHashSet<Group>()
                parentChildren[parentGroup] = children
            } else {
                children = parentChildren[parentGroup]!!
            }
            for (row in parent.getRows(parentGroup)!!) {
                var value = series?.get(row)
                //                Object value = cube.getDataFrame().getValue(row, last);
                if (dice.last == null || dice.last!!.contains(value)) {
                    if (structure.last) {
                        value = Cuboid.ALL
                    }
                    // Values
                    values.add(value!!)
                    val current: Group = parentGroup.drillDown(value)

                    // Children
                    children.add(current)

                    // Groups
                    groups.add(current)

                    // Rows
//                    final TIntArrayList r;
//                    if(!rows.containsKey(current)) {
//                        r = new TIntArrayList();
//                        rows.put(current, r);
//                    } else {
//                        r = rows.get(current);
//                    }
//
//                    assert r != null;
//
//                    r.add(series.getAddress(row));
                }
            }
        }
        this.values = values
        this.parentChildren = parentChildren
        this.groups = groups
        rows = null
    }

    protected fun buildRows(): Map<Group, TIntList?> {
        if (isDirty) {
            build()
        }
        val rows: MutableMap<Group, TIntList> = LinkedHashMap<Group, TIntList>()
        val last: Any? = dimensions.last
        val series: Series<Any?,Any?>? = cube.dataFrame.getColumn(last)
        for (parentGroup in parent.groups!!) {
            for (row in parent.getRows(parentGroup as Group)!!) {
                var value: Any? = series!!.get(row)
                //                Object value = cube.getDataFrame().getValue(row, last);
                if (dice.last == null || dice.last!!.contains(value)) {
                    if (structure.last) {
                        value = Cuboid.ALL
                    }
                    val current: Group = parentGroup.drillDown(value)
                    // Rows
                    val r: TIntList?
                    if (!rows.containsKey(current)) {
                        r = TIntArrayList()
                        rows[current] = r
                    } else {
                        r = rows[current]
                    }
//                    assert(r != null)
                    r!!.add(series.getAddress(row))
                }
            }
        }
        this.rows = SoftReference<Map<Group, TIntList>>(rows)

//        setDirty(false);
        return rows
    }

    override fun drillDown(vararg columns: Any?): Cuboid? {
        val diceValues: Array<Set<Any?>?> = arrayOfNulls(columns.size)
        val structureValues = BooleanArray(columns.size) { false }
        return cube.getCuboid(dimensions.drillDown(columns)!!, dice.drillDown(*diceValues.requireNoNulls()), structure.drillDown(*structureValues))
    }

    override fun drillUp(): Cuboid? {
        return cube.getCuboid(dimensions.drillUp()!!, dice.drillUp()!!, structure.drillUp()!!)
    }

    override fun slice(value: Any?): Cuboid? {
        return dice(setOf(value))
    }

    override fun dice(valuesSets: Set<Any?>): Cuboid? {
        return cube.getCuboid(dimensions, dice.changeLast(valuesSets), structure)
    }

    override fun collapse(): Cuboid? {
        return cube.getCuboid(dimensions, dice, structure.changeLast(true))
    }

    override fun getGroupsStartingWith(group: Group?): Iterable<Group?>? {
        if (isDirty) {
            build()
        }
        return parentChildren!![group]!!
    }

    override fun getGroupsStartingWithCount(group: Group?): Int {
        if (isDirty) {
            build()
        }
        return parentChildren!![group]!!.size
    }

    override val groupsCount: Int
        get() {
            if (isDirty) {
                build()
            }

            return groups!!.size
        }

   override fun getRows(group: Group?): Iterable<Any?> {
//        if(isDirty()) {
//            build();
//        }
//        if(rows == null) {
//            buildRows();
//        }
        return object : Iterable<Any?> {
            override fun iterator(): Iterator<Any?> {
                var rows: Map<Group, TIntList?>? = null
                if (this@DrilledCuboid.rows != null) {
                    rows = this@DrilledCuboid.rows!!.get()
                }
                if (rows == null) {
                    rows = buildRows()
                }
                val rowsForGroup: TIntList? = rows[group]
                return if (rowsForGroup != null) {
                    object : MutableIterator<Any?> {
                        val last: Any? = dimensions.last
                        var series: Series<out Any?,out Any?>? = cube.dataFrame.getColumn(last)
                        var it: Iterator<Int> = rowsForGroup.iterator()
                        override fun hasNext(): Boolean {
                            return it.hasNext()
                        }

                        override fun next(): Any? {
                            val address: Int = it.next()
                            return series?.getKey(address)
                        }

                        override fun remove() {
                            throw UnsupportedOperationException()
                        }
                    }
                } else {
                    object : MutableIterator<Any?> {
                        override fun hasNext(): Boolean {
                            return false
                        }

                        override fun next(): Any? {
                            return null
                        }

                        override fun remove() {
                            throw UnsupportedOperationException()
                        }
                    }
                }
            }
        }
    }

    override fun toString(): String {
        return "DrilledCuboid{" +
                "dimensions=" + dimensions +
                '}'
    }

    init {
        this.parent = parent
        this.dimensions = dimensions
        this.dice = dice
        this.structure = structure
    }
}