package org.molap.dataframe

import org.molap.aggregates.AggregateDataFrame
import org.molap.aggregates.aggregation.Aggregation
import org.molap.aggregates.aggregation.SumAggregation
import org.molap.index.UniqueIndex
import org.molap.series.Series
import kotlin.reflect.KClass

/**
 * An indexed table structure, i.e. a 2D array of values that can be queried row and/or column-wise using labels.
 * This is similar to the DataFrame that can be found in the R and Pandas libraries.
 *
 * @param <Row> the type of row keys maintained used to query this data frame.
 * @param <Column> the type of column keys maintained used to query this data frame.
 * @param <V> the type of values
</V></Column></Row> */
interface DataFrame<Row, Column, V> {
    /**
     * Returns the name of the column. This is a convenience method for labeling purpose.
     *
     * @param column the key of the column
     * @return the name of the column
     */
    fun getColumnName(column: Column): String?

    /**
     * Returns the most specific superclass for all cell values in a row.
     *
     * @param row the key of the row
     * @return the common ancestor class of the object values in the row.
     */
    fun getRowClass(row: Row): KClass<*>?

    /**
     * Returns the most specific superclass for all cell values in a column
     *
     * @param column the key of the column
     * @return the common ancestor class of the object values in the column.
     */
    fun getColumnClass(column: Column): KClass<*>

    /**
     * Returns the value for the cell at the intersection of the
     * `column` key and `row` key.
     *
     * @param   row        the row key whose value is to be queried
     * @param   column     the column key whose value is to be queried
     * @return  the value Object at the specified cell
     */
    fun getValueAt(row: Row, column: Column): V

    /**
     * Returns a series of all the values of a given row.
     *
     * @param row the row key
     * @return a Series object
     */
//    fun getRow(row: Row): Series<Column, *>?

    /**
     * Returns a series of all the values of a given column.
     *
     * @param column the row key
     * @return a Series object
     */
    fun getColumn(column: Column): Series<Row, V>?

    /**
     * Returns the row keys.
     *
     * @return the row keys
     */
    fun rows(): Iterable<Row>

    /**
     * Returns the column keys.
     *
     * @return the column keys
     */
    fun columns(): Iterable<Column>

    /**
     * Returns the row key at the specified absolute index. This is the inverse of [.getRowAddress].
     *
     * @param index the index
     * @return the row key
     */
    fun getRowKey(index: Int): Row

    /**
     * Returns the column key at the specified absolute index. This is the inverse of [.getColumnAddress].
     *
     * @param index the index
     * @return the column key
     */
    fun getColumnKey(index: Int): Column

    /**
     * Returns the absolute index for the specified row key. This is the inverse of [.getRowKey].
     *
     * @param row the row key
     * @return the absolute index of the specified key.
     */
    fun getRowAddress(row: Row): Int

    /**
     * Returns the absolute index for the specified column key. This is the inverse of [.getColumnKey].
     *
     * @param column the column key
     * @return the absolute index of the specified key.
     */
    fun getColumnAddress(column: Column): Int

    /**
     * Returns the number of rows contained by the this data frame.
     *
     * @return the number of rows.
     */
    val rowCount: Int

    /**
     * For interoperability with deckg.gl 'data'
     */
    val length: Int
        get() = rowCount

    /**
     * Returns the number of columns contained by this data frame.
     *
     * @return the number of columns.
     */
    val columnCount: Int

    /**
     * Returns a new data frame reindexed using integers.
     *
     * @return the reindexed data frame.
     */
//    fun reindexRows(): DataFrame<Int?, Column, V>?

    /**
     * Returns a new data frame reindexed using the values coming from the specified column.
     *
     * @param column the columns to use for the label values
     * @return the reindexed data frame.
     */
//    fun reindexRowsUsingColumn(column: Column): DataFrame<V, Column, V>?

    /**
     * Returns a new data frame reindexed using the values coming from the specified column.
     *
     * @param column the columns to use for the label values
     * @return the reindexed data frame.
     */
//    fun reindexRowsUsingColumnDefault(keepColumn: Boolean, column: Column): DataFrame<V, Column, V>?

    /**
     * Returns a new data frame reindexed using the values coming from the specified rows.
     *
     * @param columns the columns to use for the label values
     * @return the reindexed data frame.
     */
//    fun reindexRowsUsingColumns(vararg columns: Column): DataFrame<MultiKey, Column, V>?

    /**
     * Returns a new data frame reindexed using the values coming from the specified rows.
     *
     * @param columns the columns to use for the label values
     * @return the reindexed data frame.
     */
//    fun reindexRowsDefault(keepColumns: Boolean, vararg columns: Column): DataFrame<MultiKey, Column, V>?

    /**
     * Returns a new data frame with column reindexed using the specified values.
     *
     * @param columns the values to use for the reindex columns
     * @return the reindexed data frame.
     */
//    fun <Column> reindexColumns(vararg columns: Column): DataFrame<Row, Column, V>?

    /**
     * Gets the index used to access the rows.
     *
     * @return the row index
     */
    val rowIndex: UniqueIndex<Row>

    /**
     * Gets the index used to access the columns.
     *
     * @return the column index
     */
    val columnIndex: UniqueIndex<Column>

    /**
     * Returns a new data frame reordered using the values coming from the specified columns.
     *
     * @param columns the columns to use for sorting
     * @return the sorted data frame.
     */
//    fun orderRows(vararg columns: SortKey<Column>?): DataFrame<Row, Column, V>?

    /**
     * Returns a new data frame those rows are filtered by the specified filter model.
     *
     * @param filter the filter model
     * @return a filtered data frame
     */
//    fun filter(filter: MutableFilter<Row>?): FilterDataFrame<Row, Column, V>?

    /**
     * Returns a new data frame where duplicate entries for values of the specified column values.
     *
     * @param columns the columns to use to check for duplicates
     * @return a new data frame
     */
//    fun removeDuplicates(vararg columns: Column): DataFrame<Row, Column, V>?

    /**
     * Returns a new data frame suitable for data aggregation. The additional methods provided to customize the
     * aggregation criteria are provided by the [com.macrofocus.molap.aggregates.AggregateDataFrame].
     *
     * @param aggregation the aggregation methods to use as columns
     * @return an aggregated data frame.
     */
    fun aggregate(vararg aggregation: Aggregation<Any?>): AggregateDataFrame<Column>
    fun aggregate(includeIndex: Boolean, vararg aggregation: Aggregation<Any?>): AggregateDataFrame<Column>
//    fun remapColumns(vararg columns: Column): DataFrame<Row, Column, V>?
//    fun removeColumns(vararg columns: Column): DataFrame<Row, Column, V>?

    /**
     * Returns a new data frame with the rows of the specified data frame appended to the end of the this data frame.
     *
     * @param dataFrame the data frame those rows are to be appended
     * @return a combined data frame.
     */
//    fun append(dataFrame: DataFrame<Row, Column, V>?): DataFrame<Row, Column, V>?

    /**
     * Returns a new data frame with the rows of the specified data frame appended to the end of the this data frame.
     *
     * @param dataFrame the data frame those rows are to be appended
     * @return a combined data frame.
     */
//    fun appendAndReindex(dataFrame: DataFrame<Row, Column, V>?): DataFrame<Int?, Column, V>?
//    fun join(series: Series<Any?,Any?>?, columns: Array<Column>?): DataFrame<*, *, *>?
//    fun getStatistics(column: Column): UnivariateStatistics?

    /**
     * Returns data frame for aggregation.
     *
     * @return the aggregation data frame
     */
//    val dataFrameAggregation: DataFrameAggregation?

    /**
     * Returns the aggregation method that returns a constant value.
     *
     * @param value the constant value
     * @return the aggregation method
     */
//    fun getFirst(value: Column): FirstAggregation?

    /**
     * Returns the aggregation method that returns a constant value.
     *
     * @param value the constant value
     * @return the aggregation method
     */
//    fun getConstant(value: Any?): ConstantAggregation?

    /**
     * Returns the aggregation method that returns a random value.
     *
     * @return the aggregation method
     */
//    fun getRandom(min: Double, max: Double): Aggregation<Any>?

    /**
     * Returns the aggregation method for computing the sum.
     *
     * @param column the column key
     * @return the aggregation method
     */
    fun getSum(column: Column): SumAggregation?

    /**
     * Returns the aggregation method for counting the number of values.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getCount(column: Column): CountAggregation?

    /**
     * Returns the aggregation method for computing the univariate statistics.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getDistributiveStatistics(column: Column): DistributiveStatisticsAggregation?

    /**
     * Returns the aggregation method for finding the minimum value.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getMin(column: Column): MinAggregation?

    /**
     * Returns the aggregation method for finding the maximum value.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getMax(column: Column): MaxAggregation?

    /**
     * Returns the aggregation method for computing the mean value.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getMean(column: Column): MeanAggregation?

    /**
     * Returns the aggregation method for computing the variance.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getVariance(column: Column): VarianceAggregation?

    /**
     * Returns the aggregation method for computing the variance by factoring in the specified population.
     *
     * @param column the column key
     * @param population the key of the column for the population
     * @return the aggregation method
     */
//    fun getVarianceByPopulation(column: Column, population: Column): VarianceAggregation?

    /**
     * Returns the aggregation method for computing the standard deviation.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getStdDev(column: Column): StdDevAggregation?

    /**
     * Returns the aggregation method for computing the univariate statistics.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getUnivariateStatistics(column: Column): UnivariateStatisticsAggregation?

    /**
     * Returns the aggregation method for computing the median value.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getMedian(column: Column): MedianAggregation?

    /**
     * Returns the aggregation method for computing the median value.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getFirstQuartile(column: Column): FirstQuartileAggregation?

    /**
     * Returns the aggregation method for computing the median value.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getThirdQuartile(column: Column): ThirdQuartileAggregation?

    /**
     * Returns the aggregation method for computing the weighted sum.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getWeightedSum(weight: Column, column: Column): Aggregation<Any>?

    /**
     * Returns the aggregation method for computing the weigthed mean.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getWeightedMean(weight: Column, column: Column): Aggregation<Any>?

    /**
     * Returns the aggregation method for counting the number of distinct values.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getCountDistinct(column: Column): CountDistinctAggregation?

    /**
     * Returns the aggregation method for counting the number of distinct values, including null value if present.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getCountDistinctWithNull(column: Column): CountDistinctWithNullAggregation?

    /**
     * Returns the aggregation method for finding the centroid of geometries.
     *
     * @param column the column key
     * @return the aggregation method
     */
//    fun getCentroid(column: Column): Aggregation<Any>?

    /**
     * Display the content of the data frame to the console.
     */
    fun printSchema()

    /**
     * Display the content of the data frame to the console.
     */
    fun print()

    /**
     * Display the content of the data frame to the specified output.
     *
     * @param out the output
     * @param caption the title
     * @param html false for text output, true for HTML formatting
     */
//    fun print(out: PrintStream?, caption: String?, html: Boolean)

    /**
     * Perform some repetitive operation to estimate the performance of the data frame.
     */
//    fun benchmark()

    /**
     * Add a listener to the list that's notified each time a change to the data frame occurs.
     *
     * @param listener the DataFrameListener
     */
    fun addDataFrameListener(listener: DataFrameListener<Row, Column>)

    /**
     * Add a listener to the list that's notified each time a change to the data frame occurs. The listener will
     * automatically be disposed of should no other object have a reference to it.
     *
     * @param listener the DataFrameListener
     */
    fun addWeakDataFrameListener(listener: DataFrameListener<Row, Column>)

    /**
     * Remove a listener to the list that's notified each time a change to the data frame occurs.
     *
     * @param listener the DataFrameListener
     */
    fun removeDataFrameListener(listener: DataFrameListener<Row, Column>)

    /**
     * Remove all listeners to the list that's notified each time a change to the data frame occurs.
     */
    fun removeDataFrameListeners()

    /**
     * Explicit companion object declaration
     */
    companion object
}

fun <N, Row, Column> DataFrame<N, Row, Column>.printSchema() {
    for (column in columns()) {
        print(AbstractDataFrame.abbreviate(getColumnName(column), 19))
        print(": ")
        val value: KClass<*> = getColumnClass(column)
        print(AbstractDataFrame.abbreviate(if (value != null) value.simpleName else "", 19))
        println()
    }
}

fun <R, C, V> DataFrame<R, C, V>.reindexRowsUsingColumnDefault(keepColumn: Boolean, column: C): MutableDataFrame<V, C, V> {
    val recipe = ColumnDefaultReIndexRecipe(this, column, keepColumn)
    return ReIndexedDataFrame<V, C, V, R, C>(this, recipe)
}

fun <R, C, V> DataFrame<R, C, V>.find(column: C, value: V): R? {
    for (row in rows()) {
        if(value == getValueAt(row, column)) {
            return row
        }
    }
    return null
}

fun <R, C, V> DataFrame<R, C, V>.findAll(column: C, value: V): List<R> {
    val result = mutableListOf<R>()
    for (row in rows()) {
        if(value == getValueAt(row, column)) {
            result.add(row)
        }
    }
    return result
}

