package org.molap.dataframe

import com.macrofocus.common.collection.CollectionFactory
import org.molap.index.DefaultUniqueIndex
import org.molap.index.UniqueIndex
import kotlin.reflect.KClass

/**
 * Created by luc on 09/03/16.
 */
class WrappedDataFrame<R, C, V>(val dataFrame: DataFrame<R, C, V>) : ReMappedDataFrame<R, C, V>(dataFrame, object : ReMappedRecipe<R,C> {
    override fun buildRowIndex(): UniqueIndex<R> {
        return dataFrame.rowIndex
    }

    override fun buildColumnIndex(): UniqueIndex<C> {
        return dataFrame.columnIndex
    }
}) {
    private val keys: MutableList<C> = ArrayList<C>()
    private val columns: MutableMap<C, Column<R, C, V>> = LinkedHashMap()
    private val columnListener: ColumnListener<R, C, V> = object : ColumnListener<R, C, V> {
        override fun columnChanged(event: ColumnEvent<R, C, V>) {
            notifyDataFrameChanged(DataFrameEvent(event.affectedRows, setOf(event.affectedColumn), false))
        }
    }

    override fun getColumnName(column: C): String {
        return columns[column]!!.columnName
    }

    override fun getColumnClass(column: C): KClass<*> {
//        assert columns.containsKey(c): "Column " + c + " doesn't exists";
        val column = columns[column]
        return if (column != null) {
            column.getType(this)
        } else {
//            Logging.getInstance().process(IllegalArgumentException("Column $c doesn't exists"))
            Any::class
        }
    }

    override fun getValueAt(row: R, column: C): V {
//        assert columns.containsKey(c): "Column " + c + " doesn't exists";
        val column = columns[column]
        return column?.getValue(this, row) as V
    }

    fun addOriginalColumns() {
        for (column in dataFrame.columns()) {
//            assert(originalDataFrame.getColumnIndex().contains(column)) { "Column $column doesn't exists" }
//            assert(!columns.containsKey(column)) { "Column $column already exists" }
            keys.add(column)
            columns[column] = InternalDataFrameColumn(column)
        }
        invalidate()
        scheduleUpdate()
    }

    fun addOriginalColumn(vararg columns: C) {
        for (column in columns) {
//            assert(originalDataFrame.getColumnIndex().contains(column)) { "Column $column doesn't exists" }
//            assert(!this.columns.containsKey(column)) { "Column $column already exists" }
            keys.add(column)
            this.columns[column] = InternalDataFrameColumn(column)
        }
        invalidate()
        scheduleUpdate()
    }

    fun addRenamedColumn(column: C, name: C, checkExists: Boolean = true) {
//        assert(!checkExists || originalDataFrame.getColumnIndex().contains(column)) { "Column $column doesn't exists" }
//        assert(!columns.containsKey(column)) { "Column $column already exists" }
        keys.add(name)
        columns[name] = InternalDataFrameColumn(name, column)
        invalidate()
        scheduleUpdate()
    }

    fun addDerivedColumn(vararg columns: Column<R, C, V>) {
        for (column in columns) {
            keys.add(column.key)
            this.columns[column.key] = column
            column.addColumnListener(columnListener)
        }
        invalidate()
        scheduleUpdate()
    }

    fun removeDerivedColumn(vararg columns: Column<R, C, V>) {
        for (column in columns) {
            keys.remove(column.key)
            this.columns.remove(column.key)
            column.removeColumnListener(columnListener)
        }
        invalidate()
        scheduleUpdate()
    }

    override var originalDataFrame: DataFrame<R, C, V>
        get() = dataFrame
        set(originalDataFrame) {
            super.originalDataFrame = originalDataFrame
        }

    fun getCalculatedColumnAt(i: Int): Column<*, *, *> {
        return columns[keys[i]]!!
    }

    fun modifyColumn(oldColumn: C, column: Column<R, C, V>) {
        columns[oldColumn]!!.removeColumnListener(columnListener)
        keys[keys.indexOf(oldColumn)] = column.key

        // ToDo: Currently the trick is not to remove the old column as name change doesn't propagate well (yet)
//        columns.remove(oldColumn, column);
        columns[oldColumn] = column
        columns[column.key] = column
        column.addColumnListener(columnListener)
        invalidate()
        scheduleUpdate()
    }

    interface Column<R, C, V> {
        val key: C

        fun getType(dataFrame: WrappedDataFrame<R, C, V>): KClass<*>
        val columnName: String

        fun getValue(dataFrame: WrappedDataFrame<R, C, V>, r: R): V
        fun addColumnListener(listener: ColumnListener<R, C, V>)
        fun removeColumnListener(listener: ColumnListener<R, C, V>)
    }

    abstract class AbstractColumn<R, C, V> : Column<R, C, V> {
        private val listernersDelegate = lazy { CollectionFactory.copyOnWriteArrayList<ColumnListener<R, C, V>>() }
        private val listeners: MutableList<ColumnListener<R, C, V>> by listernersDelegate
        override fun addColumnListener(listener: ColumnListener<R, C, V>) {
            listeners.add(listener)
        }

        override fun removeColumnListener(listener: ColumnListener<R, C, V>) {
            listeners.remove(listener)
        }

        protected fun notifyColumnChanged(event: ColumnEvent<R, C, V>) {
            for (listener in listeners) {
                listener.columnChanged(event)
            }
        }
    }

    abstract class NamedTypedColumn<R, V>(override val columnName: String, type: KClass<*>) :
        AbstractColumn<R, String?, V>() {
        private val type: KClass<*>
        override val key: String
            get() = columnName

        override fun getType(dataFrame: WrappedDataFrame<R, String?, V>): KClass<*> {
            return type
        }

        init {
            this.type = type
        }
    }

    abstract class DerivedColumn<R, C, V>(override val key: C, type: KClass<*>) :
        Column<R, C, V> {
        private val type: KClass<*>
        private val listernersDelegate = lazy { CollectionFactory.copyOnWriteArrayList<ColumnListener<R, C, V>>() }
        private val listeners: MutableList<ColumnListener<R, C, V>> by listernersDelegate
        override val columnName: String
            get() = key.toString()

        override fun getType(dataFrame: WrappedDataFrame<R, C, V>): KClass<*> {
            return type
        }

        //        @Override
        //        public V getValue(R row) {
        //            return getValue(DerivedDataFrame.this, row);
        //        }
        //
        //        public abstract V getValue(DerivedDataFrame<R, C, V> dataFrame, R row);
        override fun toString(): String {
            return key.toString()
        }

        override fun addColumnListener(listener: ColumnListener<R, C, V>) {
            listeners.add(listener)
        }

        override fun removeColumnListener(listener: ColumnListener<R, C, V>) {
            listeners.remove(listener)
        }

        init {
            this.type = type
        }
    }

    class SumColumn<R, C, V>(column: C, vararg columns: C) :
        DerivedColumn<R, C, V>(column, Number::class) {
        private val columns: Array<out C>
        override fun getValue(dataFrame: WrappedDataFrame<R, C, V>, r: R): V {
            var sum: Number? = null
            val orig: DataFrame<R,C,V> = dataFrame.originalDataFrame
            for (c in columns) {
                val value: V = orig.getValueAt(r, c)
                if (value != null && value is Number) {
                    val n = value
                    sum = if (sum == null) {
                        n
                    } else {
                        sum.toDouble() + n.toDouble()
                    }
                }
            }
            return sum as V
        }

        init {
            this.columns = columns
        }
    }

    inner class InternalDataFrameColumn constructor(
        override val key: C,
        private val sourceColumn: C = key
    ) :
        Column<R, C, V> {
        override fun getType(dataFrame: WrappedDataFrame<R, C, V>): KClass<*> {
            return originalDataFrame.getColumnClass(sourceColumn)
        }

        override val columnName: String
            get() = key.toString()

        override fun getValue(dataFrame: WrappedDataFrame<R, C, V>, r: R): V {
            return originalDataFrame.getValueAt(r, sourceColumn)
        }

        override fun addColumnListener(listener: ColumnListener<R, C, V>) {}
        override fun removeColumnListener(listener: ColumnListener<R, C, V>) {}
    }

    class DataFrameColumn<R, C, V>(
        private val dataFrame: DataFrame<R, C, V>,
        private val sourceColumn: C,
        override val key: C
    ) :
        Column<R, C, V> {
        constructor(dataFrame: DataFrame<R, C, V>, column: C) : this(dataFrame, column, column) {}

        override fun getType(dataFrame: WrappedDataFrame<R, C, V>): KClass<*> {
            return this.dataFrame.getColumnClass(sourceColumn)
        }

        override val columnName: String
            get() = key.toString()

        override fun getValue(dataFrame: WrappedDataFrame<R, C, V>, r: R): V {
            return this.dataFrame.getValueAt(r, sourceColumn)
        }

        override fun addColumnListener(listener: ColumnListener<R, C, V>) {}
        override fun removeColumnListener(listener: ColumnListener<R, C, V>) {}
    }

    interface ColumnListener<R, C, V> {
        fun columnChanged(event: ColumnEvent<R, C, V>)
    }

    class ColumnEvent<R, C, V>(val affectedRows: Iterable<R>, val affectedColumn: C)

    init {
        val recipe: ReMappedRecipe<R, C> = object : ReMappedRecipe<R, C> {
            override fun buildRowIndex(): UniqueIndex<R> {
                return dataFrame.rowIndex
            }

            override fun buildColumnIndex(): UniqueIndex<C> {
                return DefaultUniqueIndex<C>(keys, false)
            }
        }
        setRecipe(recipe)
    }
}
