package org.molap.network

interface Link<out A, out B> {
    val from: A
    val to: B
}

/**
 * Represents a generic Link of two values.
 *
 * There is no meaning attached to values in this class, it can be used for any purpose.
 * Link exhibits value semantics, i.e. two Links are equal if both components are equal.
 *
 * @param A type of the first value.
 * @param B type of the second value.
 * @property from First value.
 * @property to Second value.
 * @constructor Creates a new instance of Link.
 */
public data class DirectedLink<out A, out B> (
    override val from: A,
    override val to: B
) : Link<A, B> {

    /**
     * Returns string representation of the [Link] including its [from] and [to] values.
     */
    public override fun toString(): String = "$from->$to"
}

public open class UndirectedLink<out A, out B>  (
    override val from: A,
    override val to: B
) : Link<A, B> {

    override fun equals(other: Any?): Boolean {
        return super.equals(other) || (other is UndirectedLink<*,*> && from == other.from && to == other.to) || (other is UndirectedLink<*,*> && from == other.to && to == other.from)
    }

    override fun hashCode(): Int {
        return from.hashCode() + to.hashCode()
    }

    /**
     * Returns string representation of the [Link] including its [from] and [to] values.
     */
    public override fun toString(): String = "$from-$to"
}

/**
 * Creates a tuple of type [Link] from this and [that].
 *
 * This can be useful for creating [Map] literals with less noise, for example:
 */
public infix fun <A, B> A.to(that: B): Link<A, B> = DirectedLink(this, that)

public infix fun <A, B> A.connect(that: B): Link<A, B> = UndirectedLink(this, that)

/**
 * Converts this Link into a list.
 */
public fun <T> Link<T, T>.toList(): List<T> = listOf(from, to)