/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.aggregates.cube

import org.molap.aggregates.aggregation.*
import org.molap.aggregates.cuboid.ApexCuboid
import org.molap.aggregates.cuboid.Cuboid
import org.molap.aggregates.cuboid.DrilledCuboid
import org.molap.aggregates.query.CuboidQuery
import org.molap.aggregates.query.Query
import org.molap.aggregates.query.Structure
import org.molap.dataframe.DataFrame
import org.molap.dataframe.DataFrameEvent
import org.molap.dataframe.DataFrameListener

/**
 * An in-memory multidimensional cube.
 */
class InMemoryCube(dataFrame: DataFrame<Any?,Any?,Any?> ) : Cube {
    override val dataFrame: DataFrame<Any?,Any?,Any?>
    private val apex: ApexCuboid
    override val root: Dimensions
    override val dice: Dice
    override val structure: Structure
    private val cuboids: MutableMap<DimensionsDiceStructure, Cuboid> = HashMap<DimensionsDiceStructure, Cuboid>()
    private val child: MutableMap<Cuboid?, Cuboid> = HashMap<Cuboid?, Cuboid>()
    private val listener: DataFrameListener<Any?,Any?> = object : DataFrameListener<Any?,Any?> {
        override fun dataFrameChanged(event: DataFrameEvent<Any?,Any?>) {
            for (cuboid in cuboids.values) {
                cuboid.isDirty = true
            }
        }
    }

    override val apexCuboid: Cuboid
        get() = apex
    override val rootGroup: Group
        get() = apex.group

    override fun getCuboid(dimensions: Dimensions, dice: Dice, structure: Structure): Cuboid? {
//        assert(dimensions.path.length === dice!!.path.length && dice!!.path.length === structure.getPath().length) { dimensions.toString() + ", " + dice + ", " + structure }
        val key = DimensionsDiceStructure(dimensions, dice, structure)
        if (!cuboids.containsKey(key)) {
//            System.err.println("Creating cuboid for " + dimensions + ", " + dice + ", " + structure);
            val drillUp: Dimensions? = dimensions?.drillUp()
            val diceUp: Dice? = dice!!.drillUp()
            val structureUp: Structure? = structure?.drillUp()

            // Recursion
            val parentCuboid: Cuboid? = getCuboid(drillUp!!, diceUp!!, structureUp!!)
            val cuboid = DrilledCuboid(this, parentCuboid!!, dimensions, dice, structure)
            cuboids[key] = cuboid
            child[parentCuboid] = cuboid
            return cuboid
        }
        return cuboids[key]
    }

    override fun getChildCuboid(cuboid: Cuboid?): Cuboid? {
        return child[cuboid]
        //
//        for (Dimensions dimensions : cuboids.keySet()) {
//            Dimensions up = dimensions.drillUp();
//            if(up != null && up.equals(cuboid.getDimensions())) {
//                return cuboids.get(dimensions);
//            }
//        }
//        return null;
    }

    fun query(vararg aggregations: Aggregation<Any?>): Query {
        return CuboidQuery(apexCuboid, *aggregations)
    }

    fun getSum(name: Any?): SumAggregation {
        return SumAggregation(dataFrame.getColumn(name)!!)
    }

    fun getSum(name: String?, filter: DistributiveStatistics.RowFilter<Any?>?): SumAggregation {
        return SumAggregation(dataFrame.getColumn(name)!!, filter)
    }

    fun getCount(name: String?): CountAggregation {
        return CountAggregation(dataFrame.getColumn(name)!!)
    }

    fun getMin(name: String?): MinAggregation {
        return MinAggregation(dataFrame.getColumn(name)!!)
    }

    fun getMax(name: String?): MaxAggregation {
        return MaxAggregation(dataFrame.getColumn(name)!!)
    }

    fun getMean(name: String?): MeanAggregation {
        return MeanAggregation(dataFrame.getColumn(name)!!)
    }

    fun getVariance(name: String?): VarianceAggregation {
        return VarianceAggregation(dataFrame.getColumn(name)!!)
    }

    fun getVariance(name: String?, population: String?): VarianceAggregation {
        return VarianceAggregation(dataFrame.getColumn(name)!!, dataFrame.getColumn(population)!!)
    }

    fun getStdDev(name: String?): StdDevAggregation {
        return StdDevAggregation(dataFrame.getColumn(name)!!)
    }

    fun getCountDistinct(name: String?): CountDistinctAggregation {
        return CountDistinctAggregation(dataFrame.getColumn(name)!!)
    }

    fun getCountDistinctWithNull(name: String?): CountDistinctWithNullAggregation {
        return CountDistinctWithNullAggregation(dataFrame.getColumn(name)!!)
    }

    override fun toString(): String {
        return "InMemoryCube{" +
                "cuboids=" + cuboids.keys +
                '}'
    }

    private class DimensionsDiceStructure(dimensions: Dimensions?, dice: Dice?, structure: Structure?) {
        private val dimensions: Dimensions?
        private val dice: Dice?
        private val structure: Structure?
        override fun equals(o: Any?): Boolean {
            if (this === o) return true
            if (o == null || this::class != o::class) return false
            val that = o as DimensionsDiceStructure
            if (!dice!!.equals(that.dice)) return false
            return if (dimensions != that.dimensions) false else structure == that.structure
        }

        override fun hashCode(): Int {
            var result: Int = dimensions.hashCode()
            result = 31 * result + dice.hashCode()
            result = 31 * result + structure.hashCode()
            return result
        }

        override fun toString(): String {
            return "DimensionsDiceStructure{" +
                    "dimensions=" + dimensions +
                    ", dice=" + dice +
                    ", structure=" + structure +
                    '}'
        }

        init {
            this.dimensions = dimensions
            this.dice = dice
            this.structure = structure
        }
    }

    init {
        this.dataFrame = dataFrame
        root = Dimensions.create()
        dice = Dice.create()
        structure = Structure.create()
        apex = ApexCuboid(this, root)
        cuboids[DimensionsDiceStructure(root, dice, structure)] = apex
        dataFrame.addWeakDataFrameListener(listener)
    }
}