/*
 * Copyright (c) 2018 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.convert

import org.locationtech.jts.geom.Geometry
import org.locationtech.jts.io.WKTReader
import kotlin.reflect.KClass

class ToGeometryTypeConverter : TypeConverter {
    val wktTypes: Array<String> = arrayOf(
        "POINT", "LINESTRING",
        "LINEARRING", "POLYGON", "MULTIPOINT", "MULTILINESTRING",
        "MULTIPOLYGON", "GEOMETRYCOLLECTION"
    )

    val geometryClasses: Set<KClass<*>> = setOf(
        org.locationtech.jts.geom.Point::class, org.locationtech.jts.geom.LineString::class,
        org.locationtech.jts.geom.LinearRing::class, org.locationtech.jts.geom.Polygon::class, org.locationtech.jts.geom.MultiPoint::class, org.locationtech.jts.geom.MultiLineString::class,
        org.locationtech.jts.geom.MultiPolygon::class, org.locationtech.jts.geom.GeometryCollection::class
    )

    override fun isConvertable(input: TypeConverter.Input): Boolean {
        try {
            if (!isAssignableFrom(Geometry::class, input.getType())) {
                var allNull = true
                for (row in 0..<input.size()) {
                    val v: Any? = input.get(row)
                    if (v != null) {
                        if (v is String) {
                            var s = v.trim { it <= ' ' }
                            if (s != "") {
                                allNull = false
                                var parsed = false

                                if (s.startsWith("SRID")) {
                                    s = s.substring(s.indexOf(";") + 1)
                                }

                                var found = false
                                for (type in wktTypes) {
                                    if (s.startsWith(type)) {
                                        found = true
                                    }
                                }
                                if (!found) {
                                    return false
                                }

                                try {
                                    val reader: WKTReader = WKTReader()
                                    reader.read(s)
                                    parsed = true
                                } catch (e: org.locationtech.jts.io.ParseException) {
                                }
                                if (!parsed) {
                                    return false
                                }
                            }
                        } else if (v !is Geometry) {
                            return false
                        }
                    }
                }
                return !allNull
            } else {
                return false
            }
        } catch (e: Error) {
            return false
        } catch (e: RuntimeException) {
            return false
        } catch (e: Exception) {
            return false
        }
    }

    override fun convert(input: TypeConverter.Input, output: TypeConverter.Output): KClass<*> {
        for (row in 0..<input.size()) {
            val v: Any? = input.get(row)
            if (v != null) {
                if (v is String) {
                    var s = v.trim { it <= ' ' }
                    if (s != "") {
                        var parsed = false

                        if (s.startsWith("SRID")) {
                            s = s.substring(s.indexOf(";") + 1)
                        }

                        var found = false
                        for (type in wktTypes) {
                            if (s.startsWith(type)) {
                                found = true
                            }
                        }
                        if (!found) {
                            output.set(row, null)
                        }

                        try {
                            val reader: WKTReader = WKTReader()
                            output.set(row, reader.read(s))
                            parsed = true
                        } catch (e1: Exception) {
                        }
                        if (!parsed) {
                            output.set(row, null)
                        }
                    } else {
                        output.set(row, null)
                    }
                } else if (v !is Geometry) {
                    output.set(row, null)
                }
            }
        }
        return Geometry::class
    }


    override val type: KClass<*>
        get() = Geometry::class

    fun isAssignableFrom(a: KClass<*>, b:KClass<*>?): Boolean {
        return b != null && (a == b || b in geometryClasses)
    }
}
