/*
 * Copyright (c) 2015 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.subset

import com.macrofocus.common.collection.TObjectDoubleHashMap
import com.macrofocus.common.collection.TObjectDoubleMap
import org.molap.dataframe.DataFrame
import org.molap.series.Series
import kotlin.math.max

class DefaultGroup<Row, Bin> : Group<Row, Bin> {
    private val reducer: Reducer<Row, Bin>
    private var map: TObjectDoubleMap<Bin>
    private var activeMap: TObjectDoubleMap<Bin>
    private var max: Double? = null
    private var activeMax: Double? = null
    private var initialized = false
    override var sumValue: Double
        private set
    override var activeSumValue: Double
        private set

    constructor(initialCapacity: Int, reducer: Reducer<Row, Bin>, series: Series<Row, Bin>) {
        this.reducer = reducer
        activeMap = TObjectDoubleHashMap<Bin>(initialCapacity, 0.5f).withDefault({ 0.0 })
        sumValue = 0.0
        for (row in series.keys()!!) {
            val value = series[row]
            add(value, row)
        }
        activeSumValue = sumValue
        map = TObjectDoubleHashMap(activeMap)
        initialized = true
    }

    constructor(
        initialCapacity: Int,
        reducer: Reducer<Row, Bin>,
        dataFrame: DataFrame<Row, *, *>,
        binningStrategy: SingleBinningDimension.SingleBinningStrategy<Row, Bin>
    ) {
        this.reducer = reducer
        activeMap = TObjectDoubleHashMap<Bin>(initialCapacity, 0.5f).withDefault({ 0.0 })
        sumValue = 0.0
        for (row in dataFrame.rows()) {
            if (binningStrategy.isBinnable(row)) {
                val value: Bin = binningStrategy!!.rowToBin(row)
                add(value, row)
            }
        }
        activeSumValue = sumValue
        map = TObjectDoubleHashMap(activeMap)
        initialized = true
    }

    constructor(
        initialCapacity: Int,
        reducer: Reducer<Row, Bin>,
        dataFrame: DataFrame<Row, *, *>,
        binningStrategy: MultiBinningDimension.MultiBinningStrategy<Row, Bin>
    ) {
        this.reducer = reducer
        activeMap = TObjectDoubleHashMap<Bin>(initialCapacity, 0.5f).withDefault({ 0.0 })
        sumValue = 0.0
        for (row in dataFrame.rows()) {
            if (binningStrategy!!.isBinnable(row)) {
                val values: Array<Bin> = binningStrategy!!.rowToBins(row)
                for (i in values.indices) {
                    val value = values[i]
                    add(value, row)
                }
            }
        }
        activeSumValue = sumValue
        map = TObjectDoubleHashMap(activeMap)
        initialized = true
    }

    protected fun getReducer(): Reducer<*, *> {
        return reducer
    }

    override fun add(bin: Bin, row: Row) {
        val sum: Double = reducer.add(activeMap, bin, row)
        activeMax = if (activeMax == null) {
            sum
        } else {
            max(activeMax!!, sum)
        }
        max = if (max == null) {
            sum
        } else {
            max(max!!, sum)
        }
        val number: Number? = reducer.getNumber(row)
        if (number != null) {
            val d = number.toDouble()
            if (!initialized) {
                sumValue += d
            }
            activeSumValue += d
        }
    }

    override fun remove(key: Bin, row: Row) {
        val max = activeMax != null && activeMax!! <= activeMap.get(key)!!
        reducer.remove(activeMap, key, row)
        if (max) {
            activeMax = null
        }
        val number: Number? = reducer.getNumber(row)
        if (number != null) {
            val d = number.toDouble()
            activeSumValue -= d
        }
    }

    private fun computeMaxValue(): Double {
        return activeMap.maxOf { it.value }
    }

    override fun getValue(key: Bin): Double {
        return activeMap.get(key)!!
    }

    override val maxValue: Double
        get() {
            if (max == null) {
                max = computeMaxValue()
            }
            return max!!
        }
    override val activeMaxValue: Double
        get() {
            if (activeMax == null) {
                activeMax = computeMaxValue()
            }
            return activeMax!!
        }

    override fun reset() {
        activeSumValue = sumValue
        activeMax = null
        activeMap = TObjectDoubleHashMap(map)
    }
}