/*
 * Copyright (c) 2016 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.subset

import kotlin.jvm.JvmOverloads
import kotlin.math.*

/**
 * Created by luc on 13/07/16.
 */
class GenedataDistributionStrategy<Value> : DistributionDimension.DistributionStrategy<Value, Int> {
    private lateinit var virtualIndices: IntArray
    private lateinit var counts: IntArray
    var totalPopulation = 0
        private set
    var minPopulation = 0
        private set
    var minNonZeroPopulation = 0
        private set
    var maxPopulation = 0
        private set
        get() = field
    var binWidth = 0.0
        private set
    override var minValue = 0.0
        private set
    private var minimumMiddle = 0.0
    private var minimumUpperBound = 0.0
    override var maxValue = 0.0
        private set
    var centralValue = 0.0
        private set
    var range: Range? = null
        private set
    val numberOfBins: Int
        get() = binCount

    fun getCountAtBin(index: Int): Int {
        return getPopulationAt(index)
    }

    override fun getBinStartValue(index: Int): Double {
        return minValue + (maxValue - minValue) * index / binCount
    }

    override fun getBinEndValue(index: Int): Double {
        return minValue + (maxValue - minValue) * (index + 1) / binCount
    }

    override fun isBinnable(value: Value): Boolean {
        return value != null && (value is Number) // || value is java.util.Date)
    }

    override fun valueToBin(value: Value): Int {
//        return if (value != null) {
//            if (value is java.util.Date) {
//                getBin((value as java.util.Date).getTime().toDouble())
//            } else if (value is Number) {
                return getBin((value as Number).toDouble())
//            } else {
//                null
//            }
//        } else {
//            null
//        }
    }

    protected constructor() {}

    @JvmOverloads
    constructor(sortedValues: Array<Number>, count: Int = sortedValues.size) {
        init(sortedValues, count, 2.0)
    }

    constructor(sortedValues: Array<Number>, count: Int, coef: Double) {
        init(sortedValues, count, coef)
    }

    constructor(sortedValues: Array<Number>, centralValue: Double, binWidth: Double) {
        init(sortedValues, binCount, centralValue, binWidth)
    }

    fun init(sortedValues: Array<Number>, valueCount: Int, coef: Double) {
        require(valueCount != 0) { "Cannot make histogram from empty array." }
        val median = quantile(0.5, sortedValues, valueCount)
        val iq = quantile(0.75, sortedValues, valueCount) - quantile(0.25, sortedValues, valueCount)
        var binWidth: Double = coef * iq * valueCount.toDouble().pow(-0.33333333333333331)
        if (binWidth == 0.0 || com.macrofocus.common.math.isNaN(binWidth) || com.macrofocus.common.math.isInfinite(binWidth)) {
            binWidth = (sortedValues[valueCount - 1].toDouble() - sortedValues[0].toDouble()) / 10
            if (binWidth == 0.0 || com.macrofocus.common.math.isNaN(binWidth) || com.macrofocus.common.math.isInfinite(binWidth)) binWidth = 1.0
        }
        init(sortedValues, valueCount, median, binWidth)
    }

    fun init(sortedValues: Array<Number>, valueCount: Int, centralValue: Double, binWidth: Double) {
        require(valueCount != 0) { "Cannot make histogram from empty array." }
        range = Range(sortedValues[0].toDouble(), sortedValues[valueCount - 1].toDouble())
        this.centralValue = centralValue
        this.binWidth = binWidth
        totalPopulation = valueCount
        virtualIndices = IntArray(totalPopulation)
        counts = IntArray(totalPopulation)
        val centralValueRightBoundary = centralValue + 0.5 * binWidth
        minimumUpperBound = centralValueRightBoundary - binWidth * floor((centralValueRightBoundary - sortedValues[0].toDouble()) / binWidth)
        if (sortedValues[0].toDouble() > minimumUpperBound) minimumUpperBound += binWidth
        minValue = minimumUpperBound - binWidth
        minimumMiddle = minimumUpperBound - binWidth / 2.0
        var k = 0
        maxValue = minimumUpperBound
        minPopulation = totalPopulation
        minNonZeroPopulation = totalPopulation
        maxPopulation = 0
        for (i in 0 until valueCount) {
            if (sortedValues[i].toDouble() > maxValue) {
                if (counts[k] < minPopulation) minPopulation = counts[k]
                if (counts[k] < minNonZeroPopulation && counts[k] > 0) minNonZeroPopulation = counts[k]
                if (counts[k] > maxPopulation) maxPopulation = counts[k]
                val deltaIndex = 1 + ((sortedValues[i].toDouble() - maxValue) / binWidth).toInt()
                virtualIndices[k + 1] = virtualIndices[k] + deltaIndex
                k++
                maxValue += deltaIndex.toDouble() * binWidth
            }
            counts[k]++
        }
        if (maxPopulation == 0) maxPopulation = totalPopulation
        virtualIndices = trim(virtualIndices, ++k)
        counts = trim(counts, k)
    }

    fun getBin(v: Double): Int {
        if (v < minValue) return -1
        return if (v > maxValue) -2 else ((v - minValue) / binWidth).toInt()
    }

    val binCount: Int
        get() = 1 + virtualIndices[virtualIndices.size - 1]

    fun getBinMiddle(i: Int): Double {
        return minimumMiddle + i.toDouble() * binWidth
    }

    fun getLowerResidual(i: Int): Double {
        if (i < 0) return 0.0
        var s = 0
        var k = 0
        while (k < virtualIndices.size && virtualIndices[k] < i) {
            s += counts[k]
            k++
        }
        return s.toDouble() * 100 / totalPopulation.toDouble()
    }

    fun getPopulationAt(i: Int): Int {
        val k: Int = com.macrofocus.common.collection.binarySearch(virtualIndices, i)
        return if (k < 0) 0 else counts[k]
    }

    fun getUpperResidual(i: Int): Double {
        if (i + 1 > virtualIndices[virtualIndices.size - 1]) return 0.0
        var s = 0
        var k: Int = com.macrofocus.common.collection.binarySearch(virtualIndices, i + 1)
        if (k < 0) k = -(k + 1)
        while (k < virtualIndices.size) {
            s += counts[k]
            k++
        }
        return s.toDouble() * 100 / totalPopulation.toDouble()
    }

    override fun toString(): String {
        return ("Histogram, range=" + range + ", total=" + totalPopulation + ", binWidth=" + binWidth + ", centralValue=" + centralValue
                + ", binCount=" + binCount + ", minRange=" + range!!.min + ", maxRange=" + range!!.max + ", range=" + range!!.range + ", binRange=" + binWidth * binCount
                + ", minimumLowerBound=" + minValue + ", maximumUpperBound=" + maxValue + ", binWidth=" + (maxValue - minValue) / binCount)
    }

    private fun quantile(quantile: Double, data: Array<Number>, count: Int): Double {
        var virtualIndex = (count - 1).toDouble() * quantile
        val baseIndex = virtualIndex.toInt()
        return if (baseIndex == count - 1) {
            data[baseIndex].toDouble()
        } else {
            virtualIndex -= baseIndex.toDouble()
            data[baseIndex].toDouble() * (1.0 - virtualIndex) + data[baseIndex + 1].toDouble() * virtualIndex
        }
    }

    class Range {
        override fun toString(): String {
            return "[$min, $max]"
        }

        val range: Double
            get() = if (com.macrofocus.common.math.isInfinite(max - min)) (0.0f / 0.0f).toDouble() else max - min

        constructor(min: Double, max: Double) {
            this.min = (0.0f / 0.0f).toDouble()
            this.max = (0.0f / 0.0f).toDouble()
            if (min <= max) {
                this.min = min
                this.max = max
            }
        }

        constructor(roleModel: Range) {
            min = (0.0f / 0.0f).toDouble()
            max = (0.0f / 0.0f).toDouble()
            min = roleModel.min
            max = roleModel.max
        }

        constructor(ranges: Array<Range>) {
            min = (0.0f / 0.0f).toDouble()
            max = (0.0f / 0.0f).toDouble()
            if (ranges.size >= 1) {
                min = ranges[0].min
                max = ranges[0].max
                for (i in 1 until ranges.size) {
                    min = min(min, ranges[i].min)
                    max = max(max, ranges[i].max)
                }
            }
        }

        constructor(data: DoubleArray) {
            min = (0.0f / 0.0f).toDouble()
            max = (0.0f / 0.0f).toDouble()
            widen(data)
        }

        constructor(data: DoubleArray, valids: IntArray) {
            min = (0.0f / 0.0f).toDouble()
            max = (0.0f / 0.0f).toDouble()
            widen(data, valids)
        }

        constructor(data: Array<DoubleArray>) {
            min = (0.0f / 0.0f).toDouble()
            max = (0.0f / 0.0f).toDouble()
            for (i in data.indices) widen(data[i])
        }

        fun scale(scale: Double): Range {
            return Range(min * scale, max * scale)
        }

        fun log(): Range {
            return Range(ln(min), ln(max))
        }

        fun getSubRanges(count: Int): Array<Range?> {
            val result = arrayOfNulls<Range>(count)
            val min = min
            val range = range
            for (i in result.indices) result[i] =
                Range(min + range * i.toDouble() / result.size.toDouble(), min + range * (i + 1) as Double / result.size.toDouble())
            return result
        }

        operator fun contains(value: Double): Boolean {
            return value >= min && value <= max
        }

        fun getClippedValue(value: Double): Double {
            return min(max, max(min, value))
        }

        fun makeSymmetrical(): Range {
            val extent: Double = max(-min, max)
            return Range(-extent, extent)
        }

        fun ensureWidth(width: Double): Range {
            val halfDiff = (width - range) / 2.0f
            return if (halfDiff <= 0.0f) this else Range(min - halfDiff, max + halfDiff)
        }

        fun widenAbsolute(extent: Double): Range {
            return Range(min - extent, max + extent)
        }

        fun widenRelative(ratio: Double): Range {
            val extend = (max - min) * ratio
            return widenAbsolute(extend)
        }

        fun widen(data: DoubleArray) {
            for (i in data.indices) {
                min = min(data[i], min)
                max = max(data[i], max)
            }
        }

        fun widen(data: DoubleArray, valids: IntArray) {
            for (i in valids.indices) {
                min = min(data[valids[i]], min)
                max = max(data[valids[i]], max)
            }
        }

        fun toDoubleArray(): DoubleArray {
            return doubleArrayOf(
                min, max
            )
        }

        var min: Double
        var max: Double
    }

    companion object {
        fun trim(vector: IntArray, len: Int): IntArray {
            return if (vector.size == len) {
                vector
            } else {
                val truncated = IntArray(len)
                com.macrofocus.common.collection.arraycopy(vector, 0, truncated, 0, len)
                truncated
            }
        }
    }
}