/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.aggregates.cube

import org.molap.series.Series
import kotlin.math.sqrt

class DistributiveStatistics() {
    //
    var count: Long = 0
        private set
    var countOfMissing: Long = 0
        private set
    var sum: Double? = null
        private set
    var uncorrectedSumOfSquares: Double? = null
        private set
    var minimum: Number? = null
        private set
    var maximum: Number? = null
        private set

    constructor(series: Series<Any?, *>) : this(series, series.keys()) {}
    constructor(series: Series<Any?, *>, keys: Iterable<Any?>?) : this(series, keys, null) {}
    constructor(
        series: Series<Any?, *>,
        keys: Iterable<Any?>?,
        filter: RowFilter<Any?>?
    ) : this() {
        for (row in keys!!) {
//            assert(row != null) { "Missing row for series $series" }
            if (filter == null || filter.accept(row)) {
                val value: Any? = series.get(row)
                if (value != null) {
                    if (value is Number) {
                        addNumber(value)
                    }
                } else {
                    addMissing()
                }
            }
        }
    }

    fun addNumber(value: Number) {
        var count: Long = this.count
        var countOfMissing: Long = this.countOfMissing
        var sum: Double? = this.sum
        var uncorrectedSumOfSquares: Double? = this.uncorrectedSumOfSquares
        var minimum: Number? = this.minimum
        var maximum: Number? = this.maximum

        val v: Double = value.toDouble()
        if (sum != null) {
            sum += v
        } else {
            sum = v
        }
        if (uncorrectedSumOfSquares != null) {
            uncorrectedSumOfSquares += v * v
        } else {
            uncorrectedSumOfSquares = v * v
        }
        if (minimum == null || v < minimum.toDouble()) {
            minimum = value
        }
        if (maximum == null || v > maximum.toDouble()) {
            maximum = value
        }
        count++

        this.count = count
        this.countOfMissing = countOfMissing
        this.sum = sum
        this.uncorrectedSumOfSquares = uncorrectedSumOfSquares
        this.minimum = minimum
        this.maximum = maximum
    }

//    constructor(matrix: Matrix<R?, C?>?) : this() {
//        for (row in matrix.rows()) {
//            for (column in matrix.columns()) {
//                if (matrix.isAvailable(row, column)) {
//                    val v: Double = matrix.getDouble(row, column)
//                    addValue(v)
//                } else {
//                    addMissing()
//                }
//            }
//        }
//    }

    fun addMissing() {
        countOfMissing++
    }

    fun addValue(v: Double) {
        var count: Long = this.count
        var countOfMissing: Long = this.countOfMissing
        var sum: Double? = this.sum
        var uncorrectedSumOfSquares: Double? = this.uncorrectedSumOfSquares
        var minimum: Number? = this.minimum
        var maximum: Number? = this.maximum

        if (sum != null) {
            sum += v
        } else {
            sum = v
        }
        if (uncorrectedSumOfSquares != null) {
            uncorrectedSumOfSquares += v * v
        } else {
            uncorrectedSumOfSquares = v * v
        }
        if (minimum == null || v < minimum.toDouble()) {
            minimum = v
        }
        if (maximum == null || v > maximum.toDouble()) {
            maximum = v
        }
        count++

        this.count = count
        this.countOfMissing = countOfMissing
        this.sum = sum
        this.uncorrectedSumOfSquares = uncorrectedSumOfSquares
        this.minimum = minimum
        this.maximum = maximum
    }

    fun add(distributiveStatistics: DistributiveStatistics) {
        var sum: Double? = this.sum
        var uncorrectedSumOfSquares: Double? = this.uncorrectedSumOfSquares
        var minimum: Number? = this.minimum
        var maximum: Number? = this.maximum

        count += distributiveStatistics.count
        countOfMissing += distributiveStatistics.countOfMissing

        val osum = distributiveStatistics.sum
        if (osum != null) {
            if (sum != null) {
                sum += osum
            } else {
                this.sum = osum
            }
        }
        val otherUncorrectedSumOfSquares = distributiveStatistics.uncorrectedSumOfSquares
        if (otherUncorrectedSumOfSquares != null) {
            if (uncorrectedSumOfSquares != null) {
                uncorrectedSumOfSquares += otherUncorrectedSumOfSquares
            } else {
                uncorrectedSumOfSquares = otherUncorrectedSumOfSquares
            }
        }
        val otherMinimum = distributiveStatistics.minimum
        if (minimum == null || otherMinimum != null && otherMinimum.toDouble() < minimum.toDouble()) {
            minimum = otherMinimum
        }
        val otherMaximum = distributiveStatistics.maximum
        if (maximum == null || otherMaximum != null && otherMaximum.toDouble() > maximum.toDouble()) {
            maximum = otherMaximum
        }

        this.sum = sum
        this.uncorrectedSumOfSquares = uncorrectedSumOfSquares
        this.minimum = minimum
        this.maximum = maximum
    }

    val mean: Double?
        get() = if (count > 0) {
            sum!! / count
        } else {
            null
        }

    val range: Double?
        get() {
            var minimum: Number? = this.minimum
            var maximum: Number? = this.maximum
            if(maximum != null && minimum != null) {
                return maximum.toDouble() - minimum.toDouble()
            } else {
                return null;
            }
        }

    val correctedSumOfSquares: Double?
        get() = null

    val variance: Double?
        get() {
            val mean = mean
            return if (mean != null) {
                uncorrectedSumOfSquares!! / count - mean * mean
            } else {
                null
            }
        }

    val standardDeviation: Double?
        get() {
            val variance = variance
            return if (variance != null) {
                sqrt(variance)
            } else {
                null
            }
        }

    val standardErrorOfMean: Double?
        get() = null

    val coefficientOfVariance: Double?
        get() = null

    val tValue: Double?
        get() = null

    val probabilityOfGreaterAbsoluteValue: Double?
        get() = null

    val lowerConfidenceLimit: Double?
        get() = null

    val upperConfidenceLimit: Double?
        get() = null

    override fun toString(): String {
        return "DistributiveStatistics{" +
                "count=" + count +
                ", countOfMissing=" + countOfMissing +
                ", sum=" + sum +
                ", uncorrectedSumOfSquares=" + uncorrectedSumOfSquares +
                ", minimum=" + minimum +
                ", maximum=" + maximum +
                ", mean=" + mean +
                ", variance=" + variance +
                ", stdDev=" + standardDeviation +
                '}'
    }

    interface RowFilter<R> {
        fun accept(row: R): Boolean
    }
}