/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.dataframe

import com.macrofocus.common.collection.CollectionFactory
import com.macrofocus.common.collection.WeakReference
import org.molap.series.AbstractSeries
import org.molap.series.Series
import kotlin.reflect.KClass

/**
 * This class provides a skeletal implementation of the DataFrame interface to minimize the effort required to
 * implement this interface.
 */
abstract class AbstractDataFrame<Row, Column, V> protected constructor() : DataFrame<Row, Column, V> {
    private val listeners: MutableList<DataFrameListener<Row, Column>> = CollectionFactory.copyOnWriteArrayList()
//    private val statisticsMap: MutableMap<Column?, UnivariateStatistics?>?
//    protected var cube: Cube? = null
//        protected get() {
//            if (field == null) {
//                field = InMemoryCube(this)
//            }
//            return field
//        }
//        private set

    override fun rows(): Iterable<Row> {
        return rowIndex.keys()
    }

    override fun columns(): Iterable<Column> {
        return columnIndex.keys()
    }

    override fun getRowKey(index: Int): Row {
        return rowIndex.getKey(index)
    }

    override fun getColumnKey(index: Int): Column {
        return columnIndex.getKey(index)
    }

    override fun getRowAddress(row: Row): Int {
        return rowIndex.getAddress(row)
    }

    override fun getColumnAddress(column: Column): Int {
        return columnIndex.getAddress(column)
    }

    override val rowCount: Int
        get() = rowIndex.size

    override val columnCount: Int
        get() = columnIndex.size

    override fun getColumn(column: Column): Series<Row, V>? {
        return ColumnSeries(column)
    }

//    fun orderRows(vararg sortKeys: SortKey<Column?>?): DataFrame<Row?, Column?, V?>? {
//        return ReMappedDataFrame(this, OrderReMappedRecipe(*sortKeys))
//    }

//    override fun <C> reindexColumns(vararg columns: C?): DataFrame<Row?, C?, V?>? {
//        return ReIndexedDataFrame(this, ColumnsReIndexRecipe<Any?>(*columns))
//    }

//    override fun remapColumns(vararg columns: Column?): DataFrame<Row?, Column?, V?>? {
//        return ReMappedDataFrame(this, ColumnsReMappedRecipe(*columns))
//    }

//    override fun removeColumns(vararg columns: Column?): DataFrame<Row?, Column?, V?>? {
//        return ReMappedDataFrame(this, RemoveReMappedRecipe(*columns))
//    }

//    override fun append(dataFrame: DataFrame<Row?, Column?, V?>?): DataFrame<Row?, Column?, V?>? {
//        return AppendDataFrame<Row?, Column?, V?>(this, dataFrame)
//    }

//    override fun appendAndReindex(dataFrame: DataFrame<Row?, Column?, V?>?): DataFrame<Int?, Column?, V?>? {
//        return AppendAndReindexDataFrame<Row?, Column?, V?>(this, dataFrame)
//    }

//    fun filter(filter: MutableFilter<Row?>?): FilterDataFrame? {
//        return FilterDataFrame<Row?, Column?, V?>(this, filter)
//    }

//    override fun removeDuplicates(vararg columns: Column?): DataFrame<Row?, Column?, V?>? {
//        return ReMappedDataFrame(this, RemoveDuplicatesReMappedRecipe(*columns))
//    }

//    fun getStatistics(column: Column?): UnivariateStatistics? {
//        return if (statisticsMap!!.containsKey(column)) {
//            statisticsMap[column]
//        } else {
//            val univariateStatistics = UnivariateStatistics(rows(), getColumn(column))
//            statisticsMap[column] = univariateStatistics
//            univariateStatistics
//        }
//    }

//    fun aggregate(vararg aggregations: Aggregation?): AggregateDataFrame<Column?>? {
//        return aggregate(true, *aggregations)
//    }

//    fun aggregate(includeIndex: Boolean, vararg aggregations: Aggregation?): AggregateDataFrame<Column?>? {
//        return QueryDataFrame(CuboidQuery(cube.getApexCuboid(), aggregations), includeIndex)
//    }

//    val dataFrameAggregation: DataFrameAggregation?
//        get() = DataFrameAggregation()

//    fun getFirst(column: Column?): FirstAggregation? {
//        return FirstAggregation(getColumn(column))
//    }

//    fun getConstant(value: Any?): ConstantAggregation? {
//        return ConstantAggregation(value)
//    }
//
//    fun getRandom(min: Double, max: Double): RandomAggregation? {
//        return RandomAggregation(min, max)
//    }
//
//    fun getSum(column: Column?): SumAggregation? {
//        return SumAggregation(getColumn(column))
//    }
//
//    fun getCount(column: Column?): CountAggregation? {
//        return CountAggregation(getColumn(column))
//    }
//
//    fun getDistributiveStatistics(column: Column?): DistributiveStatisticsAggregation? {
//        return DistributiveStatisticsAggregation(getColumn(column))
//    }
//
//    fun getMin(column: Column?): MinAggregation? {
//        return MinAggregation(getColumn(column))
//    }
//
//    fun getMax(column: Column?): MaxAggregation? {
//        return MaxAggregation(getColumn(column))
//    }
//
//    fun getMean(column: Column?): MeanAggregation? {
//        return MeanAggregation(getColumn(column))
//    }
//
//    fun getVariance(column: Column?): VarianceAggregation? {
//        return VarianceAggregation(getColumn(column))
//    }
//
//    fun getVarianceByPopulation(column: Column?, population: Column?): VarianceAggregation? {
//        return VarianceAggregation(getColumn(column), getColumn(population))
//    }
//
//    fun getStdDev(column: Column?): StdDevAggregation? {
//        return StdDevAggregation(getColumn(column))
//    }
//
//    fun getUnivariateStatistics(column: Column?): UnivariateStatisticsAggregation? {
//        return UnivariateStatisticsAggregation(getColumn(column))
//    }
//
//    fun getMedian(column: Column?): MedianAggregation? {
//        return MedianAggregation(getColumn(column))
//    }
//
//    fun getFirstQuartile(column: Column?): FirstQuartileAggregation? {
//        return FirstQuartileAggregation(getColumn(column))
//    }
//
//    fun getThirdQuartile(column: Column?): ThirdQuartileAggregation? {
//        return ThirdQuartileAggregation(getColumn(column))
//    }
//
//    fun getWeightedSum(weight: Column?, column: Column?): Aggregation? {
//        return SumAggregation(getColumn(column)!!.multiply(getColumn(weight)))
//    }
//
//    fun getWeightedMean(weight: Column?, column: Column?): Aggregation? {
//        return getWeightedSum(weight, column).divide(getSum(weight))
//    }
//
//    fun getCountDistinct(column: Column?): CountDistinctAggregation? {
//        return CountDistinctAggregation(getColumn(column))
//    }
//
//    fun getCountDistinctWithNull(column: Column?): CountDistinctWithNullAggregation? {
//        return CountDistinctWithNullAggregation(getColumn(column))
//    }
//
//    fun getCentroid(column: Column?): CentroidAggregation? {
//        return CentroidAggregation(getColumn(column))
//    }

    override fun getColumnName(column: Column): String? {
        return column?.toString()
    }

    override fun printSchema() {
        for (column in columns()) {
            print(abbreviate(getColumnName(column), 19))
            print(": ")
            val value: KClass<*> = getColumnClass(column)
            print(abbreviate(if (value != null) value.simpleName else "", 19))
            println()
        }
//        val out: PrintStream = java.lang.System.out
//        for (column in columns()!!) {
//            out.printf("%20s", abbreviate(getColumnName(column), 19))
//            out.print(": ")
//            val value: java.lang.Class? = getColumnClass(column)
//            out.printf(
//                "%20s",
//                abbreviate(
//                    if (value != null) value.getSimpleName() else "",
//                    19
//                )
//            )
//            out.println()
//        }
    }

    override fun print() {
//        val out: PrintStream = java.lang.System.out
//        out.printf("%10s", abbreviate("<Index>", 9))
//        for (column in columns()!!) {
//            out.printf("%20s", abbreviate(getColumnName(column), 19))
//        }
//        out.println()
//        out.printf("%10s", "", 9)
//        for (column in columns()!!) {
//            val value = getColumnClass(column)
//            out.printf(
//                "%20s",
//                abbreviate(
//                    if (value != null) value.getSimpleName() else "",
//                    19
//                )
//            )
//        }
//        out.println()
//        for (row in rows()!!) {
//            out.printf("%10s", abbreviate(row.toString(), 9))
//            //            System.out.print(row + "\t");
//            for (column in columns()!!) {
//                val value: Any? = getValueAt(row, column)
//                if (value != null) {
//                    if (value is Number) {
//                        out.printf(
//                            "%20s",
//                            abbreviate(
//                                DecimalFormat.getNumberInstance().format(value), 19
//                            )
//                        )
//                    } else {
//                        out.printf(
//                            "%20s",
//                            abbreviate(value.toString(), 19)
//                        )
//                    }
//                } else {
//                    out.printf("%20s", "(null)", 19)
//                }
//            }
//            out.println()
//        }
    }

//    fun print(out: PrintStream?, caption: String?, html: Boolean) {
//        if (!html) {
//            caption?.let { println(it) }
//            val cols = columnCount
//            for (i in 0 until cols) {
//                val o = getColumnName(getColumnKey(i))
//                val s: String
//                s = o ?: "n.a."
//                out.printf("%20s", abbreviate(s, 19))
//            }
//            //        System.out.println();
//            //        for (int i = 0; i < cols; i++) {
//            //            final String o = tableModel.getColumnClass(i).getSimpleName();
//            //            final String s;
//            //            if(o != null) {
//            //                s = o;
//            //            } else {
//            //                s = "n.a.";
//            //            }
//            //            out.printf("%20s", abbreviate(s, 19));
//            //        }
//            out.println()
//            for (row in 0 until rowCount) {
//                for (i in 0 until cols) {
//                    val o: Any? = getValueAt(getRowKey(row), getColumnKey(i))
//                    val s: String
//                    s = o?.toString() ?: "n.a."
//                    out.printf("%20s", abbreviate(s, 19))
//                }
//                out.println()
//            }
//        } else {
//            out.println("<table style=\"width: auto; border-collapse: separate;\">")
//            if (caption != null) {
//                out.println("<caption>$caption</caption>")
//            }
//            out.println("<tbody>")
//            out.println("<tr>")
//            val cols = columnCount
//            for (i in 0 until cols) {
//                val o = getColumnName(getColumnKey(i))
//                val s: String
//                s = o ?: "n.a."
//                out.print("<th>")
//                out.print(s)
//                out.println("</th>")
//            }
//            out.println("</tr>")
//            for (row in 0 until rowCount) {
//                out.println("<tr>")
//                for (i in 0 until cols) {
//                    val o: Any? = getValueAt(getRowKey(row), getColumnKey(i))
//                    val s: String
//                    s = o?.toString() ?: "n.a."
//                    if (o is Number) {
//                        out.print("<td align=\"right\">")
//                    } else {
//                        out.print("<td>")
//                    }
//                    out.print(s)
//                    out.println("</td>")
//                }
//                out.println("</tr>")
//            }
//            out.println("</tbody>")
//            out.println("</table>")
//        }
//    }

//    override fun benchmark() {
//        val now: Long = java.lang.System.currentTimeMillis()
//        for (i in 0..99) {
////            for (K column : columns()) {
////            }
////
////            for (K column : columns()) {
////            }
//            for (column in columns()!!) {
//                val series: Series<*, *>? = getColumn(column)
//                for (row in rows()!!) {
////                    Object value = getValue(row, column);
//                    val value = series!!.get(row)
//                }
//            }
//        }
//        println("Benchmark: " + (java.lang.System.currentTimeMillis() - now) + " ms")
//    }

//    override fun reindexRows(): DataFrame<Int?, Column?, V?>? {
//        return ReIndexedDataFrame<Int?, Column?, V?, Row?, Column?>(
//            this,
//            IntegerReIndexRecipe()
//        )
//    }

//    override fun reindexRowsUsingColumn(column: Column?): MutableDataFrame<V?, Column?, V?>? {
//        return reindexRowsUsingColumnDefault(true, column)
//    }

//    override fun reindexRowsUsingColumnDefault(
//        keepColumn: Boolean,
//        column: Column?
//    ): MutableDataFrame<V?, Column?, V?>? {
//        return ReIndexedDataFrame<V?, Column?, V?, Row?, Column?>(
//            this,
//            ColumnDefaultReIndexRecipe(column, keepColumn)
//        )
//    }

//    override fun reindexRowsUsingColumns(vararg columns: Column?): DataFrame<MultiKey?, Column?, V?>? {
//        return reindexRowsDefault(true, *columns)
//    }

//    override fun reindexRowsDefault(
//        keepColumns: Boolean,
//        vararg columns: Column?
//    ): DataFrame<MultiKey?, Column?, V?>? {
//        return ReIndexedDataFrame<MultiKey?, Column?, V?, Row?, Column?>(
//            this,
//            RowsDefaultReIndexRecipe(keepColumns, *columns)
//        )
//    }

    override fun addDataFrameListener(listener: DataFrameListener<Row, Column>) {
        listeners.add(listener)
    }

    override fun addWeakDataFrameListener(listener: DataFrameListener<Row, Column>) {
        listeners.add(WeakDataFrameListener(listener))
    }

    override fun removeDataFrameListener(listener: DataFrameListener<Row, Column>) {
        if (listener is AbstractDataFrame<Row, Column, *>.WeakDataFrameListener) {
            listeners.remove(listener)
        } else {
            var toRemove: DataFrameListener<Row, Column>? = null
            for (dataFrameListener in listeners) {
                var comparable: DataFrameListener<Row, Column>?
                if (dataFrameListener is AbstractDataFrame<Row, Column, *>.WeakDataFrameListener) {
                    comparable = dataFrameListener.reference
                } else {
                    comparable = dataFrameListener
                }
                if (listener == comparable) {
                    toRemove = dataFrameListener
                }
            }
            if (toRemove != null) {
                listeners.remove(toRemove)
            }
        }
    }

    override fun removeDataFrameListeners() {
        listeners.clear()
    }

    /**
     * Fires an event event to listeners that have been registered to track changes to the data frame.
     *
     * @param event the event describing the changes
     */
    protected fun notifyDataFrameChanged(event: DataFrameEvent<Row, Column>) {
//        statisticsMap!!.clear()
        for (listener in listeners) {
            listener.dataFrameChanged(event)
        }
    }

    private inner class WeakDataFrameListener(listener: DataFrameListener<Row, Column>) : DataFrameListener<Row, Column> {
        private val l_ref: WeakReference<DataFrameListener<Row, Column>> = WeakReference<DataFrameListener<Row, Column>>(listener)

        override fun dataFrameChanged(event: DataFrameEvent<Row, Column>) {
            val l = reference
            if (l != null) {
                l.dataFrameChanged(event)
            } else {
                removeDataFrameListener(this)
            }
        }

        val reference: DataFrameListener<Row, Column>?
            get() = l_ref.get()
    }

    inner class ColumnSeries(var column: Column) : AbstractSeries<Row, V>() {
        override val name: Any?
            get() = column

        override val type: KClass<out Any>
            get() = getColumnClass(column)

        override operator fun get(key: Row): V {
            return getValueAt(key, column)
        }

        override fun getKey(i: Int): Row {
            return rowIndex!!.getKey(i)
        }

        override fun size(): Int {
            return rowIndex.size
        }

        override fun getAddress(key: Row): Int {
            return getRowAddress(key)
        }

        override fun keys(): Iterable<Row> {
            return rowIndex!!.keys()
        }

        fun <L> reindex(vararg keys: L?): Series<L?, V?>? {
            return null
        }

//        override fun head(count: Int): Series<Row, V> {
//            return null
//        }
//
//        override fun tail(count: Int): Series<Row, V> {
//            return null
//        }

        init {
//            assert(column != null)
            this.column = column
        }
    }

//    private inner class SortComparator(vararg val sortKeys: SortKey<Column>) : Comparator<Row> {
////        private val sortKeys: Array<SortKey<Column>> = sortKeys
//        override fun compare(o1: Row, o2: Row): Int {
//            for (sortKey in sortKeys) {
//                val compare: Int = sortKey.compare(this@AbstractDataFrame, o1, o2)
//                if (compare != null && compare != 0) {
//                    return compare
//                }
//            }
//            val x = getRowAddress(o1)
//            val y = getRowAddress(o2)
//            return if (x < y) -1 else if (x == y) 0 else 1
//        }
//    }

//    private inner class ColumnsReIndexRecipe<C>(vararg columns: C?) : ReIndexRecipe<Row?, C?> {
//        private val columns: Array<C?>?
//        fun buildRowIndex(): UniqueIndex<Row?>? {
//            return rowIndex
//        }
//
//        fun buildColumnIndex(): UniqueIndex<C?>? {
//            return DefaultUniqueIndex.< C > fromArray < C ? > columns
//        }
//
//        init {
//            this.columns = columns
//        }
//    }

//    private inner class OrderReMappedRecipe(vararg sortKeys: SortKey<Column?>?) :
//        ReMappedRecipe<Row?, Column?> {
//        private val sortKeys: Array<SortKey<Column?>?>?
//        fun buildRowIndex(): UniqueIndex<Row?>? {
//            return DefaultUniqueIndex(
//                rowIndex,
//                SortComparator(*sortKeys!!)
//            )
//        }
//
//        fun buildColumnIndex(): UniqueIndex<Column?>? {
//            return columnIndex
//        }
//
//        init {
//            this.sortKeys = sortKeys
//        }
//    }

//    private inner class ColumnsReMappedRecipe(vararg columns: Column?) : ReMappedRecipe<Row?, Column?> {
//        private val columns: Array<Column?>?
//        fun buildRowIndex(): UniqueIndex<Row?>? {
//            return rowIndex
//        }
//
//        fun buildColumnIndex(): UniqueIndex<Column?>? {
//            return DefaultUniqueIndex.< Column > fromArray < Column ? > columns
//        }
//
//        init {
//            this.columns = columns
//        }
//    }

//    private inner class RemoveReMappedRecipe(vararg columns: Column?) : ReMappedRecipe<Row?, Column?> {
//        private val columns: Array<Column?>?
//        fun buildRowIndex(): UniqueIndex<Row?>? {
//            return rowIndex
//        }
//
//        fun buildColumnIndex(): UniqueIndex<Column?>? {
//            return columnIndex!!.difference(DefaultUniqueIndex(columns))
//        }
//
//        init {
//            this.columns = columns
//        }
//    }

//    private inner class RemoveDuplicatesReMappedRecipe(vararg columns: Column?) :
//        ReMappedRecipe<Row?, Column?> {
//        private val columns: Array<Column?>?
//        fun buildRowIndex(): UniqueIndex<Row?>? {
//            val rows: MutableList<Row?> = java.util.ArrayList<Row?>()
//            val unique: MutableSet<MultiKey?> = HashSet()
//            for (row in rows()!!) {
//                val values = arrayOfNulls<Any?>(columns!!.size)
//                var c = 0
//                for (column in columns) {
//                    values[c++] = getValueAt(row, column)
//                }
//                val key = MultiKey(*values)
//                if (!unique.contains(key)) {
//                    rows.add(row)
//                    unique.add(key)
//                }
//            }
//            return DefaultUniqueIndex.< Row > fromArray < Row ? > rows.toTypedArray()
//        }
//
//        fun buildColumnIndex(): UniqueIndex<Column?>? {
//            return columnIndex
//        }
//
//        init {
//            this.columns = columns
//        }
//    }

//    private inner class IntegerReIndexRecipe : ReIndexRecipe<Int?, Column?> {
//        fun buildRowIndex(): UniqueIndex<Int?>? {
//            return IntegerRangeUniqueIndex(0, rowCount - 1)
//        }
//
//        fun buildColumnIndex(): UniqueIndex<Column?>? {
//            return columnIndex
//        }
//    }

//    private inner class RowsDefaultReIndexRecipe(private val keepColumns: Boolean, vararg columns: Column?) :
//        ReIndexRecipe<MultiKey?, Column?> {
//        private val columns: Array<Column?>?
//        fun buildRowIndex(): UniqueIndex<MultiKey?>? {
//            var r = 0
//            val keys = arrayOfNulls<MultiKey?>(rowCount)
//            for (row in rows()!!) {
//                val values = arrayOfNulls<Any?>(columns!!.size)
//                var c = 0
//                for (column in columns) {
//                    values[c++] = getValueAt(row, column)
//                }
//                val key = MultiKey(*values)
//                keys[r++] = key
//            }
//            return DefaultUniqueIndex.< MultiKey > fromArray < MultiKey ? > keys
//        }
//
//        fun buildColumnIndex(): UniqueIndex<Column?>? {
//            return if (keepColumns) {
//                columnIndex
//            } else {
//                columnIndex!!.difference(DefaultUniqueIndex.< Column > fromArray < Column ? > columns)
//            }
//        }
//
//        init {
//            this.columns = columns
//        }
//    }

    companion object {
        fun abbreviate(str: String?, maxWidth: Int): String? {
            if (str == null) {
                return null
            }
            return if (str.length <= maxWidth) {
                str
            } else str.substring(0, maxWidth - 3) + "..."
        }
    }

    init {
//        statisticsMap = HashMap<Column?, UnivariateStatistics?>()
    }
}