package org.molap.dataframe

import org.molap.index.UniqueIndex
import org.molap.series.Series
import kotlin.reflect.KClass

/**
 * A reindexed data frame.
 *
 * @param <R> the rexindexed type of rows
 * @param <C> the reindexed type of columns
 * @param <V> the type of values
 * @param <X> the original type of rows
 * @param <Y> the original type of columns
</Y></X></V></C></R> */
class ReIndexedDataFrame<R, C, V, X, Y>(
    protected val dataFrame: DataFrame<X, Y, V>,
    private var recipe: ReIndexRecipe<R, C>
) :
    AbstractMutableDataFrame<R, C, V>(), MutableDataFrame<R, C, V> {
    var cacheColumnIndex: UniqueIndex<C>? = null
    var cacheRowIndex: UniqueIndex<R>? = null
    override val columnIndex: UniqueIndex<C>
        get() = if (cacheColumnIndex == null) {
            cacheColumnIndex = recipe.buildColumnIndex()
            cacheColumnIndex!!
        } else {
            cacheColumnIndex!!
        }
    override val rowIndex: UniqueIndex<R>
        get() = if (cacheRowIndex == null) {
            cacheRowIndex = recipe.buildRowIndex()
            cacheRowIndex!!
        } else {
            cacheRowIndex!!
        }
    private val dataFrameListener: DataFrameListener<X, Y> = object : DataFrameListener<X, Y> {
        override fun dataFrameChanged(event: DataFrameEvent<X, Y>) {
            this@ReIndexedDataFrame.dataFrameChanged()
        }
    }

    protected fun setRecipe(recipe: ReIndexRecipe<R, C>) {
        if (this.recipe !== recipe) {
            this.recipe = recipe
            cacheRowIndex = null
            cacheColumnIndex = null
            notifyDataFrameChanged(DataFrameEvent(emptyList(), emptyList(), true))
        }
    }

    protected fun dataFrameChanged() {
        cacheRowIndex = null
        cacheColumnIndex = null
        notifyDataFrameChanged(DataFrameEvent(emptyList(), emptyList(), false))
    }

    override fun getRowClass(row: R): KClass<*>? {
        return dataFrame.getRowClass(dataFrame.getRowKey(rowIndex.getAddress(row)))
    }

    override fun getColumnClass(column: C): KClass<*> {
        return dataFrame.getColumnClass(dataFrame.getColumnKey(columnIndex.getAddress(column)))
    }

    override fun getValueAt(row: R, column: C): V {
        val r: Int = rowIndex.getAddress(row)
        val c: Int = columnIndex.getAddress(column)

//        assert r >= 0: row + "," + column;
//        assert c >= 0: row + "," + column;
//
//        assert r < getRowCount(): row + "," + column;
//        assert c < getColumnCount(): row + "," + column;
//
//        assert getRowCount() == dataFrame.getRowCount();
//        return if (r >= 0 && c >= 0) {
            val rowKey = dataFrame.getRowKey(r)
            val columnKey = dataFrame.getColumnKey(c)
//            assert(dataFrame.getColumnIndex().contains(columnKey)) { "Column $columnKey doesn't exists" }
            return dataFrame.getValueAt(rowKey, columnKey)
//        } else {
//            null
//        }
    }

    override fun setValueAt(row: R, column: C, value: V) {
        if (dataFrame is MutableDataFrame) {
            val r: Int = rowIndex.getAddress(row)
            val c: Int = columnIndex.getAddress(column)
            if (r >= 0 && c >= 0) {
                dataFrame.setValueAt(dataFrame.getRowKey(r), dataFrame.getColumnKey(c), value)
            }
        }
    }

    fun getRow(row: R): Series<C, *>? {
        return null
    }

//    override fun getColumn(column: C): Series<R, V> {
//        return ColumnSeries2(column)
//    }

    override fun rows(): Iterable<R> {
        return rowIndex.keys()
    }

    override fun columns(): Iterable<C> {
        return columnIndex.keys()
    }

    override fun getRowKey(index: Int): R {
        return rowIndex.getKey(index)
    }

    override fun getColumnKey(index: Int): C {
        return columnIndex.getKey(index)
    }

    override fun getRowAddress(row: R): Int {
        return rowIndex.getAddress(row)
    }

    override fun getColumnAddress(column: C): Int {
        return columnIndex.getAddress(column)
    }

    override val rowCount: Int
        get() = rowIndex.size
    override val columnCount: Int
        get() = columnIndex.size

//    fun join(series: Series?, columns: Array<C>?): DataFrame {
//        throw IllegalStateException("Not implemented")
//    }
//
//    private inner class ColumnSeries2 private constructor(column: C?) : AbstractSeries<R, V>(),
//        MutableSeries<R, V> {
//        private val column: C
//        private var s: Series<X, V>? = null
//        val name: Any
//            get() = column
//        val type: java.lang.Class?
//            get() = if (s != null) s.getType() else null
//
//        operator fun get(key: R): V? {
//            val r: Int = getRowIndex().getAddress(key)
//            return if (r >= 0) {
//                s.get(dataFrame.getRowKey(r))
//            } else {
//                null
//            }
//        }
//
//        operator fun set(key: R, value: V) {
//            throw UnsupportedOperationException()
//        }
//
//        fun getKey(i: Int): R {
//            return getRowKey(i)
//        }
//
//        fun size(): Int {
//            return rowCount
//        }
//
//        fun getAddress(key: R): Int {
//            return getRowAddress(key)
//        }
//
//        fun keys(): Iterable<R> {
//            return rows()
//        }
//
//        fun <L> reindex(vararg keys: L): Series<L, V>? {
//            return null
//        }
//
//        fun head(count: Int): Series<R, V>? {
//            return null
//        }
//
//        fun tail(count: Int): Series<R, V>? {
//            return null
//        }
//
//        override fun equals(o: Any?): Boolean {
//            if (this === o) return true
//            if (!super.equals(o)) return false
//            val that: ColumnSeries2 = o as ColumnSeries2
//            return column == that.column
//        }
//
//        override fun hashCode(): Int {
//            var result = super.hashCode()
//            result = 31 * result + column.hashCode()
//            return result
//        }
//
//        init {
//            assert(column != null)
//            this.column = column
//            val address: Int = getColumnIndex().getAddress(column)
//            s = if (address >= 0) {
//                dataFrame.getColumn(dataFrame.getColumnKey(address))
//            } else {
//                null
//            }
//        }
//    }

    init {
        dataFrame.addWeakDataFrameListener(dataFrameListener)
    }
}
