package org.molap.dataframe.rowmajor

import org.molap.dataframe.AbstractMutableDataFrame
import org.molap.dataframe.DataFrameEvent
import org.molap.dataframe.rowmajor.RowMajorDataFrame.Query
import org.molap.dataframe.rowmajor.RowMajorDataFrame.QueryBuilder
import org.molap.index.DefaultUniqueIndex
import org.molap.index.UniqueIndex
import kotlin.reflect.KClass

/**
 * Created by luc on 22.03.17.
 */
abstract class AbstractRowMajorDataFrame<R, C, V>(vararg columns: C) :
    AbstractMutableDataFrame<R, C, V>(), RowMajorDataFrame<R, C, V> {
    override val columnIndex: UniqueIndex<C> = DefaultUniqueIndex(*columns)
    override val rowIndex: UniqueIndex<R>
        get() {
            if (lazRowIndex == null) {
                lazRowIndex = DefaultUniqueIndex<R>(rows, true)
            }
            return lazRowIndex!!
        }
    private var lazRowIndex: UniqueIndex<R>? = null
    private val rows: MutableList<R> = ArrayList<R>()
    override fun getRowClass(r: R): KClass<*>? {
        return Any::class
    }

    override fun addRow(row: R) {
        rows.add(row)
        lazRowIndex = null
        notifyDataFrameChanged(DataFrameEvent<R, C>(listOf(row), null, false))
    }

    override fun replaceRow(oldRow: R, newRow: R) {
        rows[rowIndex.getAddress(oldRow)] = newRow
        lazRowIndex = null
        notifyDataFrameChanged(DataFrameEvent<R, C>(listOf(oldRow, newRow), null, false))
    }

    override fun addRow(i: Int, row: R) {
        rows.add(i, row)
        lazRowIndex = null
        notifyDataFrameChanged(DataFrameEvent<R, C>(listOf(row), null, false))
    }

    override fun addBeforeRow(insertionPoint: R, row: R) {
        rows.add(getRowAddress(insertionPoint), row)
        lazRowIndex = null
        notifyDataFrameChanged(DataFrameEvent<R, C>(listOf(row), null, false))
    }

    override fun addAfterRow(insertionPoint: R, row: R) {
        rows.add(getRowAddress(insertionPoint) + 1, row)
        lazRowIndex = null
        notifyDataFrameChanged(DataFrameEvent<R, C>(listOf(row), null, false))
    }

    override fun remove(row: R) {
        rows.removeAt(getRowAddress(row))
        lazRowIndex = null
        notifyDataFrameChanged(DataFrameEvent<R, C>(listOf(row), null, false))
    }

    override fun retrieve(query: Query<R, C, V>): List<R> {
        val result: MutableList<R> = ArrayList<R>()
        for (row in rows()) {
            if (query.matches(row)) {
                result.add(row)
            }
        }
        return result
    }

    override val queryBuilder: QueryBuilder<R, C, V> = DefaultQueryBuilder()

    inner class DefaultQueryBuilder internal constructor() : QueryBuilder<R, C, V> {
        override fun equal(column: C, value: V): Query<R, C, V> {
            return Equal(column, value)
        }

        override fun smallerThan(column: C, value: V): Query<R, C, V> {
            return SmallerThan(column, value, false)
        }

        override fun smallerThanOrEqualTo(column: C, value: V): Query<R, C, V> {
            return SmallerThan(column, value, true)
        }

        override fun greaterThan(column: C, value: V): Query<R, C, V> {
            return GreaterThan(column, value, false)
        }

        override fun greaterThanOrEqualTo(column: C, value: V): Query<R, C, V> {
            return GreaterThan(column, value, true)
        }

        override fun overlap(start1: C, end1: C, start2: V, end2: V): Query<R, C, V> {
            return Overlap(start1, end1, start2, end2)
        }

        override fun and(q1: Query<R, C, V>, q2: Query<R, C, V>): Query<R, C, V>? {
            return q1.and(q2)
        }

        override fun or(q1: Query<R, C, V>, q2: Query<R, C, V>): Query<R, C, V>? {
            return q1.or(q2)
        }
    }

    abstract inner class AbstractQuery : Query<R, C, V> {
        override fun equal(column: C, value: V): Query<R, C, V> {
            return Equal(column, value)
        }

        override fun smallerThan(column: C, value: V): Query<R, C, V> {
            return SmallerThan(column, value, false)
        }

        override fun smallerThanOrEqualTo(column: C, value: V): Query<R, C, V> {
            return SmallerThan(column, value, true)
        }

        override fun greaterThan(column: C, value: V): Query<R, C, V> {
            return GreaterThan(column, value, false)
        }

        override fun greaterThanOrEqualTo(column: C, value: V): Query<R, C, V> {
            return GreaterThan(column, value, true)
        }

        override fun and(query: Query<R, C, V>): Query<R, C, V> {
            return And(this, query)
        }

        override fun or(query: Query<R, C, V>): Query<R, C, V> {
            return Or(this, query)
        }
    }

    inner class And(vararg queries: Query<R, C, V>) : AbstractQuery() {
        private val queries: Array<Query<R, C, V>>
        override fun matches(row: R): Boolean {
            for (query in queries) {
                if (!query.matches(row)) {
                    return false
                }
            }
            return true
        }

        init {
            this.queries = queries.toList().toTypedArray()
        }
    }

    inner class Or(vararg queries: Query<R, C, V>) : AbstractQuery() {
        private val queries: Array<Query<R, C, V>>
        override fun matches(row: R): Boolean {
            for (query in queries) {
                if (query.matches(row)) {
                    return true
                }
            }
            return false
        }

        init {
            this.queries = queries.toList().toTypedArray()
        }
    }

    inner class Equal(private val column: C, value: V) : AbstractQuery() {
        private val value: V?
        override fun matches(row: R): Boolean {
            return if (value == null) {
                getValueAt(row, column) == null
            } else {
                value == getValueAt(row, column)
            }
        }

        init {
            this.value = value
        }
    }

    inner class SmallerThan(private val column: C, value: V, valueInclusive: Boolean) :
        AbstractQuery() {
        private val value: V?
        private val valueInclusive: Boolean
        override fun matches(row: R): Boolean {
            return if (value == null || value !is Comparable<*>) {
                false
            } else {
                val compareTo = (value as Comparable<V>).compareTo(getValueAt(row, column))
                if (valueInclusive) {
                    compareTo <= 0
                } else {
                    compareTo < 0
                }
            }
        }

        init {
            this.value = value
            this.valueInclusive = valueInclusive
        }
    }

    inner class GreaterThan(private val column: C, value: V, valueInclusive: Boolean) :
        AbstractQuery() {
        private val value: V?
        private val valueInclusive: Boolean
        override fun matches(row: R): Boolean {
            return if (value == null || value !is Comparable<*>) {
                false
            } else {
                val compareTo = (value as Comparable<V>).compareTo(getValueAt(row, column))
                if (valueInclusive) {
                    compareTo >= 0
                } else {
                    compareTo > 0
                }
            }
        }

        init {
            this.value = value
            this.valueInclusive = valueInclusive
        }
    }

    inner class Overlap(private val start1: C, private val end1: C, start2: V, end2: V) : AbstractQuery() {
        private val start2: V?
        private val end2: V?
        override fun matches(row: R): Boolean {
            return if (start2 == null || start2 !is Comparable<*> || end2 == null || end2 !is Comparable<*>) {
                false
            } else {
//                if (start2 is java.util.Date && end2 is java.util.Date) {
//                    val s1: java.util.Date = getValueAt(row, start1) as java.util.Date
//                    val e1: java.util.Date = getValueAt(row, end1) as java.util.Date
//                    val s2: java.util.Date = start2 as java.util.Date
//                    val e2: java.util.Date = end2 as java.util.Date
//                    s1.before(e2) && s2.before(e1)
//                } else {
                    val s1 = getValueAt(row, start1) as Comparable<V>
                    val e1 = getValueAt(row, end1) as V
                    val s2 = start2 as Comparable<V>
                    val e2 = end2 as V
                    s1.compareTo(e2) < 0 && s2.compareTo(e1) < 0
//                }
            }
        }

        init {
            this.start2 = start2
            this.end2 = end2
        }
    }
}
