/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.index

import org.molap.index.UniqueIndex.Duplicate.UseFirst

class DefaultUniqueIndex<K> : AbstractUniqueIndex<K> {
    private val lazy: Lazy?

    /**
     * Creates a new index with a set of keys.
     * Note: the keys do not need to be unique
     *
     * @param keys the keys
     */
    constructor(keys: Iterable<K>, includeNull: Boolean) {
        val index: MutableMap<K,Int> = HashMap()
        val list: MutableList<K> = ArrayList()
        var i = 0
        for (key in keys) {
            if (includeNull || key != null) {
                if (!index.containsKey(key)) {
                    index.put(key,i)
                    list.add(key)
                    i++
                }
            }
        }
        lazy = object : Lazy() {
            override fun createIndex(): MutableMap<K,Int> {
                return index
            }

            override fun createList(): MutableList<K> {
                return list
            }
        }
//        assert(size >= 0)
    }

    /**
     * Creates a new index with a set of keys.
     * Note: the keys do not need to be unique
     *
     * @param keys the keys
     */
    constructor(keys: MutableList<K>) {
        lazy = object : Lazy() {
            override fun createIndex(): MutableMap<K,Int> {
                val index: MutableMap<K,Int> = HashMap(keys.size)
                var i = 0
                for (key in keys) {
                    if (!index.containsKey(key)) {
                        index.put(key, i)
                        i++
                    }
                }
                return index
            }

            override fun createList(): MutableList<K> {
                return keys
            }
        }
//        assert(size >= 0)
    }

    constructor(vararg keys: K) : this(UniqueIndex.Duplicate.UseFirstWarn, *keys) {}

    /**
     * Creates a new index with an ordered list of unique keys.
     *
     * @param keys the unique keys
     */
    constructor(duplicate: UniqueIndex.Duplicate, vararg keys: K) {
        val index: MutableMap<K,Int> = HashMap(keys?.size ?: 0)
        val list: MutableList<K> = ArrayList<K>()
        for (i in 0 until keys.size) {
            val key = keys[i]
            if (!index.containsKey(key)) {
                index.put(key, i)
                list.add(key)
            } else {
                when (duplicate) {
                    UseFirst -> {
                    }
                    UniqueIndex.Duplicate.UseFirstWarn -> //                        list.add(key);
                        try {
                            throw IllegalStateException("Index not unique for value $key (address=$i)")
                        } catch (e: IllegalStateException) {
//                            e.printStackTrace()
                        }
                    UniqueIndex.Duplicate.UseLast -> {
                        index.put(key, i)
                        list.remove(key)
                        list.add(key)
                        println("Index not unique for value $key (address=$i): will use last")
                    }
                    UniqueIndex.Duplicate.RemoveFirst -> {
                        index.put(key, i)
                        list.remove(key)
                        list.add(key)
                        println("Index not unique for value $key (address=$i): will remove first")
                    }
                    UniqueIndex.Duplicate.RemoveLast -> {
                    }
                    UniqueIndex.Duplicate.ThrowException -> throw IllegalStateException("Index not unique for value $key (address=$i)")
                }
            }
        }
        lazy = object : Lazy() {
            override fun createIndex(): MutableMap<K,Int> {
                return index
            }

            override fun createList(): MutableList<K> {
                return list
            }
        }
//        assert(size >= 0)
    }

    constructor(index: UniqueIndex<K>, comparator: Comparator<K>) {
        lazy = object : Lazy() {
            override fun createIndex(): MutableMap<K,Int> {
                val thisindex = HashMap<K,Int>(index.size)
                for (key in index!!.keys()!!) {
                    thisindex.put(key, -1)
                }
                return thisindex
            }

            override fun populateIndex(index: MutableMap<K,Int>) {
                val list = list
                for (i in list!!.indices) {
                    val key = list[i]
                    index.put(key, i)
                }
            }

            override fun createList(): MutableList<K> {
                val list: MutableList<K> = ArrayList<K>()
                for (key in index.keys()!!) {
                    list.add(key)
                }
                list.sortWith(comparator)
                return list
            }
        }
//        assert(size >= 0)
    }

    private constructor(index: MutableMap<K,Int>, list: MutableList<K>) {
        val thisindex: MutableMap<K,Int> = index
        lazy = object : Lazy() {
            override fun createIndex(): MutableMap<K,Int> {
                return thisindex
            }

            override fun createList(): MutableList<K> {
                return list
            }
        }
//        assert(size >= 0)
    }

    override operator fun contains(key: K): Boolean {
        return lazy!!.index!!.containsKey(key)
    }

    override fun getAddress(key: K): Int {
        return lazy!!.getAddress(key)
    }

    override fun keys(): Iterable<K> {
        return lazy!!.list
    }

    override fun getKey(i: Int): K {
        return if (i >= 0 && i < size) {
            lazy!!.list!![i]
        } else {
            throw RuntimeException("Key is out of range")
        }
    }

    override val size: Int
        get() = lazy!!.size

    override fun head(count: Int): UniqueIndex<K> {
        val index: MutableMap<K,Int> = HashMap(count)
        val list: MutableList<K> = ArrayList<K>()
        for (i in 0 until count) {
            val key = lazy!!.list!![i]
            val loc = getAddress(key)
            index.put(key, loc)
            list.add(key)
        }
        return DefaultUniqueIndex<K>(index, list)
    }

    override fun tail(count: Int): UniqueIndex<K> {
        val index: MutableMap<K,Int> = HashMap(count)
        val list: MutableList<K> = ArrayList<K>()
        for (i in 0 until count) {
            val key = lazy!!.list!![lazy.list!!.size - 1 - i]
            val loc = getAddress(key)
            index.put(key, loc)
            list.add(key)
        }
        return DefaultUniqueIndex<K>(index, list)
    }

    fun keepAddresses(addresses: IntArray): UniqueIndex<K> {
        val index: MutableMap<K,Int> = HashMap(addresses!!.size)
        val list: MutableList<K> = ArrayList<K>()
        for (i in addresses.indices) {
            val address = addresses[i]
            val key = lazy!!.list!![address]
            index.put(key, i)
            list.add(key)
        }
        return DefaultUniqueIndex<K>(index, list)
    }

//    fun remove(vararg keys: K): DefaultUniqueIndex<K> {
//        val removed: MutableList<K> = ArrayList<K>(lazy!!.list)
//        removed.removeAll(keys.asList())
//        val index: MutableMap<K,Int> = lazy.index.copy(keys)
//        return DefaultUniqueIndex<K>(index, removed)
//    }

    private abstract inner class Lazy {
        val indexDelegate = lazy { createIndex() }
        val index: MutableMap<K,Int> by indexDelegate
        val listDelegate = lazy { createList() }
        val list: MutableList<K> by listDelegate
        private var populated = false

        fun getAddress(key: K): Int {
            val index: MutableMap<K,Int> = index
            if (!populated) {
                populateIndex(index)
                populated = true
            }
            val i = index.get(key)
            if(i != null) {
                return i
            } else {
                throw RuntimeException("Illegal key")
            }
        }

        val size: Int
            get() = list.size

        protected abstract fun createIndex(): MutableMap<K,Int>
        protected abstract fun createList(): MutableList<K>
        protected open fun populateIndex(index: MutableMap<K,Int>) {}
    }

    companion object {
        fun <K> fromArray(keys: Array<K>): DefaultUniqueIndex<K>? {
            return DefaultUniqueIndex(UniqueIndex.Duplicate.UseFirstWarn, *keys)
        }
    }
}