package org.molap.aggregates.cube

import org.molap.series.Series
import kotlin.math.ceil
import kotlin.math.floor
import kotlin.math.round
import kotlin.math.sqrt

class UnivariateStatistics : Comparable<UnivariateStatistics?> {
    private enum class Statistics {
        Min, Max, Mean, Median
    }

    private val statistics: Map<Statistics, Double>? = null
    private val values: MutableList<Number>
    var sum: Double? = null
        private set
    var uncorrectedSumOfSquares: Double? = null
        private set
    var count: Long = 0
        private set

    constructor(column: Series<Any,Any>) {
        values = ArrayList<Number>()
        var sum: Double? = null
        var uncorrectedSumOfSquares: Double? = null
        for (row in column.keys()!!) {
            val value: Any = column.get(row)
            if (value != null && value is Number) {
                val number = value as Number
                val v: Double = number.toDouble()
                if (!v.isInfinite() && !v.isNaN()) {
                    values.add(number)
                    if (sum != null) {
                        sum += number.toDouble()
                    } else {
                        sum = number.toDouble()
                    }
                    if (uncorrectedSumOfSquares != null) {
                        uncorrectedSumOfSquares += v * v
                    } else {
                        uncorrectedSumOfSquares = v * v
                    }
                    count++
                }
            }
        }
        this.sum = sum
        this.uncorrectedSumOfSquares = uncorrectedSumOfSquares
        values.sortWith(numberComparator)
    }

    constructor(rows: Iterable<Any>, column: Series<Any, *>) {
        values = ArrayList<Number>()
        var sum: Double? = null
        var uncorrectedSumOfSquares: Double? = null
        for (row in rows) {
            val value: Any? = column.get(row)
            if (value != null && value is Number) {
                val number = value
                val v: Double = number.toDouble()
                if (!v.isInfinite() && !v.isNaN()) {
                    values.add(number)
                    if (sum != null) {
                        sum += number.toDouble()
                    } else {
                        sum = number.toDouble()
                    }
                    if (uncorrectedSumOfSquares != null) {
                        uncorrectedSumOfSquares += v * v
                    } else {
                        uncorrectedSumOfSquares = v * v
                    }
                    count++
                }
            }
        }
        this.sum = sum
        this.uncorrectedSumOfSquares = uncorrectedSumOfSquares
        values.sortWith(numberComparator)
    }

    val minimum: Number?
        get() = if (values.size > 0) values[0] else null

    val maximum: Number?
        get() = if (values.size > 0) values[values.size - 1] else null

    val mean: Double?
        get() = if (count > 0) sum!! / count else null

    val variance: Double?
        get() {
            val mean = mean
            return if (mean != null) {
                uncorrectedSumOfSquares!! / count - mean * mean
            } else {
                null
            }
        }

    val stdDev: Double?
        get() {
            val variance = variance
            return if (variance != null) {
                sqrt(variance)
            } else {
                null
            }
        }

    fun getSigma(sigma: Double): Any? {
        val mean = mean
        return if (mean != null) {
            val stdDev = stdDev
            if (stdDev != null) {
                mean + sigma * stdDev
            } else {
                null
            }
        } else {
            null
        }
    }

    //        if(values.size() > 0) {
//            int middle = values.size() / 2;
//            if (values.size() %2 == 1) {
//                return values.get(middle);
//            } else {
//                return (values.get(middle-1).doubleValue() + values.get(middle).doubleValue()) / 2.0;
//            }
//        } else {
//            return null;
//        }
    val median: Number?
        get() = getPercentile(50.0)

    //        if(values.size() > 0) {
    //            int middle = values.size() / 2;
    //            if (values.size() %2 == 1) {
    //                return values.get(middle);
    //            } else {
    //                return (values.get(middle-1).doubleValue() + values.get(middle).doubleValue()) / 2.0;
    //            }
    //        } else {
    //            return null;
    //        }

    /*
     * Returns an estimate of the <code>p</code>th percentile
     */
    fun getPercentile(type: EstimationType, p: Double): Double? {
        val n = values.size
        return if (n > 0) {
            val arr = DoubleArray(n)
            for (i in values.indices) {
                val number = values[i]
                arr[i] = number.toDouble()
            }
            type.evaluate(arr, p, KthSelector())
        } else {
            null
        }
    }

    fun toPercentile(value: Double): Double? {
        val n = values.size
        return if (n > 1) {
            var percentileValue = getPercentile(0.0)
            var lowerPercentile = 0
            for (p in 0..100) {
                percentileValue = getPercentile(p.toDouble())
                if (value <= percentileValue!!) {
                    lowerPercentile = p
                    break
                }
            }
            var upperPercentile = lowerPercentile
            for (p in lowerPercentile + 1..100) {
                upperPercentile = if (getPercentile(p.toDouble())!!.toDouble() == percentileValue!!.toDouble()) {
                    p
                } else {
                    break
                }
            }
            val mid = (lowerPercentile + upperPercentile) / 2.0
            if (mid > 50) {
                ceil(mid)
            } else {
                floor(mid)
            }
            //return 100D;
        } else if (n > 0) {
            50.0
        } else {
            null
        }
    }

    /*
     * Returns an estimate of the <code>p</code>th percentile
     */
    fun getPercentile(p: Double): Double? {
        return getPercentile(EstimationType.R_7, p)
        //        double n = values.size();
//        if(n > 0) {
//            double pos = p * (n + 1) / 100;
//            double fpos = Math.floor(pos);
//            int intPos = (int) fpos;
//            double dif = pos - fpos;
//
//            if (pos < 1) {
//                return values.get(0).doubleValue();
//            }
//            if (pos >= n) {
//                return values.get(values.size() - 1).doubleValue();
//            }
//            double lower = values.get(intPos - 1).doubleValue();
//            double upper = values.get(intPos).doubleValue();
//            return lower + dif * (upper - lower);
//        } else {
//            return null;
//        }
    }

    override operator fun compareTo(o: UnivariateStatistics?): Int {
        return if (o != null && mean != null) {
            val oMean = o.mean
            if (oMean != null) {
                mean!!.compareTo(oMean)
            } else {
                1
            }
        } else {
            1
        }
    }

    override fun toString(): String {
        return "UnivariateStatistics{" +
                "count=" + count +
                ", min=" + minimum +
                ", median=" + median +
                ", mean=" + mean +
                ", max=" + maximum +
                '}'
    }

    /**
     * An enum for various estimation strategies of a percentile referred in
     * [wikipedia on quantile](http://en.wikipedia.org/wiki/Quantile)
     * with the names of enum matching those of types mentioned in
     * wikipedia.
     *
     *
     * Each enum corresponding to the specific type of estimation in wikipedia
     * implements  the respective formulae that specializes in the below aspects
     *
     *  * An **index method** to calculate approximate index of the
     * estimate
     *  * An **estimate method** to estimate a value found at the earlier
     * computed index
     *  * A ** minLimit** on the quantile for which first element of sorted
     * input is returned as an estimate
     *  * A ** maxLimit** on the quantile for which last element of sorted
     * input is returned as an estimate
     *
     *
     *
     * References:
     *
     *  1.
     * [Wikipedia on quantile](http://en.wikipedia.org/wiki/Quantile)
     *
     *  1.
     * [
 * Hyndman, R. J. and Fan, Y. (1996) Sample quantiles in statistical
 * packages, American Statistician 50, 361–365](https://www.amherst.edu/media/view/129116/.../Sample+Quantiles.pdf)
     *  1.
     * [
 * R-Manual ](http://stat.ethz.ch/R-manual/R-devel/library/stats/html/quantile.html)
     *
     */
    enum class EstimationType
    /**
     * Constructor
     *
     * @param type name of estimation type as per wikipedia
     */(
        /** Simple name such as R-1, R-2 corresponding to those in wikipedia.  */
        val longName: String,
        /**
         * Gets the name of the enum
         *
         * @return the name
         */
        private val func: EstimationFunction
    ) {
        /**
         * This is the default type used in the percentile estimation.This method
         * has the following formulae for index and estimates<br></br>
         * \( \begin{align}
         * &amp;index    = (N+1)p\ \\
         * &amp;estimate = x_{\lceil h\,-\,1/2 \rceil} \\
         * &amp;minLimit = 0 \\
         * &amp;maxLimit = 1 \\
         * \end{align}\)
         */
        LEGACY("Legacy Apache Commons Math", LegacyEstimationFunction()),

        /**
         * The method R_1 has the following formulae for index and estimates<br></br>
         * \( \begin{align}
         * &amp;index= Np + 1/2\,  \\
         * &amp;estimate= x_{\lceil h\,-\,1/2 \rceil} \\
         * &amp;minLimit = 0 \\
         * \end{align}\)
         */
        R_1("R-1", R1EstimationFunction()),

        /**
         * The method R_2 has the following formulae for index and estimates<br></br>
         * \( \begin{align}
         * &amp;index= Np + 1/2\, \\
         * &amp;estimate=\frac{x_{\lceil h\,-\,1/2 \rceil} +
         * x_{\lfloor h\,+\,1/2 \rfloor}}{2} \\
         * &amp;minLimit = 0 \\
         * &amp;maxLimit = 1 \\
         * \end{align}\)
         */
        R_2("R-2", R2EstimationFunction()),

        /**
         * The method R_3 has the following formulae for index and estimates<br></br>
         * \( \begin{align}
         * &amp;index= Np \\
         * &amp;estimate= x_{\lfloor h \rceil}\, \\
         * &amp;minLimit = 0.5/N \\
         * \end{align}\)
         */
        R_3("R-3", R3EstimationFunction()),

        /**
         * The method R_4 has the following formulae for index and estimates<br></br>
         * \( \begin{align}
         * &amp;index= Np\, \\
         * &amp;estimate= x_{\lfloor h \rfloor} + (h -
         * \lfloor h \rfloor) (x_{\lfloor h \rfloor + 1} - x_{\lfloor h
         * \rfloor}) \\
         * &amp;minLimit = 1/N \\
         * &amp;maxLimit = 1 \\
         * \end{align}\)
         */
        R_4("R-4", R4EstimationFunction()),

        /**
         * The method R_5 has the following formulae for index and estimates<br></br>
         * \( \begin{align}
         * &amp;index= Np + 1/2\\
         * &amp;estimate= x_{\lfloor h \rfloor} + (h -
         * \lfloor h \rfloor) (x_{\lfloor h \rfloor + 1} - x_{\lfloor h
         * \rfloor}) \\
         * &amp;minLimit = 0.5/N \\
         * &amp;maxLimit = (N-0.5)/N
         * \end{align}\)
         */
        R_5("R-5", R5EstimationFunction()),

        /**
         * The method R_6 has the following formulae for index and estimates<br></br>
         * \( \begin{align}
         * &amp;index= (N + 1)p \\
         * &amp;estimate= x_{\lfloor h \rfloor} + (h -
         * \lfloor h \rfloor) (x_{\lfloor h \rfloor + 1} - x_{\lfloor h
         * \rfloor}) \\
         * &amp;minLimit = 1/(N+1) \\
         * &amp;maxLimit = N/(N+1) \\
         * \end{align}\)
         *
         *
         * **Note:** This method computes the index in a manner very close to
         * the default Commons Math Percentile existing implementation. However
         * the difference to be noted is in picking up the limits with which
         * first element (p&lt;1(N+1)) and last elements (p&gt;N/(N+1))are done.
         * While in default case; these are done with p=0 and p=1 respectively.
         */
        R_6("R-6", R6EstimationFunction()),

        /**
         * The method R_7 implements Microsoft Excel style computation has the
         * following formulae for index and estimates.<br></br>
         * \( \begin{align}
         * &amp;index = (N-1)p + 1 \\
         * &amp;estimate = x_{\lfloor h \rfloor} + (h -
         * \lfloor h \rfloor) (x_{\lfloor h \rfloor + 1} - x_{\lfloor h
         * \rfloor}) \\
         * &amp;minLimit = 0 \\
         * &amp;maxLimit = 1 \\
         * \end{align}\)
         */
        R_7("R-7", R7EstimationFunction()),

        /**
         * The method R_8 has the following formulae for index and estimates<br></br>
         * \( \begin{align}
         * &amp;index = (N + 1/3)p + 1/3  \\
         * &amp;estimate = x_{\lfloor h \rfloor} + (h -
         * \lfloor h \rfloor) (x_{\lfloor h \rfloor + 1} - x_{\lfloor h
         * \rfloor}) \\
         * &amp;minLimit = (2/3)/(N+1/3) \\
         * &amp;maxLimit = (N-1/3)/(N+1/3) \\
         * \end{align}\)
         *
         *
         * As per Ref [2,3] this approach is most recommended as it provides
         * an approximate median-unbiased estimate regardless of distribution.
         */
        R_8("R-8", R8EstimationFunction()),

        /**
         * The method R_9 has the following formulae for index and estimates<br></br>
         * \( \begin{align}
         * &amp;index = (N + 1/4)p + 3/8\\
         * &amp;estimate = x_{\lfloor h \rfloor} + (h -
         * \lfloor h \rfloor) (x_{\lfloor h \rfloor + 1} - x_{\lfloor h
         * \rfloor}) \\
         * &amp;minLimit = (5/8)/(N+1/4) \\
         * &amp;maxLimit = (N-3/8)/(N+1/4) \\
         * \end{align}\)
         */
        R_9("R-9", R9EstimationFunction());

        /**
         * Finds the index of array that can be used as starting index to
         * [estimate][.estimate]
         * percentile. The calculation of index calculation is specific to each
         * [EstimationType].
         *
         * @param p the p<sup>th</sup> quantile
         * @param length the total number of array elements in the work array
         * @return a computed real valued index as explained in the wikipedia
         */
        protected fun index(p: Double, length: Int): Double {
            return func.index(p, length)
        }

        /**
         * Estimation based on K<sup>th</sup> selection. This may be overridden
         * in specific enums to compute slightly different estimations.
         *
         * @param work array of numbers to be used for finding the percentile
         * @param pos indicated positional index prior computed from calling
         * [.index]
         * @param pivotsHeap an earlier populated cache if exists; will be used
         * @param length size of array considered
         * @param selector a [KthSelector] used for pivoting during search
         * @return estimated percentile
         */
        protected fun estimate(
            work: DoubleArray, pivotsHeap: IntArray?,
            pos: Double, length: Int,
            selector: KthSelector
        ): Double {
            return func.estimate(work, pivotsHeap, pos, length, selector)
        }

        /**
         * Evaluate method to compute the percentile for a given bounded array
         * using earlier computed pivots heap.<br></br>
         * This basically calls the [index][.index] and then
         * [estimate][.estimate]
         * functions to return the estimated percentile value.
         *
         * @param work array of numbers to be used for finding the percentile
         * @param pivotsHeap a prior cached heap which can speed up estimation
         * @param p the p<sup>th</sup> quantile to be computed
         * @param selector a [KthSelector] used for pivoting during search
         * @return estimated percentile
         * @throws OutOfRangeException if p is out of range
         * @throws NullArgumentException if work array is null
         */
        protected fun evaluate(
            work: DoubleArray, pivotsHeap: IntArray?, p: Double,
            selector: KthSelector
        ): Double {
            require(!(p > 100 || p < 0)) { "OUT_OF_BOUNDS_QUANTILE_VALUE" }
            return estimate(work, pivotsHeap, index(p / 100.0, work.size), work.size, selector)
        }

        /**
         * Evaluate method to compute the percentile for a given bounded array.
         * This basically calls the [index][.index] and then
         * [estimate][.estimate]
         * functions to return the estimated percentile value. Please
         * note that this method does not make use of cached pivots.
         *
         * @param work array of numbers to be used for finding the percentile
         * @param p the p<sup>th</sup> quantile to be computed
         * @return estimated percentile
         * @param selector a [KthSelector] used for pivoting during search
         * @throws OutOfRangeException if length or p is out of range
         * @throws NullArgumentException if work array is null
         */
        fun evaluate(work: DoubleArray, p: Double, selector: KthSelector): Double {
            return this.evaluate(work, null, p, selector)
        }

        private class LegacyEstimationFunction : AbstractEstimationFunction() {
            /**
             * {@inheritDoc}.This method in particular makes use of existing
             * Apache Commons Math style of picking up the index.
             */
            override fun index(p: Double, length: Int): Double {
                val minLimit = 0.0
                val maxLimit = 1.0
                return if (p.compareTo(minLimit) == 0) 0.0 else if (p.compareTo(maxLimit) == 0) length.toDouble() else p * (length + 1.0)
            }
        }

        private class R1EstimationFunction : AbstractEstimationFunction() {
            override fun index(p: Double, length: Int): Double {
                val minLimit = 0.0
                return if (p.compareTo(minLimit) == 0) 0.0 else length * p + 0.5
            }

            /**
             * {@inheritDoc}This method in particular for R_1 uses ceil(pos-0.5)
             */
            override fun estimate(
                values: DoubleArray,
                pivotsHeap: IntArray?, pos: Double,
                length: Int, selector: KthSelector
            ): Double {
                return super.estimate(values, pivotsHeap, ceil(pos - 0.5), length, selector)
            }
        }

        private class R2EstimationFunction : AbstractEstimationFunction() {
            override fun index(p: Double, length: Int): Double {
                val minLimit = 0.0
                val maxLimit = 1.0
                return (if (p.compareTo(maxLimit) == 0) length.toDouble() else if (p.compareTo(
                        minLimit
                    ) == 0
                ) 0.0 else length * p + 0.5)
            }

            /**
             * {@inheritDoc}This method in particular for R_2 averages the
             * values at ceil(p+0.5) and floor(p-0.5).
             */
            override fun estimate(
                values: DoubleArray,
                pivotsHeap: IntArray?, pos: Double,
                length: Int, selector: KthSelector
            ): Double {
                val low =
                    super.estimate(values, pivotsHeap, ceil(pos - 0.5), length, selector)
                val high =
                    super.estimate(values, pivotsHeap, floor(pos + 0.5), length, selector)
                return (low + high) / 2
            }
        }

        private class R3EstimationFunction : AbstractEstimationFunction() {
            override fun index(p: Double, length: Int): Double {
                val minLimit = 1.0 / 2 / length
                return (if (p.compareTo(
                        minLimit
                    ) <= 0
                ) 0.0 else round(length * p)).toDouble() // java.lang.Math.rint
            }
        }

        private class R4EstimationFunction : AbstractEstimationFunction() {
            override fun index(p: Double, length: Int): Double {
                val minLimit = 1.0 / length
                val maxLimit = 1.0
                return if (p.compareTo(minLimit) < 0) 0.0 else if (p.compareTo(
                        maxLimit
                    ) == 0
                ) length.toDouble() else length * p
            }
        }

        private class R5EstimationFunction : AbstractEstimationFunction() {
            override fun index(p: Double, length: Int): Double {
                val minLimit = 1.0 / 2 / length
                val maxLimit = (length - 0.5) / length
                return if (p.compareTo(minLimit) < 0) 0.0 else if (p.compareTo(
                        maxLimit
                    ) >= 0
                ) length.toDouble() else length * p + 0.5
            }
        }

        private class R6EstimationFunction : AbstractEstimationFunction() {
            override fun index(p: Double, length: Int): Double {
                val minLimit = 1.0 / (length + 1)
                val maxLimit = 1.0 * length / (length + 1)
                return if (p.compareTo(minLimit) < 0) 0.0 else if (p.compareTo(
                        maxLimit
                    ) >= 0
                ) length.toDouble() else (length + 1) * p
            }
        }

        private class R7EstimationFunction : AbstractEstimationFunction() {
            override fun index(p: Double, length: Int): Double {
                val minLimit = 0.0
                val maxLimit = 1.0
                return if (p.compareTo(minLimit) == 0) 0.0 else if (p.compareTo(
                        maxLimit
                    ) == 0
                ) length.toDouble() else 1 + (length - 1) * p
            }
        }

        private class R8EstimationFunction : AbstractEstimationFunction() {
            override fun index(p: Double, length: Int): Double {
                val minLimit = 2 * (1.0 / 3) / (length + 1.0 / 3)
                val maxLimit = (length - 1.0 / 3) / (length + 1.0 / 3)
                return if (p.compareTo(minLimit) < 0) 0.0 else if (p.compareTo(
                        maxLimit
                    ) >= 0
                ) length.toDouble() else (length + 1.0 / 3) * p + 1.0 / 3
            }
        }

        private class R9EstimationFunction : AbstractEstimationFunction() {
            override fun index(p: Double, length: Int): Double {
                val minLimit = 5.0 / 8 / (length + 0.25)
                val maxLimit = (length - 3.0 / 8) / (length + 0.25)
                return if (p.compareTo(minLimit) < 0) 0.0 else if (p.compareTo(
                        maxLimit.toDouble()
                    ) >= 0
                ) length.toDouble() else (length + 0.25) * p + 3.0 / 8
            }
        }

    }

    interface EstimationFunction {
        /**
         * Finds the index of array that can be used as starting index to
         * [estimate][.estimate]
         * percentile. The calculation of index calculation is specific to each
         * [EstimationType].
         *
         * @param p the p<sup>th</sup> quantile
         * @param length the total number of array elements in the work array
         * @return a computed real valued index as explained in the wikipedia
         */
        fun index(p: Double, length: Int): Double

        /**
         * Estimation based on K<sup>th</sup> selection. This may be overridden
         * in specific enums to compute slightly different estimations.
         *
         * @param work array of numbers to be used for finding the percentile
         * @param pos indicated positional index prior computed from calling
         * [.index]
         * @param pivotsHeap an earlier populated cache if exists; will be used
         * @param length size of array considered
         * @param selector a [KthSelector] used for pivoting during search
         * @return estimated percentile
         */
        fun estimate(
            work: DoubleArray, pivotsHeap: IntArray?,
            pos: Double, length: Int,
            selector: KthSelector
        ): Double
    }

    abstract class AbstractEstimationFunction : EstimationFunction {
        override fun estimate(
            work: DoubleArray,
            pivotsHeap: IntArray?,
            pos: Double,
            length: Int,
            selector: KthSelector
        ): Double {
            val fpos: Double = floor(pos)
            val intPos = fpos.toInt()
            val dif = pos - fpos
            if (pos < 1) {
                return selector.select(work, pivotsHeap, 0)
            }
            if (pos >= length) {
                return selector.select(work, pivotsHeap, length - 1)
            }
            val lower: Double = selector.select(work, pivotsHeap, intPos - 1)
            val upper: Double = selector.select(work, pivotsHeap, intPos)
            return lower + dif * (upper - lower)
        }
    }

    private class NumberComparator : Comparator<Number> {
        override fun compare(o1: Number, o2: Number): Int {
            return o1.toDouble().compareTo(o2.toDouble())
        }
    }

    companion object {
        private val numberComparator: Comparator<Number> =
            object : Comparator<Number> {
                override fun compare(o1: Number, o2: Number): Int {
                    return o1.toDouble().compareTo(o2.toDouble())
                }
            }
    }
}
