package org.molap.exporter

import org.molap.dataframe.DataFrame
import kotlin.reflect.KClass

object DataFrameExporter {
    fun <R,C,V> exportCSV(dataFrame: DataFrame<R,C,V>): String {
        return exportDelimited(dataFrame, ",")
    }

    internal fun <R,C,V> exportDelimited(dataFrame: DataFrame<R,C,V>, delimiter: String): String {
        val newLine = "\n"
        val sb = StringBuilder("\ufeff")
        val writer: DataFrameWriter = DelimitedDataFrameWriter(sb, dataFrame.columnCount, delimiter, newLine)
        export(dataFrame, writer)
        return sb.toString()
    }

    internal fun <R,C,V> export(dataFrame: DataFrame<R,C,V>, writer: DataFrameWriter) {
        writer.start()
        for (column in 0 until dataFrame.columnCount) {
            val value: String? = dataFrame.getColumnName(dataFrame.getColumnKey(column))
            writer.writeColumnName(value, column < dataFrame.columnCount - 1)
        }
        writer.nextRow()
        if (writer.includeType()) {
            for (column in 0 until dataFrame.columnCount) {
                val value: KClass<*> = dataFrame.getColumnClass(dataFrame.getColumnKey(column))
                writer.writeType(value, column < dataFrame.columnCount - 1)
            }
            writer.nextRow()
        }
        for (row in 0 until dataFrame.rowCount) {
            for (column in 0 until dataFrame.columnCount) {
                try {
                    var value: V
                    val rowKey: R = dataFrame.getRowKey(row)
                    val columnKey: C = dataFrame.getColumnKey(column)
                    value = dataFrame.getValueAt(rowKey, columnKey)
                    writer.writeCell(value, dataFrame, rowKey, columnKey)
                } catch (e: Exception) {
                    e.printStackTrace()
                } finally {
                    writer.nextColumn(column < dataFrame.columnCount - 1)
                }
            }
            writer.nextRow()
        }
        writer.close()
    }
}
