package org.molap.dataframe

import com.macrofocus.common.json.Json.instance
import com.macrofocus.common.json.JsonArray
import com.macrofocus.common.json.JsonObject
import com.macrofocus.common.json.JsonType
import com.macrofocus.common.json.JsonValue
import org.molap.index.DefaultUniqueIndex
import org.molap.index.IntegerRangeUniqueIndex
import org.molap.index.UniqueIndex
import org.molap.series.Series
import java.util.*
import java.util.logging.Logger
import kotlin.reflect.KClass

class JsonDataFrame(private val array: JsonArray) : AbstractDataFrame<Int, String, Any?>() {
    private val labels: Array<String?>
    private val classes: Array<KClass<*>?>

    init {
        val columns = LinkedHashSet<String>()
        val types = HashMap<String, KClass<*>>()
        for (row in 0 until array.length()) {
            val o = array.get<JsonObject>(row)
            for (column in o.keys()) {
                if (!columns.contains(column)) {
                    columns.add(column)
                    val value = o.get<JsonValue>(column)
                    val type = getClass(value)
                    if (type != null) {
                        types[column] = type
                    }
                } else {
                    val value = o.get<JsonValue>(column)
                    val type = getClass(value)
                    if (type != null) {
                        val currentType = types[column]
                        if (currentType == null) {
                            types[column] = type
                        } else if (currentType != type) {
                            types[column] = Any::class
                        }
                    }
                }
            }
        }
        labels = arrayOfNulls(columns.size)
        classes = arrayOfNulls(columns.size)
        var index = 0
        for (column in columns) {
            labels[index] = column
            val type = types[column]
            classes[index] = type ?: Any::class
            index++
        }

//        for (String column : columns) {
//            getLogger().log(Level.INFO, column + ": " + getColumnClass(column));
//        }
    }

    override val rowIndex: UniqueIndex<Int> by lazy { IntegerRangeUniqueIndex(0, array.length() - 1) }
    override val columnIndex: UniqueIndex<String> by lazy {
        val names = ArrayList<String>(labels.size)
        for (c in labels.indices) {
            names.add(labels[c]!!)
        }
        DefaultUniqueIndex<String>(names)
    }

    constructor(json: String) : this((instance().parse(json) as JsonObject).get<JsonValue>("data") as JsonArray) {}

    override fun getColumnName(column: String): String? {
        return labels[columnIndex.getAddress(column)]
    }

    override fun getColumnClass(column: String): KClass<*> {
        return classes[columnIndex.getAddress(column)]!!
    }

    override val rowCount: Int
        get() = array.length()

    override fun getRowClass(row: Int): KClass<*>? {
        return Any::class
    }

    override fun getValueAt(row: Int, column: String): Any? {
        val jsonObject = array.get<JsonObject>(row)
        val jsonValue = jsonObject.get<JsonValue>(column)

        /*
         * This is subject to the bug where number 0 is treated as null! See
         * https://github.com/gwtproject/gwt/issues/9484
         */return if (jsonValue != null && (jsonValue.type === JsonType.NULL || jsonValue.type == null)) {
            null
        } else {
            if (getColumnClass(column) == Double::class.java) {
                jsonObject.getNumber(column)
            } else if (getColumnClass(column) == Boolean::class.java) {
                jsonObject.getBoolean(column)
            } else {
                try {
                    getObject(jsonValue)
                } catch (e: AssertionError) {
                    // GWT may throw this exception, probably if the column doesn't exist
                    e.printStackTrace()
                    null
                }
            }
        }
    }

    fun getRow(integer: Int?): Series<String, *>? {
        return null
    }

    fun join(series: Series<*,*>?, strings: Array<String?>?): DataFrame<*, *, *>? {
        return null
    }

    private fun getObject(value: JsonValue?): Any? {
        return if (value != null) {
            if (value.type === JsonType.STRING) {
                value.asString()
            } else if (value.type === JsonType.NUMBER) {
                value.asNumber()
            } else if (value.type === JsonType.BOOLEAN) {
                value.asBoolean()
            } else if (value.type === JsonType.ARRAY) {
                val array = value as JsonArray
                val a = arrayOfNulls<Any>(array.length())
                for (i in 0 until array.length()) {
                    a[i] = getObject(array.get<JsonValue>(i))
                }
                a
            } else if (value.type === JsonType.OBJECT) {
                value.toNative()
            } else if (value.type === JsonType.NULL) {
                null
            } else {
                value.asString()
            }
        } else {
            null
        }
    }

    private fun getClass(value: JsonValue): KClass<*>? {
        return if (value.type === JsonType.NULL) {
            null
        } else if (value.type === JsonType.STRING) {
            String::class
        } else if (value.type === JsonType.NUMBER) {
            Double::class
        } else if (value.type === JsonType.BOOLEAN) {
            Boolean::class
        } else if (value.type === JsonType.ARRAY) {
            Array<Any>::class
        } else if (value.type === JsonType.OBJECT) {
            JsonObject::class
        } else {
            String::class
        }
    }

    companion object {
        private val logger: Logger
            private get() = Logger.getLogger(JsonDataFrame::class.java.name)
    }
}