/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.aggregates

import org.molap.aggregates.aggregation.Aggregation
import org.molap.aggregates.cube.Group
import org.molap.aggregates.cuboid.Cuboid
import org.molap.dataframe.DataFrame

/**
 * A data frame specialized for aggregating values. For obtaining an instance of such an interface,
 * see [com.macrofocus.molap.dataframe.DataFrame.aggregate].
 *
 * @param <C> the original type of columns
</C> */
interface AggregateDataFrame<C> : DataFrame<Group?, String?, Any?> {
    /**
     * Returns the cuboid used to aggregate the values.
     *
     * @return the cuboid
     */
    val cuboid: Cuboid?

    /**
     * Returns a new data frame that group the rows by values of the specified columns.
     *
     * @param columns the columns
     * @return a new data frame aggregating the values according to the new criteria
     */
    fun drillDown(vararg columns: C): AggregateDataFrame<C>

    /**
     * Returns a new data frame with the last group by level removed, i.e. one level up.
     *
     * @return a new data frame aggregating the values according to the new criteria
     */
    fun drillUp(): AggregateDataFrame<C>
    fun slice(value: Any?): AggregateDataFrame<C>
    fun dice(valuesSets: Set<Any?>?): AggregateDataFrame<C>
    fun collapse(): AggregateDataFrame<C>

    /**
     * Returns a new data frame that change the order of the groups according to the
     * specified aggregation methods.
     *
     * @param aggregations the aggregation methods
     * @return a new data frame
     */
    fun order(vararg aggregations: Aggregation<Any?>?): AggregateDataFrame<C>

    /**
     * Returns a new data frame that pivot the specified aggregation method into column values.
     *
     * @param aggregation the aggregation to pivot
     * @return a new data frame
     */
    fun pivot(aggregation: Aggregation<Any?>?): AggregateDataFrame<C>

    /**
     * Combine the values from multiple cuboids into that data frame.
     *
     * @param cuboids the cuboids
     * @return a new data frame
     */
    fun on(vararg cuboids: Cuboid): AggregateDataFrame<C>
}