/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.aggregates.aggregation

import org.molap.aggregates.cube.Group
import org.molap.aggregates.cuboid.Cuboid
import kotlin.math.max
import kotlin.math.min
import kotlin.reflect.KClass

abstract class AbstractAggregation<T> : Aggregation<T> {
    override fun `as`(name: String?): Aggregation<T> {
        return object : AbstractAggregation<T>() {
            override val name: String?
                get() = name

            override val type: KClass<Any>
                get() = this@AbstractAggregation::class as KClass<Any>

            override fun compute(cuboid: Cuboid?, group: Group?): T? {
                return compute(cuboid, group)
            }
        }
    }

    override val name: String?
        get() = toString()

    override fun add(aggregation: Aggregation<T>): Aggregation<Double> {
        return AddAggregation(this, aggregation)
    }

    override fun substract(aggregation: Aggregation<T>): Aggregation<Double> {
        return SubstractAggregation(this, aggregation)
    }

    override fun multiply(aggregation: Aggregation<T>): Aggregation<Double> {
        return MultiplyAggregation(this, aggregation)
    }

    override fun divide(aggregation: Aggregation<T>): Aggregation<Double> {
        return DivideAggregation(this, aggregation)
    }

    override fun min(aggregation: Aggregation<T>): Aggregation<Double> {
        return MinAggregation(this, aggregation)
    }

    override fun max(aggregation: Aggregation<T>): Aggregation<Double> {
        return MaxAggregation(this, aggregation)
    }

    private abstract class TwoAggregation<T>(a1: Aggregation<T>, a2: Aggregation<T>) : AbstractAggregation<Double>() {
        private val a1: Aggregation<T>
        private val a2: Aggregation<T>

        override val type: KClass<Any>
            get() = Double::class as KClass<Any>

        override fun compute(cuboid: Cuboid?, group: Group?): Double? {
            val v1: T? = a1.compute(cuboid, group)
            val v2: T? = a2.compute(cuboid, group)
            return if (v1 != null && v2 != null) {
                if (v1 is Number && v2 is Number) {
                    compute(v1, v2)
                } else {
                    null
                }
            } else {
                null
            }
        }

        protected abstract fun compute(v1: Number, v2: Number): Double?
        protected abstract fun name(n1: String?, n2: String?): String?
        override fun toString(): String {
            return name(a1.name, a2.name)!!
        }

        init {
            this.a1 = a1
            this.a2 = a2
        }
    }

    private class AddAggregation<T> (a1: Aggregation<T>, a2: Aggregation<T>) : TwoAggregation<T>(a1, a2) {
        override fun compute(v1: Number, v2: Number): Double {
            return v1.toDouble() + v2.toDouble()
        }

        override fun name(n1: String?, n2: String?): String? {
            return "$n1+$n2"
        }
    }

    private class SubstractAggregation<T>(a1: Aggregation<T>, a2: Aggregation<T>) : TwoAggregation<T>(a1, a2) {
        override fun compute(v1: Number, v2: Number): Double? {
            return v1.toDouble() - v2.toDouble()
        }

        override fun name(n1: String?, n2: String?): String? {
            return "$n1-$n2"
        }
    }

    private class MultiplyAggregation<T>(a1: Aggregation<T>, a2: Aggregation<T>) : TwoAggregation<T>(a1, a2) {
        override fun compute(v1: Number, v2: Number): Double? {
            return v1.toDouble() * v2.toDouble()
        }

        override fun name(n1: String?, n2: String?): String? {
            return "$n1*$n2"
        }
    }

    private class DivideAggregation<T>(a1: Aggregation<T>, a2: Aggregation<T>) : TwoAggregation<T>(a1, a2) {
        override fun compute(v1: Number, v2: Number): Double? {
            return if (v2 != null && v2.toDouble() != 0.0) {
                v1.toDouble() / v2.toDouble()
            } else {
                null
            }
        }

        override fun name(n1: String?, n2: String?): String? {
            return "$n1/$n2"
        }
    }

    private class MinAggregation<T>(a1: Aggregation<T>, a2: Aggregation<T>) : TwoAggregation<T>(a1, a2) {
        override fun compute(v1: Number, v2: Number): Double {
            return min(v1.toDouble(), v2.toDouble())
        }

        override fun name(n1: String?, n2: String?): String? {
            return "min($n1,$n2)"
        }
    }

    private class MaxAggregation<T>(a1: Aggregation<T>, a2: Aggregation<T>) : TwoAggregation<T>(a1, a2) {
        override fun compute(v1: Number, v2: Number): Double {
            return max(v1.toDouble(), v2.toDouble())
        }

        override fun name(n1: String?, n2: String?): String? {
            return "max($n1,$n2)"
        }
    }
}