/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.aggregates.cuboid

import com.macrofocus.common.collection.CollectionFactory.concurrentHashMap
import org.molap.aggregates.aggregation.AggregateSeriesFunction
import org.molap.aggregates.cube.Cube
import org.molap.aggregates.cube.DistributiveStatistics
import org.molap.aggregates.cube.Group
import org.molap.aggregates.cube.UnivariateStatistics
import org.molap.series.Series

abstract class AbstractCuboid protected constructor(cube: Cube) : Cuboid {
    override val cube: Cube
    override var isDirty = true
    private var distributiveStatisticsCache: MutableMap<GroupColumn, DistributiveStatistics?>? = null
    private var univariateStatisticsCache: MutableMap<GroupColumn, UnivariateStatistics?>? = null
    private var aggregateCache: MutableMap<GroupAggregation, Any?>? = null
    private var distinctCache: MutableMap<GroupColumn, MutableSet<Any?>?>? = null
    protected open fun build() {
        isDirty = false

        distributiveStatisticsCache = concurrentHashMap<GroupColumn, DistributiveStatistics?>()
        univariateStatisticsCache = concurrentHashMap<GroupColumn, UnivariateStatistics?>()
        aggregateCache = concurrentHashMap<GroupAggregation, Any?>()
        distinctCache = concurrentHashMap<GroupColumn, MutableSet<Any?>?>()
    }

    override fun getDistributiveStatistics(group: Group?, column: Series<Any?, Any?>, rowExcluder: DistributiveStatistics.RowFilter<Any?>?): DistributiveStatistics? {
//        assert(group != null)
//        assert(column != null)
        if (isDirty) {
            build()
        }
        return if (group != null) {
            val key = GroupColumn(group, column, rowExcluder)
            if (!distributiveStatisticsCache!!.containsKey(key)) {
                val child: Cuboid? = cube.getChildCuboid(this)
                val distributiveStatistics: DistributiveStatistics?
                if (child != null) {
                    distributiveStatistics = DistributiveStatistics()
                    for (childGroup in child.getGroupsStartingWith(group)!!) {
//                        assert(childGroup.startsWith(group))
                        val childDistributiveStatistics: DistributiveStatistics? = child.getDistributiveStatistics(childGroup, column, rowExcluder)
                        distributiveStatistics.add(childDistributiveStatistics!!)
                    }
                } else {
                    val rows: Iterable<Any?>? = getRows(group)
                    if (rows != null) {
                        distributiveStatistics = DistributiveStatistics(column, rows, rowExcluder)
                    } else {
                        distributiveStatistics = null
                    }
                }
                distributiveStatisticsCache!![key] = distributiveStatistics
                return distributiveStatistics
            }
            distributiveStatisticsCache!![key]
        } else {
            null
        }
    }

    override fun getUnivariateStatistics(group: Group?, column: Series<Any?, Any?>?): UnivariateStatistics? {
//        assert(group != null)
//        assert(column != null)
        if (isDirty) {
            build()
        }
        return if (group != null) {
            val key = GroupColumn(group, column, null)
            if (!univariateStatisticsCache!!.containsKey(key)) {
                val univariateStatistics: UnivariateStatistics?
                val rows: Iterable<Any?>? = getRows(group)
                if (rows != null) {
                    univariateStatistics = UnivariateStatistics(rows, column!!)
                } else {
                    univariateStatistics = null
                }
                univariateStatisticsCache!![key] = univariateStatistics
                return univariateStatistics
            }
            univariateStatisticsCache!![key]
        } else {
            null
        }
    }

    fun <T> getAggregate(group: Group?, func: AggregateSeriesFunction<T>?): T? {
//        assert(group != null)
//        assert(func != null)
        if (isDirty) {
            build()
        }
        return if (group != null) {
            val key = GroupAggregation(group, func)
            if (!aggregateCache!!.containsKey(key)) {
                val aggregate: T?
                val rows: Iterable<Any?>? = getRows(group)
                aggregate = if (rows != null) {
                    func!!.aggregate(this, rows)
                } else {
                    null
                }
                aggregateCache!![key] = aggregate
                return aggregate
            }
            aggregateCache!![key] as T?
        } else {
            null
        }
    }

    override fun getDistinct(group: Group?, column: Series<Any?,Any?>): Set<Any?>? {
        if (isDirty) {
            build()
        }
        return if (group != null) {
            val key = GroupColumn(group, column, null)
            if (!distinctCache!!.containsKey(key)) {
                val child: Cuboid? = cube.getChildCuboid(this)
                val set: MutableSet<Any?>?
                if (child != null) {
                    set = HashSet<Any?>()
                    for (childGroup in child.getGroupsStartingWith(group)!!) {
//                        assert(childGroup.startsWith(group))
                        val childSet: Set<Any?>? = child.getDistinct(childGroup, column)
                        set.addAll(childSet!!)
                    }
                } else {
                    val rows: Iterable<Any?>? = getRows(group)
                    if (rows != null) {
                        set = HashSet<Any?>()
                        for (row in rows) {
                            val value: Any? = column.get(row)
                            set.add(value!!)
                        }
                    } else {
                        set = null
                    }
                }
                distinctCache!![key] = set
                return set
            }
            distinctCache!![key]
        } else {
            null
        }
    }

    private class GroupColumn(group: Group?, column: Series<*,*>?, rowFilter: DistributiveStatistics.RowFilter<*>?) {
        private val group: Group?
        private val column: Series<*,*>?
        private val rowFilter: DistributiveStatistics.RowFilter<*>?
        override fun equals(o: Any?): Boolean {
            if (this === o) return true
            if (o == null || this::class != o::class) return false
            val that = o as GroupColumn
            if (group != that.group) return false
            return if (column != that.column) false else !if (rowFilter != null) !rowFilter.equals(that.rowFilter) else that.rowFilter != null
        }

        override fun hashCode(): Int {
            var result: Int = group.hashCode()
            result = 31 * result + column.hashCode()
            result = 31 * result + if (rowFilter != null) rowFilter.hashCode() else 0
            return result
        }

        override fun toString(): String {
            return "GroupColumn{" +
                    "group=" + group +
                    "group=" + group.hashCode() +
                    ", column=" + column +
                    ", column=" + column.hashCode() +
                    '}'
        }

        init {
//            assert(group != null)
//            assert(column != null)
            this.group = group
            this.column = column
            this.rowFilter = rowFilter
        }
    }

    private class GroupAggregation(group: Group?, aggregateSeriesFunction: AggregateSeriesFunction<*>?) {
        private val group: Group?
        private val aggregateSeriesFunction: AggregateSeriesFunction<*>?
        override fun equals(o: Any?): Boolean {
            if (this === o) return true
            if (o == null || this::class != o::class) return false
            val that = o as GroupAggregation
            return if (group != that.group) false else !if (aggregateSeriesFunction != null) !aggregateSeriesFunction.equals(that.aggregateSeriesFunction) else that.aggregateSeriesFunction != null
        }

        override fun hashCode(): Int {
            var result: Int = group.hashCode()
            result = 31 * result + if (aggregateSeriesFunction != null) aggregateSeriesFunction.hashCode() else 0
            return result
        }

        override fun toString(): String {
            return "GroupColumn{" +
                    "group=" + group +
                    "group=" + group.hashCode() +
                    ", aggregateSeriesFunction=" + aggregateSeriesFunction +
                    ", aggregateSeriesFunction=" + aggregateSeriesFunction.hashCode() +
                    '}'
        }

        init {
//            assert(group != null)
//            assert(aggregateSeriesFunction != null)
            this.group = group
            this.aggregateSeriesFunction = aggregateSeriesFunction
        }
    }

    init {
        this.cube = cube
    }
}