/*
 * Copyright (c) 2014 Macrofocus GmbH. All Rights Reserved.
 */
package org.molap.aggregates.query

import org.molap.aggregates.aggregation.Aggregation
import org.molap.aggregates.cube.Dimensions
import org.molap.aggregates.cube.Group
import org.molap.aggregates.cuboid.Cuboid
import org.molap.dataframe.DataFrameEvent
import org.molap.dataframe.DataFrameListener

class CuboidQuery : AbstractQuery {
    override var cuboid: Cuboid? = null
    get() {
        if (field == null) {
            field = operation!![query!!.cuboid]
        }
        return field
    }
    private var query: Query? = null
    private var operation: Operation? = null
    override val aggregations: Array<Aggregation<Any?>>
    override var groups: List<Group>? = null
    get() {
        if (isDirty) {
            run()
        }
        return field
    }
    val queryListener: QueryListener = object : QueryListener {
        override fun queryChanged() {
            cuboid = null
            isDirty = true
            notifyQueryChanged()
        }
    }
    val dataFrameListener: DataFrameListener<Any?,Any?> = object : DataFrameListener<Any?,Any?> {
        override fun dataFrameChanged(event: DataFrameEvent<Any?,Any?>) {
            isDirty = true
            notifyQueryChanged()
        }
    }

    constructor(cuboid: Cuboid?, vararg aggregations: Aggregation<Any?>) {
//        assert(cuboid != null)
        this.cuboid = cuboid
        this.aggregations = aggregations as Array<Aggregation<Any?>>
        this.cuboid!!.cube!!.dataFrame.addWeakDataFrameListener(dataFrameListener)
    }

    constructor(query: Query?, operation: Operation?, vararg aggregations: Aggregation<Any?>) {
//        assert(query != null)
        this.query = query
        this.operation = operation
        this.aggregations = aggregations as Array<Aggregation<Any?>>
        this.query!!.addWeakQueryListener(queryListener)
    }

    private fun run() {
        isDirty = false
        val groups: MutableList<Group> = ArrayList<Group>()
        for (group in cuboid!!.groups!!) {
            groups.add(group)
        }
        this.groups = groups
    }

    override fun getValue(path: Group?, aggregation: Aggregation<Any?>): Any? {
        return aggregation.compute(cuboid, path)!!
    }

    override val dimensions: Dimensions
        get() { return cuboid!!.dimensions!! }

    override fun setDice(valuesSets: Set<Any?>) {
        operation = DiceOperation(valuesSets)
        cuboid = null
        isDirty = true
        notifyQueryChanged()
    }

    interface Operation {
        operator fun get(cuboid: Cuboid?): Cuboid?
    }

    class DrillDownOperation(vararg columns: Any?) : Operation {
        private val columns: Array<out Any?>
        override fun get(cuboid: Cuboid?): Cuboid? {
            for (column in columns) {
            }
//            assert(cuboid != null && columns != null) { cuboid.toString() + ", " + columns }
            return cuboid!!.drillDown(*columns)
        }

        init {
            this.columns = columns
        }
    }

    class DrillUpOperation : Operation {
        override fun get(cuboid: Cuboid?): Cuboid? {
            return cuboid!!.drillUp()
        }
    }

    class SliceOperation(private val slice: Any?) : Operation {
        override fun get(cuboid: Cuboid?): Cuboid? {
            return cuboid!!.slice(slice)
        }
    }

    class DiceOperation(private val dice: Set<Any?>?) : Operation {
        override fun get(cuboid: Cuboid?): Cuboid? {
            return cuboid!!.dice(dice!!)
        }
    }

    class CollapseOperation : Operation {
        override fun get(cuboid: Cuboid?): Cuboid? {
            return cuboid!!.collapse()
        }
    }
}